/**********************************************************************
zyGrib: meteorological GRIB file viewer
Copyright (C) 2008-2010 - Jacques Zaninetti - http://www.zygrib.org

This program is free software: you can redistribute it and/or modify
it under the terms of the GNU General Public License as published by
the Free Software Foundation, either version 3 of the License, or
(at your option) any later version.

This program is distributed in the hope that it will be useful,
but WITHOUT ANY WARRANTY; without even the implied warranty of
MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
GNU General Public License for more details.

You should have received a copy of the GNU General Public License
along with this program.  If not, see <http://www.gnu.org/licenses/>.
***********************************************************************/

#include "Util.h"
#include "DataPointInfo.h"

DataPointInfo::DataPointInfo ( 
					GriddedReader *reader, 
					float x, float y, time_t  date )
{
	this->x = x;
	this->y = y;
	this->date = date;
	this->reader = reader;

	pressure = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_PRESSURE,LV_MSL,0), x,y,date);
	temp = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_TEMP,LV_ABOV_GND,2), x,y,date);
	tempPot = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_TEMP_POT,LV_SIGMA,9950), x,y,date);
	tempMin = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_TMIN,LV_ABOV_GND,2), x,y,date);
	tempMax = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_TMAX,LV_ABOV_GND,2), x,y,date);
	rain    = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_PRECIP_TOT,LV_GND_SURF,0), x,y,date);
	cloud   = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_CLOUD_TOT,LV_ATMOS_ALL,0), x,y,date);
	humidRel = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_HUMID_REL,LV_ABOV_GND,2), x,y,date);
	humidSpec = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_HUMID_SPEC,LV_ABOV_GND,2), x,y,date);
	dewPoint = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_DEWPOINT,LV_ABOV_GND,2), x,y,date);
	isotherm0HGT = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_GEOPOT_HGT,LV_ISOTHERM0,0), x,y,date);
	snowDepth    = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_SNOW_DEPTH,LV_GND_SURF,0), x,y,date);
	snowCateg    = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_SNOW_CATEG,LV_GND_SURF,0), x,y,date);
	frzRainCateg = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_FRZRAIN_CATEG,LV_GND_SURF,0), x,y,date);
	CAPEsfc = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_CAPE,LV_GND_SURF,0), x,y,date);
	//-----------------------------------------
	vx = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_WIND_VX,LV_ABOV_GND,10), x,y,date);
	vy = reader==NULL ? GRIB_NOTDEF
			: reader->getDateInterpolatedValue (DataCode(GRB_WIND_VY,LV_ABOV_GND,10), x,y,date);
	if (vx!=GRIB_NOTDEF && vy!=GRIB_NOTDEF) {
		windSpeed = sqrt (vx*vx + vy*vy);
		windDir = - atan2 (-vx, vy) *180.0/M_PI + 180;
		if (windDir < 0)    windDir += 360.0;
		if (windDir >= 360) windDir -= 360.0;
	}
	else {
		windSpeed = GRIB_NOTDEF;
		windDir = GRIB_NOTDEF;
	}
	//-------------------------------------------------------------
	if (reader && reader->hasAltitudeData())
	{
		double SH, T;
		int P, i;
		for (i=0; i<5; i++)
		{
			P = GEOPOThgt(i);	// 850 700 500 300 200
			SH = reader->getDateInterpolatedValue (DataCode(GRB_HUMID_SPEC,LV_ISOBARIC,P), x,y,date);
			T = reader->getDateInterpolatedValue (DataCode(GRB_TEMP,LV_ISOBARIC,P), x,y,date);
			hThetae [i] = DataRecordAbstract::computeThetaE (T,P,SH);
			hTemp [i] = T;
			hGeopot [i] = reader->getDateInterpolatedValue (DataCode(GRB_GEOPOT_HGT,LV_ISOBARIC,P), x,y,date);
			hHumidRel [i]  = reader->getDateInterpolatedValue (DataCode(GRB_HUMID_REL,LV_ISOBARIC,P), x,y,date);
			hHumidSpec [i] = reader->getDateInterpolatedValue (DataCode(GRB_HUMID_SPEC,LV_ISOBARIC,P), x,y,date);

			hVx [i] = reader->getDateInterpolatedValue (DataCode(GRB_WIND_VX,LV_ISOBARIC,P), x,y,date);
			hVy [i] = reader->getDateInterpolatedValue (DataCode(GRB_WIND_VY,LV_ISOBARIC,P), x,y,date);
			if (hVx[i]!=GRIB_NOTDEF && hVy[i]!=GRIB_NOTDEF) {
				hWindSpeed[i] = sqrt (hVx[i]*hVx[i] + hVy[i]*hVy[i]);
				hWindDir[i] = -atan2(-hVx[i], hVy[i]) *180.0/M_PI + 180;
				if (hWindDir[i] < 0)    hWindDir[i] += 360.0;
				if (hWindDir[i] >= 360) hWindDir[i] -= 360.0;
			}
		}
		hasAltitudeData = true;
	}
	else {
		hasAltitudeData = false;
	}
}
//------------------------------------------------------------------
float DataPointInfo::getDataValue (const DataCode &dtc)
{
	switch (dtc.dataType) {
		case GRB_CAPE 		  : 
			return CAPEsfc;
		case GRB_SNOW_DEPTH   : 
			return snowDepth;
		case GRB_PRESSURE     : 
			return pressure;
		case GRB_PRECIP_TOT   : 
			return rain;
		case GRB_PRECIP_RATE  : 
			return rain;
		case GRB_CLOUD_TOT    : 
			return cloud;
		case GRB_SNOW_CATEG   : 
			return snowCateg;
		case GRB_FRZRAIN_CATEG: 
			return frzRainCateg;
		case GRB_TEMP_POT     : 
			return tempPot;
		case GRB_TMAX         : 
			return tempMax;
		case GRB_TMIN         : 
			return tempMin;
		case GRB_DEWPOINT     : 
			return dewPoint;
		case GRB_DIFF_TEMPDEW : 
			return 
				hasDewPoint()&&hasTemp() ? temp-dewPoint : GRIB_NOTDEF;
			
		case GRB_TEMP         : 
			if (dtc.getAltitude().levelType == LV_ISOBARIC)
				return hTemp [dtc.getAltitude().index()];
			else
				return temp;
		case GRB_GEOPOT_HGT   :
			if (dtc.levelType == LV_ISOTHERM0)
				return isotherm0HGT;
			else if (dtc.getAltitude().levelType == LV_ISOBARIC)
				return hGeopot [dtc.getAltitude().index()];
		case GRB_HUMID_REL    : 
			if (dtc.getAltitude().levelType == LV_ISOBARIC)
				return hHumidRel [dtc.getAltitude().index()];
			else
				return humidRel;
		case GRB_HUMID_SPEC   : 
			if (dtc.getAltitude().levelType == LV_ISOBARIC)
				return hHumidSpec [dtc.getAltitude().index()];
			else
				return humidSpec;
		case GRB_THETA_E      : 
			if (dtc.getAltitude().levelType == LV_ISOBARIC)
				return hThetae [dtc.getAltitude().index()];
			else
				return GRIB_NOTDEF;
			
		case GRB_WIND_VX      : 
		case GRB_WIND_VY      : 
		case GRB_WIND_XY2D    : 
		case GRB_WIND_DIR     : 
			
		default:
			return GRIB_NOTDEF;
	}
}
//--------------------------------------------------------
bool DataPointInfo::hasWind (const Altitude &alt)  const
{	
	if (alt.levelType == LV_ISOBARIC) {
		int i = alt.index();
		return i>=0 && hVx[i]!=GRIB_NOTDEF && hVy[i]!=GRIB_NOTDEF;
	} 
	else
		return vx!=GRIB_NOTDEF && vy!=GRIB_NOTDEF;
}
//--------------------------------------------------------
bool DataPointInfo::getWindValues (const Altitude &alt, float *speed, float*dir) const
{	
	if (! hasWind (alt)) {
		*speed = GRIB_NOTDEF;
		*dir = GRIB_NOTDEF;
		return false;
	}
	if (alt.levelType == LV_ISOBARIC) {
		int i = alt.index();
		*speed = hWindSpeed [i];
		*dir = hWindDir [i];
	} 
	else {
		*speed = windSpeed;
		*dir = windDir;
	}
	return true;
}
//--------------------------------------------------------
void  DataPointInfo::getWindVxVy (const Altitude &alt, float *vx, float*vy) const
{	
	if (! hasWind (alt)) {
		*vx = GRIB_NOTDEF;
		*vy = GRIB_NOTDEF;
	}
	if (alt.levelType == LV_ISOBARIC) {
		int i = alt.index();
		*vx = hVx [i];
		*vy = hVy [i];
	} 
	else {
		*vx = this->vx;
		*vy = this->vy;
	}
}

