;==================================================================
function rcoinf,a
  ; DOCUMENT func rcoinf(a)
  ;  Inflates RCO compressed matrix to full form.
  ;
type = size(a,/sname)
if type eq 'RCO' then x = fltarr(a.c,a.r) else $
if type eq 'RCO_D' then x = dblarr(a.c,a.r) else $
message,'Unsupported Data Type' 
for i=0,a.r-1 do begin
    if (*a.ix)(i+1) gt (*a.ix)(i) then begin
        for j=(*a.ix)(i),(*a.ix)(i+1)-1 do x((*a.jx)(j),i)=(*a.xn)(j)
    endif
endfor
return,x
end

;==================================================================
function ruoinf,a
  ; DOCUMENT func ruoinf(a)
  ;  Inflates RUO compressed matrix to full form.
  ;
type = size(*a.xn,/tname)
if type eq 'FLOAT' then x = fltarr(a.r,a.r) else $
if type eq 'DOUBLE' then x = dblarr(a.r,a.r) else $
message,'Unsupported Data Type' 
for i=0,a.r-1 do x(i,i)=(*a.xd)(i) 
for i=0,a.r-2 do begin 
    if (*a.ix)(i+1) gt (*a.ix)(i) then begin 
        for j=(*a.ix)(i),(*a.ix)(i+1)-1 do begin
            x(i,(*a.jx)(j)) = (*a.xn)(j)
            x((*a.jx)(j),i) = (*a.xn)(j)
        endfor
    endif
endfor
return,x
end

;==================================================================
function rcos,a
  ; DOCUMENT func rcos(a)
  ;  Computes the sparseness (or rather, the "fill") of a matrix on RCO format.

if a.r ne 0 then xfill = float(a.n)/float(a.c)/float(a.r) else xfill = 0
return,xfill
end

;==================================================================
function ruos,a
  ; DOCUMENT func ruos(a)
  ;  Computes the sparseness (or rather, the "fill") of a matrix on RUO format.

if a.r ne 0 then xfill = (float(a.n)*2.0+float(a.r))/float(a.r)^2. $
else xfill = 0
return,xfill
end

;==================================================================
function sprco,x,t=t,ur=ur,un=un,so=so
  ; DOCUMENT func sprco(x,t=,ur=,un=)
  ;   Compress a 2D matrix on sparse RCO format.
  ;   SEE ALSO: sprco_float (soy.c)

; test = findgen(6,10) & s = sprco(test,t=float(1.e-6),ur=100,un=200)
; tmp = rcoinf(s) & print,minmax(test-tmp)

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'                  
if keyword_set(ur) then ur = ur else ur = MR
if keyword_set(un) then un = un else un = MN
sz = size(x)
c = long(sz(1))
r = long(sz(2))
ix = lonarr(ur)              
jx = lonarr(un)  
if sz(3) eq 4 then begin
  if keyword_set(t) then t=float(t) else t = 0.0
  xn = fltarr(un)
  n = call_external(so+'soy.so','sprco_float',$
                    x,r,c,t,ix,jx,xn,$
                    value=bytarr(7),/i_value)
  s = {rco}
endif else if sz(3) eq 5 then begin
  if keyword_set(t) then t=double(t) else t = 0.0D
  xn = dblarr(un)
  n = call_external(so+'soy.so','sprco_double',$
                    x,r,c,t,ix,jx,xn,$
                    value=bytarr(7),/i_value)
  s = {rco_d}
endif
s.r = r
s.c = c
s.n = n
s.t = t
s.ix = ptr_new(ix)
s.jx = ptr_new(jx)
s.xn = ptr_new(xn)
return,s
end

;==================================================================
function spruo,x,t=t,ur=ur,un=un,so=so

; test = randomu(seed,10,10) & test = test##transpose(test) 
; test = test*(test gt 2.) & r = spruo(test,t=float(1.e-6),ur=100,un=200)
; tmp = ruoinf(r) & print,minmax(test-tmp)

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
if keyword_set(ur) then ur = ur else ur = MR
if keyword_set(un) then un = un else un = MN
sz = size(x)
r = long(sz(2))
ix = lonarr(ur)              
jx = lonarr(un)  
if sz(3) eq 4 then begin
  if keyword_set(t) then t=float(t) else t = 0.0
  xn = fltarr(un)
  xd = fltarr(ur)
  n = call_external(so+'soy.so','spruo_float',$
                    x,r,t,ix,jx,xn,xd,$
                    value=bytarr(7),/i_value)
  s = {ruo}
endif else if sz(3) eq 5 then begin
  if keyword_set(t) then t=double(t) else t = 0.0D
  xn = dblarr(un)
  xd = dblarr(ur)
  n = call_external(so+'soy.so','spruo_double',$
                    x,r,t,ix,jx,xn,xd,$
                    value=bytarr(7),/i_value)
  s = {ruo_d}
endif
s.r = r
s.n = n
s.t = t
s.ix = ptr_new(ix)
s.jx = ptr_new(jx)
s.xn = ptr_new(xn)
s.xd = ptr_new(xd)
return,s
end

;==================================================================
function rcoxv,a,v,so=so
 ;  DOCUMENT func rcoxv(a,v)
 ;  Sparse matrix-vector multiplication of RCO and real vector.
 ;  SEE ALSO: rcoxv_float (soy.c)

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
vtype = size(v,/tname)
atype = size(a,/sname)
if vtype eq 'FLOAT' and atype eq 'RCO' then begin
    u = fltarr(a.r)
    n = call_external(so+'soy.so','rcoxv_float',$
                      v,u,a.r,*a.ix,*a.jx,*a.xn,$
                      value=bytarr(6),/i_value)
endif else if vtype eq 'DOUBLE' and atype eq 'RCO_D' then begin
    u = dblarr(a.r)
    n = call_external(so+'soy.so','rcoxv_double',$
                      v,u,a.r,*a.ix,*a.jx,*a.xn,$
                      value=bytarr(6),/i_value)
endif else message,'objects are of incompatible types'
return,u
end


;==================================================================
function ruoxv,a,v,so=so

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
vtype = size(v,/tname)
atype = size(a,/sname)
if vtype eq 'FLOAT' and atype eq 'RUO' then begin
    u = fltarr(a.r)
    w = fltarr(a.r)
    n = call_external(so+'soy.so','ruoxv_float',$
                      v,u,w,a.r,*a.ix,*a.jx,*a.xn,*a.xd,$
                      value=bytarr(8),/i_value)
endif else if vtype eq 'DOUBLE' and atype eq 'RUO_D' then begin
    u = dblarr(a.r)
    w = dblarr(a.r)
    n = call_external(so+'soy.so','ruoxv_double',$
                      v,u,w,a.r,*a.ix,*a.jx,*a.xn,*a.xd,$
                      value=bytarr(8),/i_value)
endif else message,'objects are of incompatible types'
return,u
end

;==================================================================
function rcoadd,a,b,ur=ur,un=un,so=so
  ; DOCUMENT func rcoadd(a,b,ur=,un=)
  ; Sparse addition of two RCO matrices.
  ; SEE ALSO: rcoadd_float (soy.c)

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
if size(a,/sname) ne size(b,/sname) then message,'Objects have incompatible data types'
if (a.r ne b.r) or (a.c ne b.c) then message,'Objects have incompatible dimensions'
if not keyword_set(ur) then ur = 2*(a.r+b.r)
if not keyword_set(un) then un = 2*(a.n+b.n)
ss = lonarr(ur)
if size(a,/sname) eq 'RCO' then begin
    c = {rco}
    c.r = a.r
    c.c = a.c
    c.t = a.t
    c.ix = ptr_new(lonarr(ur))
    c.jx = ptr_new(lonarr(un))
    c.xn = ptr_new(fltarr(un))
    t = fltarr(ur)
    n = call_external(so+'soy.so','rcoadd_float',$
                      a.r,a.c,*a.ix,*a.jx,*a.xn,$
                      b.r,b.c,*b.ix,*b.jx,*b.xn,$
                      c.r,c.c,*c.ix,*c.jx,*c.xn,t,ss,$
                      value=bytarr(17),/i_value)
endif else if size(a,/sname) eq 'RCO_D' then begin
    c = {rco_d}
    c.r = a.r
    c.c = a.c
    c.t = a.t
    c.ix = ptr_new(lonarr(ur))
    c.jx = ptr_new(lonarr(un))
    c.xn = ptr_new(dblarr(un))
    t = dblarr(ur)
    n = call_external(so+'soy.so','rcoadd_double',$
                      a.r,a.c,*a.ix,*a.jx,*a.xn,$
                      b.r,b.c,*b.ix,*b.jx,*b.xn,$
                      c.r,c.c,*c.ix,*c.jx,*c.xn,t,ss,$
                      value=bytarr(17),/i_value)
endif
c.n = n
return,c
end

;==================================================================
function ruoadd,a,b,ur=ur,un=un,so=so

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
if size(a,/sname) ne size(b,/sname) then message,'Objects have incompatible data types'
if a.r ne b.r then message,'Objects have incompatible dimensions'
if not keyword_set(ur) then ur = 2*(a.r+b.r)
if not keyword_set(un) then un = 2*(a.n+b.n)
ss = lonarr(ur)
if size(a,/sname) eq 'RUO' then begin
    c = {ruo}
    c.r = a.r
    c.t = a.t
    c.ix = ptr_new(lonarr(ur))
    c.jx = ptr_new(lonarr(un))
    c.xn = ptr_new(fltarr(un))
    c.xd = ptr_new(lonarr(ur))
    tt = fltarr(ur)
    n = call_external(so+'soy.so','ruoadd_float',$
                      a.r,*a.xd,*a.ix,*a.jx,*a.xn,$
                      b.r,*b.xd,*b.ix,*b.jx,*b.xn,$
                      c.r,*c.xd,*c.ix,*c.jx,*c.xn,tt,ss,$
                      value=bytarr(17),/i_value)
endif else if size(a,/sname) eq 'RUO_D' then begin
    c = {ruo_d}
    c.r = a.r
    c.t = a.t
    c.ix = ptr_new(lonarr(ur))
    c.jx = ptr_new(lonarr(un))
    c.xn = ptr_new(dblarr(un))
    c.xd = ptr_new(lonarr(ur))
    tt = dblarr(ur)
    n = call_external(so+'soy.so','ruoadd_double',$
                      a.r,a.c,*a.ix,*a.jx,*a.xn,$
                      b.r,b.c,*b.ix,*b.jx,*b.xn,$
                      c.r,c.c,*c.ix,*c.jx,*c.xn,t,ss,$
                      value=bytarr(17),/i_value)
endif
c.n = n
return,c
end

;==================================================================
function rcoata,a,ur=ur,un=un,t=t,so=so
  ; DOCUMENT func rcoata(a,ur=,un=,t=)
  ; Sparse mutiplication of an RCO matrix with its transpose from
  ; the left, i.e. transpose(a)##a
  ; SEE ALSO: rcoata_float (soy.c)

if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
if not keyword_set(ur) then ur = a.r+2
if not keyword_set(un) then un = a.n*5
if size(a,/sname) eq 'RCO' then begin
    b = {ruo}
    b.r = a.r
    if keyword_set(t) then b.t = t else b.t = float(a.t^2.);
    b.ix = ptr_new(lonarr(ur))
    b.jx = ptr_new(lonarr(un))
    b.xn = ptr_new(fltarr(un))
    b.xd = ptr_new(fltarr(ur))
    n = call_external(so+'soy.so','rcoata_float',$
                      a.r,*a.ix,*a.jx,*a.xn,b.t,*b.ix,*b.jx,*b.xn,*b.xd,$
                      value=bytarr(9),/i_value)
endif else if size(a,/sname) eq 'RCO_D' then begin
    b = {ruo_d}
    b.r = a.r
    if keyword_set(t) then b.t = t else b.t = double(a.t^2.);
    b.ix = ptr_new(lonarr(ur))
    b.jx = ptr_new(lonarr(un))
    b.xn = ptr_new(dblarr(un))
    b.xd = ptr_new(dblarr(ur))
    n = call_external(so+'soy.so','rcoata_double',$
                      a.r,*a.ix,*a.jx,*a.xn,b.t,*b.ix,*b.jx,*b.xn,*b.xd,$
                      value=bytarr(9),/i_value)
endif else message,'Unrecognized data type (must be RCO or RCO_D)'
b.n = n
return,b
end

;==================================================================
function rcoatb,a,b,ur=ur,un=un,t=t,u=u,so=so

;DOCUMENT func rcoatb(a,b,ur=,un=,t=,u=)
;Sparse mutiplication of two RCO matrices, a'.b. Setting u=1
;("upper") computes only the upper triangular and diagonal
;elements of the matrix product, and returns an RUO matrix.
;Use this when computing e.g. the final step of a 3-matrix
;product of the type M = A'.W.A, where A is RCO, W is RUO
;and you know that the final result must be RUO.
;SEE ALSO: rcoatb_float (soy.c)
  
if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
if not keyword_set(ur) then ur = long(max([a.r,a.c]+2))
if not keyword_set(un) then un = (a.n+b.n)*2
if size(a,/sname) ne size(b,/sname) then message,'Incompatible data types'
if size(a,/sname) eq 'RCO' then begin
    if keyword_set(u) then begin
        c = {ruo}
        c.r = a.r
        if keyboard_set(t) then c.t = t else c.t = float(min([a.t,b.t]))
        c.ix = ptr_new(lonarr(ur))
        c.jx = ptr_new(lonarr(un))
        c.xn = ptr_new(fltarr(un))
        c.xd = ptr_new(fltarr(ur))
        n = call_external(so+'soy.so','rcoatb2_float',$
                          a.r,*a.ix,*a.jx,*a.xn,$
                          b.r,*b.ix,*b.jx,*b.xn,$
                          c.t,*c.ix,*c.jx,*c.xn,*c.xd,$
                          value=bytarr(13),/i_value)
    endif else begin
        c = {rco}
        c.r = a.r
        c.c = b.r
        if keyboard_set(t) then c.t = t else c.t = float(min([a.t,b.t]))
        c.ix = ptr_new(lonarr(ur))
        c.jx = ptr_new(lonarr(un))
        c.xn = ptr_new(fltarr(un))
        n = call_external(so+'soy.so','rcoatb_float',$
                          a.r,*a.ix,*a.jx,*a.xn,$
                          b.r,*b.ix,*b.jx,*b.xn,$
                          c.t,*c.ix,*c.jx,*c.xn,$
                          value=bytarr(12),/i_value)
    endelse
endif else if size(a,/sname) eq 'RCO_D' then begin
    if keyword_set(u) then begin
        c = {ruo_d}
        c.r = a.r
        if keyboard_set(t) then c.t = t else c.t = double(min([a.t,b.t]))
        c.ix = ptr_new(lonarr(ur))
        c.jx = ptr_new(lonarr(un))
        c.xn = ptr_new(dblarr(un))
        c.xd = ptr_new(dblarr(ur))
        n = call_external(so+'soy.so','rcoatb2_double',$
                          a.r,*a.ix,*a.jx,*a.xn,$
                          b.r,*b.ix,*b.jx,*b.xn,$
                          c.t,*c.ix,*c.jx,*c.xn,*c.xd,$
                          value=bytarr(13),/i_value)
    endif else begin
        c = {rco_d}
        c.r = a.r
        c.c = b.r
        if keyboard_set(t) then c.t = t else c.t = double(min([a.t,b.t]))
        c.ix = ptr_new(lonarr(ur))
        c.jx = ptr_new(lonarr(un))
        c.xn = ptr_new(dblarr(un))
        n = call_external(so+'soy.so','rcoatb_double',$
                          a.r,*a.ix,*a.jx,*a.xn,$
                          b.r,*b.ix,*b.jx,*b.xn,$
                          c.t,*c.ix,*c.jx,*c.xn,$
                          value=bytarr(12),/i_value)
    endelse
endif
c.n = n                         
return,c                        
end

;==================================================================
function rcotr,a,so=so
  ; DOCUMENT func rcotr(arg)
  ; Transposes an RCO matrix. Uses one builtin Yorick function
  ; (sort) to make the rest a little bit easier.
  
if not keyword_set(so) then so = expand_path('$SOI_DIR')+'/'
ur = n_elements(*a.ix)
un = n_elements(*a.xn)
if size(a,/sname) eq 'RCO' then begin
    at = {rco}
    at.xn = ptr_new(fltarr(un))
endif else if size(a,/sname) eq 'RCO_D' then begin
    at = {rco_d}
    at.xn = ptr_new(dblarr(un))
endif else message,'wrong data type'
at.n = a.n                      
at.r = a.c                      
at.c = a.r                      
at.t = a.t
at.ix = ptr_new(lonarr(a.c+2))
at.jx = ptr_new(lonarr(un))
sjx = long(sort((*a.jx)(0:a.n-1)))
hjx = (*a.jx)(sjx)
ax = lonarr(a.c)
acx = lonarr(a.c+2)
rind = lonarr(at.n)
n = call_external(so+'soy.so','rcotr_fix',$
                  ax,acx,hjx,rind,a.r,a.c,a.n,*a.ix,$
                  value=bytarr(8),/i_value)
(*at.ix)(0:at.r) = acx(0:at.r) 
(*at.jx)(0:at.n-1) = rind(sjx)    
(*at.xn)(0:at.n-1) = (*a.xn)(sjx) 
return,at                       
end

;==================================================================
function ruopcg,a,b,x0,nit=nit,tol=tol,itmax=itmax
  ; DOCUMENT func ruopcg(a,b,x0,&nit,tol=,itmax=,sgs=)
  ;   Preconditioned conjugate gradient solver for a symmetric positive
  ;   definite sparse linear system, with Jacobi preconditioner. This
  ;   algorithm is implemented straight out of Numerical Recipes, with
  ;   the matrix-vector multiplications carried out sparsely by the
  ;   ruoxv(a,v) function.
  ;   Optionally one may invoke symmetric Gauss-Seidel iterations upon
  ;   the Jacobi preconditioning, by setting the keyword sgs=#iters.
  ;   (at least 1). SGS requires the U, D and L to be externally defined.
  ;   SEE ALSO: ruoxv, aotest

prec = size(*a.xn,/tname)
if total(prec eq [size(b,/tname),size(x0,/tname)]) ne 2 then $
  message,'Incompatible data types (must be all float or double)'
if not keyword_set(itmax) then itmax = a.r
if not keyword_set(tol) then tol = 1.e-4
x = dblarr(a.r)
bnrm = total(b^2.)
u = ruoxv(a,x0(0:a.r-1))
r = b-u
tmp = (*a.xd)(0:a.r-1)
z = r/tmp                       ; Jacobi preconditioner
k = 0
err = 1.0D

while (k le itmax) and (err gt tol) do begin
    k += 1
    bknum = total(z*r)
    if k gt 1 then begin
        bk = bknum/bkden
        p = bk*p+z
    endif else p = z
    bkden = bknum
    if prec eq 'FLOAT' then u = ruoxv(a,float(p(0:a.r-1))) $
    else u = ruoxv(a,double(p(0:a.r-1)))
    z = u
    akden = total(z*p)
    ak = bknum/akden
    x += ak*p
    r -= ak*z
    z = r/tmp                   ; Jacobi preconditioner
    err = total(r^2.)/bnrm
endwhile
nit = k
if prec eq 'FLOAT' then return,float(x) $
else return,double(x)
end

;==================================================================
function restore_rco,fn

tmp = strsplit(fn,".",/extract,count=sn)
if tmp(sn-1) eq 'dat' then restore,fn $
else if tmp(sn-1) eq 'fits' then v = readfits(fn) $
else message,'Unrecognized file format : '+tmp(sn-1)+' (must be dat or fits)'
if size(v,/tname) eq 'FLOAT' then a = {rco} $
else if size(v,/tname) eq 'DOUBLE' then a = {rco_d} $
else message,'Stored data not a float'
a.n = long(v(0))                 
a.r = long(v(1))                 
a.c = long(v(2))                 
a.t = v(3)                      
a.xn = ptr_new(v(4:a.n+5))            
a.jx = ptr_new(long(v(a.n+4:2*a.n+3))) 
a.ix = ptr_new(long(v(2*a.n+4:2*a.n+a.r+4))) 
return,a   
end

;==================================================================
function restore_ruo,fn

tmp = strsplit(fn,".",/extract,count=sn)
if tmp(sn-1) eq 'dat' then restore,fn $
else if tmp(sn-1) eq 'fits' then v = readfits(fn) $
else message,'Unrecognized file format : '+tmp(sn-1)+' (must be dat or fits)'
if size(v,/tname) eq 'FLOAT' then a = {ruo} $
else if size(v,/tname) eq 'DOUBLE' then a = {ruo_d} $
else message,'Stored data not a float'
a.n = long(v(0))                 
a.r = long(v(1))                                  
a.t = v(2)                      
a.xn = ptr_new(v(3:a.n+2))
a.jx = ptr_new(long(v(a.n+3:2*a.n+2)))
a.ix = ptr_new(long(v(2*a.n+3:2*a.n+a.r+2)))
a.xd = ptr_new(v(2*a.n+a.r+3:2*a.n+2*a.r+2))
return,a   
end



;==================================================================
;
; PRO here
;
;==================================================================
pro save_rco,b,fn,dat=dat
  ; DOCUMENT save_rco(a,fn,bin=)
  ; Saves an RCO structure a to the binary file "fnX.bin" by converting
  ; all of its elements to float (double) and putting them into a
  ; single vector.

ns = size(b,/n_elements)
for nx=0,ns-1 do begin
    a = b(nx)
    r = a.r
    c = a.c
    n = a.n
    t = a.t    
    if size(a,/sname) eq 'RCO' then v = fltarr(n*2+r+6) $
    else if size(a,/sname) eq 'RCO_D' then v = dblarr(n*2+r+6) $
    else message,'Unsupported data type(s)'
    v(0:3) = [n,r,c,t]
    v(4:n+3) = (*a.xn)(0:n-1)
    v(n+4:2*n+3) = (*a.jx)(0:n-1)
    v(2*n+4:2*n+r+4) = (*a.ix)(0:r)
    if ns gt 1 then fname = fn+strcompress(nx,/rem) $
    else fname = fn
    if keyword_set(dat) then save,fname+'.dat',v $
    else writefits,fname+'.fits',v
endfor
end

;==================================================================
pro save_ruo,b,fn,dat=dat
  ; DOCUMENT save_rco(a,fn,bin=)
  ; Saves an RCO structure a to the binary file "fnX.bin" by converting
  ; all of its elements to float (double) and putting them into a
  ; single vector.

ns = size(b,/n_elements)
for nx=0,ns-1 do begin
    a = b(nx)
    r = a.r
    n = a.n
    t = a.t    
    if size(a,/sname) eq 'RUO' then v = fltarr(n*2+r*2+6) $
    else if size(a,/sname) eq 'RUO_D' then v = dblarr(n*2+r*2+6) $
    else message,'Unsupported data type(s)'
    v(0:2) = [n,r,t]
    v(3:n+2) = (*a.xn)(0:n-1)
    v(n+3:2*n+2) = (*a.jx)(0:n-1)
    v(2*n+3:2*n+r+2) = (*a.ix)(0:r-1)
    v(2*n+r+3:2*n+2*r+2) = (*a.xd)(0:r-1)
    if ns gt 1 then fname = fn+strcompress(nx,/rem) $
    else fname = fn
    if keyword_set(dat) then save,fname+'.dat',v $
    else writefits,fname+'.fits',v
endfor
end

;==================================================================
pro rcox,a,c
  ; DOCUMENT func rcox(a,c)
  ;  Multiplies RCO compressed matrix by a scalar.

if size(*a.xn,/type) ne size(c,/type) then message,'Mixed Data Types'
(*a.xn)(0:a.n-1) = c*(*a.xn)(0:a.n-1)
end

;==================================================================
pro ruox,a,c
  ; DOCUMENT func ruox(a,c)
  ;  Multiplies RUO matrix "a" by a scalar "c".

if size(*a.xn,/type) ne size(c,/type) then message,'Mixed Data Types'
(*a.xn)(0:a.n-1) = c*(*a.xn)(0:a.n-1)
(*a.xd)(0:a.r-1) = c*(*a.xd)(0:a.r-1)
end

;==================================================================
pro spinfo,a
  ; DOCUMENT func spinfo(a)
  ;  Prints information about a sparse matrix in RCO or RUO format.

if size(a,/tname) eq 'STRUCT' then begin
    nsx = size(a,/n_elements)
    for nx=0,nsx-1 do begin
        if nsx gt 1 then print,format='Block # %i\n',nx
        b = a(nx)
        stype = size(b,/sname)
        if b.n ne 0 then begin
            if stype eq 'RCO' then xfill = rcos(b)*100. 
            if stype eq 'RUO' then xfill = ruos(b)*100. 
        endif else message,'empty structure'
        ur = n_elements(*b.ix)
        un = n_elements(*b.xn)
        sfilln = float(b.n)/float(un)*100.
        sfillr = float(b.r)/float(ur)*100.
        ptype = size(*b.xn,/tname)
        print,stype+', '+ptype+'   stored      max.   usage'
        print,format='(%"no. rows : %8i %9i   %4.2f%%")',b.r,ur,sfillr 
        print,format='(%"elements : %8i %9i   %4.2f%%")',b.n,un,sfilln 
        if stype eq 'RCO' then print,format='(%"    cols : %8i")',b.c
        print,'matrix fill : '+strcompress(xfill,/rem)+'%'
    endfor
endif else message,'Argument not a RCO or RUO structure!'
end
