/******************************************************************
 
        MUSCLE SmartCard Development ( http://www.musclecard.com )
            Title  : xcardMain.cpp
            Author : David Corcoran
            Date   : 10/05/01
	    License: Copyright (C) 2001 David Corcoran
	             <corcoran@linuxnet.com>
            Purpose: This provides functionality for the xcard
	             personalization tool for the musclecard.
 
********************************************************************/ 

#include <qapplication.h>
#include <qtooltip.h>
#include <qpopupmenu.h>
#include <qmenudata.h>
#include <qmessagebox.h>
#include <qfiledialog.h>
#include <qprogressdialog.h>
#include <qdragobject.h>
#include <qurl.h>

#include "xcardPutKey.h"
#include "xcardPINV.h"
#include "xcardCreateObj.h"
#include "xcardManagePIN.h"
#include "xcardViewer.h"
#include "xcardMain.h"
#include "xcardPersonalize.h"

#ifdef MSC_TARGET_OSX
#include <PCSC/musclecard.h>
#else
#include <musclecard.h>
#endif

#include <string.h>
#include <stdlib.h>
#include <stdio.h>
#include <ctype.h>
#include <errno.h>

#ifdef MSC_TARGET_UNIX
#include <pthread.h>
#include <unistd.h>
#endif

#include <sys/types.h>
#include <sys/stat.h>

unsigned char *dataBuffer;

#define Inherited xcardMainData

/* MSCString pcsc_stringify_error ( MSCLong32 Error ); */

#ifdef MSC_TARGET_UNIX
pthread_t parentThread;
#endif

xcardMain *myGlobalClass;
MSCLPTokenInfo tokenArray;
MSCULong32 arrayLength;
static MSCULong32 suppFunctions = 0;
static MSCUChar8 idStateMain = 0;


MSCUChar8 AID[6] = {0xA0, 0x00, 0x00, 0x00, 0x01, 0x01}; 

QProgressDialog *progressDialog;

void xcardMainReaderEvent(LPSCARD_READERSTATE_A preaderState);


int ioCallbackForProgress(void *addparams, int currentProgress) {

    progressDialog->setProgress(currentProgress);

    qApp->processEvents();
    
    if ( progressDialog->wasCancelled() ) {
      progressDialog->setProgress(MSC_PERCENT_STEPSIZE);
      progressDialog->reset();
      return MSC_CANCELLED;
    } else {
      return MSC_SUCCESS;
    }
}


void mySignal(int sig) {



  //  myGlobalClass->setSizeGripEnabled( TRUE );

}


xcardMain::xcardMain
(
	QWidget* parent,
	const char* name
)
	:
	Inherited( parent, name )
{
  unsigned int i, p;
  MSCLong32 rv;

  setAcceptDrops(TRUE); 
  setCaption( "XCard II" );

  arrayLength    = 0;
  numStars       = 0;
  objectViewHex  = TRUE;
  getViewerDialog = new xcardViewer();
  putKeyDialog    = new xcardPutKey( &pConnection, this );
  progressDialog  = new QProgressDialog("Transferring data...", 
				       "Abort Transfer", 
				       MSC_PERCENT_STEPSIZE, this, 
				       "progress", TRUE );

  fileMenu    = new QPopupMenu();
  viewMenu    = new QPopupMenu();
  optionsMenu = new QPopupMenu();
  helpMenu    = new QPopupMenu();

  fileMenu->insertItem("Open Template", this,
		       SLOT(XCOpenObjectConfig()));
  fileMenu->insertItem("Save Template", this,
		       SLOT(XCSaveObjectConfig()));
  fileMenu->insertItem("Exit Program", this, 
		       SLOT(XCMenuExit()));

  viewMenu->insertItem("Refresh Contents", this,
		       SLOT(XCRefreshContents()));
  viewMenu->insertItem("Refresh Tokens", this,
		       SLOT(XCRefreshReaders()));

  optionsMenu->insertItem("Set Viewer", this,
			  SLOT(XCSetViewer()));
  optionsMenu->insertItem("Format Card", this,
			  SLOT(XCInitCard()));

  helpMenu->insertItem("About XCardII", this,
		       SLOT(XCAboutXCardII()));

  g_mainMenuBar1->insertItem("&File ", fileMenu);
  g_mainMenuBar1->insertItem("&View ", viewMenu);
  g_mainMenuBar1->insertItem("&Options ", optionsMenu);
  g_mainMenuBar1->setSeparator( QMenuBar::InWindowsStyle );
  g_mainMenuBar1->insertSeparator();
  g_mainMenuBar1->insertItem("&Help", helpMenu);


  pConnection.hCard = 0;

  myGlobalClass = this;

  for (i=0; i < XCARD_MAX_OBJECTS; i++) {
      objectList[i] = 0;
  }

  for (i=0; i < XCARD_MAX_KEYS; i++) {
      keyList[i] = 0;
  }

  for (i=0; i < XCARD_MAX_PINS; i++) {
      pinList[i] = 0;
  }

  for (i=0; i < XCARD_MAX_READERS; i++) {
    readerList[i] = 0;
  }

  memset(&pConnection, 0, sizeof(pConnection));
  memset(&clickedReader, 0, sizeof(clickedReader));

    do {
      arrayLength = 0;
      rv = MSCListTokens(MSC_LIST_ALL, 0, &arrayLength);
      if (tokenArray) { free(tokenArray); }
      tokenArray = (MSCLPTokenInfo)malloc(sizeof(MSCTokenInfo)*arrayLength);
      rv = MSCListTokens(MSC_LIST_ALL, tokenArray, &arrayLength);

      if ( rv != MSC_SUCCESS ) {
	sprintf(errorMessage, "%s%s", "MSCListTokens failed: ",
		msc_error(rv));
	g_statusLabel1->setText(errorMessage);
	break;
      }

      readerRoot =  new QListViewItem(g_browseReaders1, "Tokens");

      QToolTip::add(g_browseReaders1, 
		    "Double click an available\ntoken to begin");

      p = 0;

      for ( i=0; i < arrayLength; i++ ) {
	readerList[p] = new QListViewItem(readerRoot, 
					  tokenArray[i].tokenName);

        /* Associate list item with array */
	tokenArray[i].addParams = (void *)readerList[p];
	readerRoot->insertItem(readerList[p]);
	g_browseReaders1->ensureItemVisible(readerList[p]);

	/* Now set up the event handler */
       
	/*
	cardEventInf[p] = 
	  (LPCardEventInfo)malloc(sizeof(CardEventInfo));

	SCardEstablishEventCallback(hContext, &readerChars[i],
				    cardEventInf[p], 
				    (LPCardEventCallback)xcardMainReaderEvent,
				    this);
	*/
	
	++p;
      }

      mainBox = g_browseBox1;

      g_browseReaders1->insertItem(readerRoot); 
      g_statusLabel1->setText("Select a token ...");
      
    } while (0);
    
}

xcardMain::~xcardMain()
{
  SCardReleaseContext(hContext);
  exit(0);
}
void xcardMain::XCResetButtonClicked()
{
  /* Call Reconnect */
}


void xcardMain::XCReaderListClicked(QListViewItem* clickedItem) {

  MSCLong32 rv;
  //QListViewItem *itemsParent;
  //MSCUChar8 bytesCap[4];
  MSCULong32 capLength;
  unsigned int i;

  if ( clickedItem == readerRoot ) {
    return;
  }

  for (i=0; i < arrayLength; i++) {
    if ( clickedItem == tokenArray[i].addParams ) {
      clickedReader = &tokenArray[i];
      break;
    }
  }

  if ( i == arrayLength ) { printf("Bad error\n"); return; }
  
  if ( pConnection.hCard != 0 ) {
    rv = MSCReleaseConnection(&pConnection, MSC_LEAVE_TOKEN);
     pConnection.hCard = 0;
  }

  do {
    rv = MSCEstablishConnection(clickedReader, MSC_SHARE_SHARED, 0, 0,
				&pConnection);
    if ( rv != MSC_SUCCESS ) {
      sprintf(errorMessage, "%s%s", "MSCEstablishConnection failed: ",
	      msc_error(rv));
      g_statusLabel1->setText(errorMessage);
      sprintf(errorMessage, "Connection Failure: %s", 
	      msc_error(rv));
      
      QMessageBox::information( 0, "XCard II",
				errorMessage );
      break;
    } 
    
    sprintf(errorMessage, "%s%s", "Connected to: ",
	    (char *)clickedReader->tokenName);    
    g_statusLabel1->setText(errorMessage);

    rv = MSCGetCapabilities(&pConnection, MSC_TAG_SUPPORT_FUNCTIONS, 
			    (MSCPUChar8)&suppFunctions, &capLength);

    if ( rv != MSC_SUCCESS ) {
      suppFunctions = 0;
      sprintf(errorMessage, "Error: Token contains no capabilites");
      QMessageBox::warning( 0, "XCard II",
			  errorMessage );
      rv = MSCReleaseConnection(&pConnection, MSC_LEAVE_TOKEN);
      pConnection.hCard = 0;
      return;
    }

    rv = MSCGetCapabilities(&pConnection, MSC_TAG_CAPABLE_ID_STATE, 
			    (MSCPUChar8)&idStateMain, &capLength);

    if ( rv != MSC_SUCCESS ) {
      idStateMain = 0;
    }

    XCUpdateList();

  } while(0);

  strcpy(currentActiveReader, (char *)clickedReader->tokenName); 

}

void xcardMain::XCtargetObjectClicked(QListViewItem *clickedItem) {

  MSCLong32 rv;
  MSCULong32 keyNumber;
  QString clickedItemText;
  MSCObjectInfo objInfo;
  MSCKeyInfo keyInf;
  //MSCULong32 objectID;
  char infoPanelText[500];
  char tempText[50];
  char keyTypeString[50];
  //int i;
  //bool isOK;

  clickedItemText = clickedItem->text(0);
  strcpy(tempText, clickedItemText.latin1());

  if ( clickedItem->parent() == objectRoot ) {  
    strncpy(objInfo.objectID, &tempText[8], MSC_MAXSIZE_OBJID);

    rv = MSCGetObjectAttributes( &pConnection, &tempText[8], &objInfo );

    snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s\n%s\n%s\n%s",
	     "Object ID", "Object Size", "Read Access", "Write Access",
	     "Delete Access");
    
   g_infopanelLabel1->setText(infoPanelText);

   snprintf(infoPanelText, sizeof(infoPanelText),"%s\n%ld\n%X\n%X\n%X",
	    objInfo.objectID, objInfo.objectSize, 
	    objInfo.objectACL.readPermission, 
	    objInfo.objectACL.writePermission, 
	    objInfo.objectACL.deletePermission);

    g_infopanelLabel2->setText(infoPanelText);

  } else if ( clickedItem->parent() == keyRoot ) {

    keyNumber = strtoul(&tempText[5], 0, 10);

    /* List through keys until information is found */

    rv = MSCListKeys( &pConnection, MSC_SEQUENCE_RESET, &keyInf );
      
    while ( rv == MSC_SUCCESS ) {
      if ( keyInf.keyNum == keyNumber )
	break;
      rv = MSCListKeys( &pConnection, MSC_SEQUENCE_NEXT, &keyInf );
    }    

    snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s\n%s\n%s\n%s\n%s",
	     "Key Number", "Key Type", "Key Size", "Read Access", 
	     "Write Access", "Use Access");
    
    g_infopanelLabel1->setText(infoPanelText);
    

    switch(keyInf.keyType) {
    case MSC_KEY_RSA_PUBLIC:
      strncpy(keyTypeString, "RSA Public", sizeof(keyTypeString));
      break;
    case MSC_KEY_RSA_PRIVATE:
      strncpy(keyTypeString, "RSA Priv Exp", sizeof(keyTypeString)); 
      break;
    case MSC_KEY_RSA_PRIVATE_CRT:
      strncpy(keyTypeString, "RSA Priv CRT", sizeof(keyTypeString));
      break;
    case MSC_KEY_DSA_PUBLIC:
      strncpy(keyTypeString, "DSA Public", sizeof(keyTypeString));
      break;
    case MSC_KEY_DSA_PRIVATE:
      strncpy(keyTypeString, "DSA Private", sizeof(keyTypeString));
      break;
    case MSC_KEY_DES:
      strncpy(keyTypeString, "DES", sizeof(keyTypeString));
      break;
    case MSC_KEY_3DES:
      strncpy(keyTypeString, "3DES", sizeof(keyTypeString));
      break;
    case MSC_KEY_3DES3:
      strncpy(keyTypeString, "3 Key 3DES", sizeof(keyTypeString));
      break;
    default:
      strncpy(keyTypeString, "Unknown Key", sizeof(keyTypeString));
      break;
    }

    snprintf(infoPanelText, sizeof(infoPanelText),"%d\n%s\n%d\n%X\n%X\n%X",
	     keyInf.keyNum, keyTypeString, keyInf.keySize, 
	     keyInf.keyACL.readPermission, 
	     keyInf.keyACL.writePermission, 
	     keyInf.keyACL.usePermission);

    g_infopanelLabel2->setText(infoPanelText);
    
  } else if (clickedItem->parent() == pinRoot) {
    /* TODO : RFU */
    if ( idStateMain ) {
      snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s\n%s",
	       "PIN Number", "PIN Policy", "State");
      g_infopanelLabel1->setText(infoPanelText);
      
      snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s\n%s",
	       clickedItem->text(0).latin1(), "Default", 
	       clickedItem->text(1).latin1());
      g_infopanelLabel2->setText(infoPanelText);
    } else {
      snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s",
	       "PIN Number", "PIN Policy");
      g_infopanelLabel1->setText(infoPanelText);
      
      snprintf(infoPanelText, sizeof(infoPanelText), "%s\n%s",
	       clickedItem->text(0).latin1(), "Default");
      g_infopanelLabel2->setText(infoPanelText);
    }


  } else {
    g_infopanelLabel1->setText("No Data Available");
    g_infopanelLabel2->setText(" ");
  }

}

void xcardMain::xcardMainAddCard(xcardMain *myClass, const char *readerName, 
				 MSCPUChar8 idValue,
				 MSCULong32 idLength ) {

  int i;
  unsigned int p;
  QString eventReader;
  char cardIDString[100];
  char cardMessage[200];
  QListViewItem *newCard;


  for (i=0; i < XCARD_MAX_READERS; i++) {
    eventReader = readerList[i]->text(0);
    if (!strcmp(eventReader.latin1(), readerName) ) {
      break;
    }
  }

  if ( i != XCARD_MAX_READERS ) {
    if ( readerList[i]->firstChild() ) {
      readerList[i]->takeItem(readerList[i]->firstChild());
    }

    for (p=0; p<idLength; p++) {
      sprintf(&cardIDString[p*2], "%02X ", idValue[p]);
    }

    newCard = new QListViewItem(readerList[i], cardIDString);
    readerList[i]->insertItem(newCard);

    snprintf(cardMessage, sizeof(cardMessage), 
	     "A Card has been Inserted\n%s", cardIDString);


#ifdef MSC_QT_THREADS
    qApp->lock();

    QMessageBox::information( 0, "XCard II",
			     cardMessage );

    qApp->unlock(TRUE);
#endif

  }

}

void xcardMain::xcardMainRemoveCard(xcardMain *myClass,
				    const char *readerName ) {

  int i;
  QString eventReader;
  QListViewItem *readersChild;

  for (i=0; i < XCARD_MAX_READERS; i++) {
    eventReader = readerList[i]->text(0);
    if (!strcmp(eventReader.latin1(), readerName) ) {
      break;
    }
  }

  if ( i != XCARD_MAX_READERS ) {
    if ( readerList[i]->firstChild() ) {
      readersChild = readerList[i]->firstChild();
      readerList[i]->takeItem(readersChild);
    }
  }

  if (!strcmp(currentActiveReader, readerName)) {
    myClass->g_browseBox1->clear();
  }

#ifdef MSC_QT_THREADS
  w
qApp->lock();
  QMessageBox::information( 0, "XCard II",
			    "A Card has been Removed");
  qApp->unlock(TRUE);
#endif

}

void xcardMainReaderEvent(LPSCARD_READERSTATE_A preaderState) {


  xcardMain *xcardMainChannel;
  xcardMainChannel = (xcardMain *)preaderState->pvUserData;

  printf("Update %lX\n", preaderState->dwEventState);

  if ( preaderState->dwEventState & SCARD_STATE_PRESENT ) {
    xcardMainChannel->xcardMainAddCard(xcardMainChannel,
				       preaderState->szReader,
				       preaderState->rgbAtr,
				       preaderState->cbAtr);

  } else if (preaderState->dwEventState & SCARD_STATE_EMPTY ) {
    xcardMainChannel->xcardMainRemoveCard(xcardMainChannel,
					  preaderState->szReader);
				       
  }

}

void xcardMain::XCRightSelectObjectType(QListViewItem *itemSelected,
					const QPoint &localPoint, 
					int id ) {

  QPopupMenu *rightClickMenu;

  rightClickMenu = new QPopupMenu();

  do {

    if ( itemSelected == 0 ) {
      break;
    }

    itemCurrentlySelected = itemSelected;

    /* Main properties */
    if ( itemSelected == theRoot ) {
      rightClickMenu->insertItem("Properties", this,
				 SLOT(XCProperties()));
    } else if ( itemSelected->parent() == 0 ) {
      break;
    } else if ( itemSelected == objectRoot ) {
     if ( suppFunctions & MSC_SUPPORT_CREATEOBJECT ) { 
       rightClickMenu->insertItem("Create Object", this,
				  SLOT(XCCreateObject()));
       rightClickMenu->insertItem("Open Template", this,
				  SLOT(XCOpenObjectConfig()));
       rightClickMenu->insertItem("Save Template", this,
				  SLOT(XCSaveObjectConfig()));
     }      

     if ( suppFunctions & MSC_SUPPORT_DELETEOBJECT ) {
       rightClickMenu->insertItem("Delete All", this,
				  SLOT(XCDeleteAll()));
     }
      
    } else if ( itemSelected == keyRoot ) {
      if ( suppFunctions & MSC_SUPPORT_GENKEYS ) {
	rightClickMenu->insertItem("Generate Keys", this,
				   SLOT(XCGenerateKey()));
	rightClickMenu->insertItem("Regenerate Keys", this,
				   SLOT(XCReGenerateKey()));
      }

      if ( suppFunctions & MSC_SUPPORT_IMPORTKEY ) {
	rightClickMenu->insertItem("Import New Key", this,
				   SLOT(XCImportKey()));
      }

    } else if ( itemSelected == pinRoot ) {
      if ( suppFunctions & MSC_SUPPORT_CREATEPIN ) {
	rightClickMenu->insertItem("Create PIN", this,
				   SLOT(XCCreatePIN()));
      }

      if ( suppFunctions & MSC_SUPPORT_LOGOUTALL ) {
	rightClickMenu->insertItem("Logout All", this, 
				   SLOT(XCLogoutAll()));
      }

    } else if ( itemSelected->parent() == pinRoot ) {
      if ( suppFunctions & MSC_SUPPORT_VERIFYPIN ) {
	rightClickMenu->insertItem("Verify PIN", this, 
				   SLOT(XCVerifyPIN()));
      }

      if ( suppFunctions & MSC_SUPPORT_CHANGEPIN ) {
	rightClickMenu->insertItem("Change PIN", this, 
				   SLOT(XCChangePIN()));
      }

      if ( suppFunctions & MSC_SUPPORT_UNBLOCKPIN ) {
	rightClickMenu->insertItem("Unblock PIN", this, 
				   SLOT(XCUnblockPIN()));
      }

    } else if ( itemSelected->parent() == objectRoot ) {
      if ( suppFunctions & MSC_SUPPORT_WRITEOBJECT ) {
	rightClickMenu->insertItem("Import Data", this, 
				   SLOT(XCImportObject()));
      }

      if ( suppFunctions & MSC_SUPPORT_READOBJECT ) {
	rightClickMenu->insertItem("Export Data", this, 
				   SLOT(XCExportObject()));
      }

      if ( suppFunctions & MSC_SUPPORT_DELETEOBJECT ) {
	rightClickMenu->insertItem("Delete Object", this, 
				   SLOT(XCDeleteObject()));
      }

      if ( suppFunctions & MSC_SUPPORT_READOBJECT ) {
	rightClickMenu->insertItem("View Object", this, 
				   SLOT(XCViewObject()));
      }

    } else if ( itemSelected->parent() == keyRoot ) {

      if ( suppFunctions & MSC_SUPPORT_IMPORTKEY ) {
	rightClickMenu->insertItem("Import Key", this,
				   SLOT(XCImportKey()));
      }

      if ( suppFunctions & MSC_SUPPORT_EXPORTKEY ) {
	rightClickMenu->insertItem("Export Key", this,
				   SLOT(XCExportKey()));
      }
    } else {
      break;
    }
        
    rightClickMenu->popup(localPoint);
        
  } while (0);


}


void xcardMain::XCVerifyPIN() {  

  xcardPINV *pinDialog;
  QString itemString;
  MSCUChar8 pinDialogString[50];
  MSCULong32 pinNumber;
  
  itemString = itemCurrentlySelected->text(0);
  strcpy((char *)pinDialogString, itemString.latin1());
  pinNumber = strtoul((char *)&pinDialogString[5], 0, 10);
  
  pinDialog = new xcardPINV(&pConnection, itemCurrentlySelected,
			    pinNumber, FALSE);

  pinDialog->show();
}

void xcardMain::XCUnblockPIN() {  

  xcardPINV *pinDialog;
  QString itemString;
  MSCUChar8 pinDialogString[50];
  MSCULong32 pinNumber;
  
  itemString = itemCurrentlySelected->text(0);
  strcpy((char *)pinDialogString, itemString.latin1());
  pinNumber = strtoul((char *)&pinDialogString[5], 0, 10);
  
  pinDialog = new xcardPINV(&pConnection, itemCurrentlySelected,
			    pinNumber, TRUE);

  pinDialog->show();

}

void xcardMain::XCChangePIN() {  
  XCCreatePIN();
}

void xcardMain::XCViewObject() {

  MSCLong32 rv;
  QString clickedItemText;
  QString xcardDefaultViewer;
  char tempText[75];
  char systemExecString[200];
  char objectID[MSC_MAXSIZE_OBJID];

#ifdef MSC_TARGET_UNIX
#define XCARD_OUTPUT_FILE "/tmp/xcard0102"
#else
#define XCARD_OUTPUT_FILE "C:\\temp\\xcard0102.bin"
#endif

  dataBuffer = 0;

  do {    
    clickedItemText = itemCurrentlySelected->text(0);
    strcpy(tempText, clickedItemText.latin1());
    
    strncpy(objectID, &tempText[8], MSC_MAXSIZE_OBJID);

    XCOutputDataObject(objectID, XCARD_OUTPUT_FILE, TRUE);

    xcardDefaultViewer = getViewerDialog->getDefaultViewer();

    snprintf(systemExecString, sizeof(systemExecString), "%s %s",
	     (char *)xcardDefaultViewer.latin1(), XCARD_OUTPUT_FILE);

printf("output  %s\n", XCARD_OUTPUT_FILE);
printf("sysexec %s\n", systemExecString);

    rv = system(systemExecString);

    if ( rv == -1 || rv == 127 ) {
      QMessageBox::critical( 0, "XCard II",
			     "Could Not Launch Viewer.");
    } 

#ifdef MSC_TARGET_UNIX
    unlink(XCARD_OUTPUT_FILE);
#endif

  } while (0);

}

void xcardMain::XCImportObject() {  

  //MSCLong32 rv;
  QString clickedItemText;
  QString importFile;
  char tempText[75];
  char objectID[MSC_MAXSIZE_OBJID];

  do {
    
    clickedItemText = itemCurrentlySelected->text(0);
    strcpy(tempText, clickedItemText.latin1());
    
    strncpy(objectID, &tempText[8], MSC_MAXSIZE_OBJID);
    
    importFile = QFileDialog::getOpenFileName(".", "*");
    
    if ( importFile.isNull() ) {
      printf("NULL Filename\n");
      break;
    }
    
    XCInputDataObject(objectID, (char *)importFile.latin1());
    
  } while (0);

}

void xcardMain::XCExportObject() {  

  QString clickedItemText;
  char tempText[75];
  char objectID[MSC_MAXSIZE_OBJID];
  QString exportFile;

  dataBuffer = 0;

  do {    
    clickedItemText = itemCurrentlySelected->text(0);
    strcpy(tempText, clickedItemText.latin1());
    
    strncpy(objectID, &tempText[8], MSC_MAXSIZE_OBJID);    

    exportFile = QFileDialog::getSaveFileName(".", "*");
    
    if ( exportFile.isNull() ) {
      break;
    }
    
    XCOutputDataObject(objectID, (char *)exportFile.latin1(), FALSE);
  } while (0);

}


void xcardMain::XCInputDataObject(char *objectID,
				  char *inputFile) {

  MSCLong32 rv;
  char errorMessage[200];
  //MSCULong32 dataSize;
  unsigned int st_size;
  MSCObjectInfo objInf;

  st_size=0;

  QFile inFile(inputFile);
	st_size = inFile.size();
	inFile.open(IO_ReadOnly);

  do {
	printf("Input file %s\n", inputFile);

    if ( inFile.exists() == FALSE ) {
      snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
	       "Open File Failed\nCode: ",
	       "File not found");
      QMessageBox::critical( 0, "XCard II",
			     QString(errorMessage));

      break;
    }
    
    rv = MSCGetObjectAttributes(&pConnection, objectID, &objInf);
    if (XCHECK_RESP(rv, "Get Object Info Failed\nCode: ")) {
      break;
    }
    
    if ( st_size >  objInf.objectSize ) {
      XCHECK_RESP(MSC_NO_MEMORY_LEFT, "Write Object Failed\nCode: ");
      break;
    }

    dataBuffer = (unsigned char *)malloc(sizeof(unsigned char)*
				  st_size);
    
	inFile.flush();

	if ((unsigned)inFile.readBlock((char *)dataBuffer, st_size) != st_size) {
	snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
		 "Read File Failed\nCode: ",
		 "Not enough data");
	QMessageBox::critical( 0, "XCard II",
			       QString(errorMessage));
	break;
      } 

	rv = MSCWriteObject(&pConnection, objectID, 0, 
			    dataBuffer, st_size,
			    (LPRWEventCallback)&ioCallbackForProgress,
			    NULL);
    
    if (XCHECK_RESP(rv, "Write Object Failed\nCode: ")) {
      break;
    }

    snprintf(errorMessage, sizeof(errorMessage), 
	     "Your object %s has been written from:\n%s", objectID,
	     inputFile);
    QMessageBox::information( 0, "XCard II",
			      QString(errorMessage));
    
  } while (0);


  inFile.close();

}



void xcardMain::XCOutputDataObject(char *objectID, 
				   char *exportFile,
				   bool suppressMesg) {
  
  MSCLong32 rv;
  char errorMessage[200];
  FILE *outFile;
  MSCULong32 dataSize;
  outFile    = 0;
  dataBuffer = 0;

  do {        
    outFile = fopen(exportFile, "w+");
    if ( outFile == 0 ) {
      snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
	       "Open File Failed\nCode: ",
	       strerror(errno));
      QMessageBox::critical( 0, "XCard II",
			     QString(errorMessage));

      break;
    }

#ifdef MSC_TARGET_UNIX
    chmod(exportFile, 0600);
#endif
    
    rv = MSCReadAllocateObject(&pConnection, objectID, 
			       &dataBuffer, &dataSize, 
			       (LPRWEventCallback)&ioCallbackForProgress,
			       NULL);
    
    if (XCHECK_RESP(rv, "Read Object Failed\nCode: ")) {
      break;
    }
    
    if ( fwrite(dataBuffer, 1, dataSize, outFile) != dataSize ) {
      snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
	       "Write File Failed\nCode: ",
	       strerror(errno));
      QMessageBox::critical( 0, "XCard II",
			     QString(errorMessage));
      break;
    }
    if ( suppressMesg == FALSE ) {
      snprintf(errorMessage, sizeof(errorMessage), 
	       "Your object %s has been written to:\n%s", objectID,
	       exportFile);
      QMessageBox::information( 0, "XCard II",
				QString(errorMessage));
    }    

  } while (0);


  if ( outFile ) {
    fclose(outFile);
  }

  if ( dataBuffer ) {
    free(dataBuffer);
  }

}

void xcardMain::XCDeleteObject() {  

  MSCLong32 rv;
  QString clickedItemText;
  char tempText[75];
  char objectID[MSC_MAXSIZE_OBJID];

  do {

    clickedItemText = itemCurrentlySelected->text(0);
    strcpy(tempText, clickedItemText.latin1());
    
    strncpy(objectID, &tempText[8], MSC_MAXSIZE_OBJID);

    rv = MSCDeleteObject( &pConnection, objectID, MSC_ZF_WRITE_ZERO );
    
    if (XCHECK_RESP(rv, "Unable to Delete Object\nCode: ")) {
      break;
    }

    XCUpdateList();

  } while (0);


}

void xcardMain::XCViewObjectParams(){  

  MSCLong32 rv;
  char propertiesString[500];
  MSCObjectInfo objInf;
  QString clickedItemText;
  char tempText[75];
  //char errorMessage[200];
  char objectID[MSC_MAXSIZE_OBJID];

  do { 
    
    clickedItemText = itemCurrentlySelected->text(0);
    strcpy(tempText, clickedItemText.latin1());
    
    strncpy(objectID, &tempText[8], MSC_MAXSIZE_OBJID);

    rv = MSCGetObjectAttributes(&pConnection, objectID, &objInf);
    if (XCHECK_RESP(rv, "Get Object Info Failed\nCode: ")) {
      break;
    }

   snprintf(propertiesString, sizeof(propertiesString), 
	    "%s%p\n%s%ld\n", 
	    "Object ID:    ", objInf.objectID,
	    "Object Size:  ", objInf.objectSize
	    );

   QMessageBox::information( this, "XCard II",
			     propertiesString );

  } while (0);

}

void xcardMain::XCExportKey() {  

  MSCLong32 rv;
  char errorMessage[200];
  FILE *outFile;
  MSCULong32 dataSize;
  QString itemString;
  QString exportFile;
  MSCUChar8 keyDialogString[50];
  MSCULong32 keyNumber;
  outFile    = 0;
  dataBuffer = 0;  

  itemString = itemCurrentlySelected->text(0);
  strcpy((char *)keyDialogString, itemString.latin1());
  keyNumber = strtoul((char *)&keyDialogString[5], 0, 10);
  
  do {        

    exportFile = QFileDialog::getSaveFileName(".", "*");
    
    if ( exportFile.isNull() ) {
      break;
    }
    
    outFile = fopen(exportFile.latin1(), "w+");
    if ( outFile == 0 ) {
      snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
	       "Open File Failed\nCode: ",
	       strerror(errno));
      QMessageBox::critical( 0, "XCard II",
			     QString(errorMessage));

      break;
    }
    

    dataBuffer = (unsigned char *)malloc(sizeof(char)*5000);
    dataSize   = 5000;

    rv = MSCExportKey(&pConnection, (MSCUChar8)keyNumber, dataBuffer,
		      &dataSize, 0, 0);

    if (XCHECK_RESP(rv, "Export Key Failed\nCode: ")) {
      break;
    }

    if ( fwrite(dataBuffer, 1, dataSize, outFile) != dataSize ) {
      snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
	       "Write File Failed\nCode: ",
	       strerror(errno));
      QMessageBox::critical( 0, "XCard II",
			     QString(errorMessage));
      break;
    } 
    
    snprintf(errorMessage, sizeof(errorMessage), 
	     "Your key #%ld has been written to:\n%s", keyNumber,
	     exportFile.latin1());
    QMessageBox::information( 0, "XCard II",
			      QString(errorMessage));
    

  } while (0);


  if ( outFile ) {
    fclose(outFile);
  }

  if ( dataBuffer ) {
    free(dataBuffer);
  }


}

void xcardMain::XCViewKeyParams() {

}


void xcardMain::XCProperties() {

  MSCLong32 rv;
  char propertiesString[500];
  MSCStatusInfo statusInf;

  do { 
    
    rv = MSCGetStatus(&pConnection, &statusInf);
    if (XCHECK_RESP(rv, "Get Status Info Failed\nCode: ")) {
      break;
    }

   snprintf(propertiesString, sizeof(propertiesString), 
	    "%s\n%s%s\n%s%d%s%d\n%s%d%s%d\n%s%ld\n%s%ld\n%s%d\n%s%d\n%s%X\n",
	    "XCARD II",
	    "Author:                ", "David Corcoran",
	    "Protocol Version:  ",  statusInf.appVersion >> 8,
	    ".",                     statusInf.appVersion & 0xFF,
	    "Software Version:  ",  statusInf.swVersion >> 8,
	    ".",                     statusInf.swVersion & 0xFF, 
	    "Free Memory:       ",   statusInf.freeMemory,
	    "Total Memory:       ",  statusInf.totalMemory,
	    "Used PINS:           ", statusInf.usedPINs,
	    "Used Keys:           ", statusInf.usedKeys,
	    "Logged IDs:          ", statusInf.loggedID);


   QMessageBox::information( this, "XCard II",
			     propertiesString );

  } while (0);

}


void xcardMain::XCCreateObject() {
  xcardCreateObj *createObjWindow;
  createObjWindow = new xcardCreateObj(&pConnection, this, 0);
  createObjWindow->show();
}

void xcardMain::XCGenerateKey() {
  
  putKeyDialog->XCInitializeKeyDialog(TRUE, FALSE, -1);
  putKeyDialog->show();



}

void xcardMain::XCImportKey() {

  int ret;
  MSCULong32 keyNumber;
  char keyDialogString[50];


  if (itemCurrentlySelected == keyRoot) {
    ret = putKeyDialog->XCInitializeKeyDialog(FALSE, FALSE, -1);
  } else {
    QString itemString = itemCurrentlySelected->text(0);
    strcpy((char *)keyDialogString, itemString.latin1());
    keyNumber = strtoul((char *)&keyDialogString[5], 0, 10);
    ret = putKeyDialog->XCInitializeKeyDialog(FALSE, FALSE, keyNumber);
  }

  if ( ret == -1 ) {
    return;
  }

  putKeyDialog->show();

}

void xcardMain::XCCreatePIN() {

  xcardManagePIN *pinDialog;
  QString itemString;
  MSCUChar8 pinDialogString[50];
  MSCULong32 pinNumber;
  
  if (itemCurrentlySelected == pinRoot) {
    pinNumber = 0xFFFF;
  } else {
    itemString = itemCurrentlySelected->text(0);
    strcpy((char *)pinDialogString, itemString.latin1());
    pinNumber = strtoul((char *)&pinDialogString[5], 0, 10);
  }

  pinDialog = new xcardManagePIN(&pConnection, pinNumber, this);

  pinDialog->show();

  printf("Create/Change PIN\n");

}

void xcardMain::XCLogoutAll() {

  MSCLong32 rv;

  rv = MSCLogoutAll(&pConnection);
  
  XCUpdateList();

  if ( XCHECK_RESP(rv, "MSCLogoutAll Failed\nCode: ") ) {
    return;
  }
  
}

 void xcardMain::XCUpdateMemory() {

  MSCLong32 rv;
  int memoryAvailable;
  double memoryPercent, freeMem, totalMem;
  MSCStatusInfo statusInf;

  rv = MSCGetStatus(&pConnection, &statusInf);
  
   if ( XCHECK_RESP(rv, "MSCGetStatus Failed\nCode: ") ) {
     return;
   }
  
  g_memoryLCD1->setSegmentStyle(QLCDNumber::Filled);
  g_memoryLCD1->display((int)statusInf.freeMemory);
    
  freeMem = statusInf.freeMemory;
  totalMem = statusInf.totalMemory;
  
  memoryPercent   = freeMem / totalMem;
  memoryAvailable = (int)(100 * memoryPercent);
  
  g_freeMemoryBar1->setTotalSteps(100);
  g_freeMemoryBar1->setProgress(0);
  g_freeMemoryBar1->setProgress(memoryAvailable);
  
}

 void xcardMain::XCUpdateList() {

  MSCLong32 rv;
  MSCObjectInfo objInfo;
  int p;
  unsigned int i;
  char idString[25];
  MSCStatusInfo statusInf;
  MSCKeyInfo keyInf;
  MSCUShort16 pinInf;

  g_browseBox1->clear();

  if ( clickedReader == 0 ) {
    snprintf(errorMessage, sizeof(errorMessage), "%s", 
	     "No token has been selected");
    QMessageBox::critical( 0, "XCard II",
			   QString(errorMessage));
    return;
  }    

  clickedReader->tokenState = MSC_STATE_UNAWARE;

  rv = MSCWaitForTokenEvent(clickedReader, 1, 100);

  if ( clickedReader->tokenState & MSC_STATE_EMPTY ) {
    snprintf(errorMessage, sizeof(errorMessage), "%s", 
	     "The token has been removed");
    QMessageBox::critical( 0, "XCard II",
			   QString(errorMessage));

    return;
  } else if ( clickedReader->tokenState & MSC_STATE_UNAVAILABLE ) {
    snprintf(errorMessage, sizeof(errorMessage), "%s", 
	     "The token and slot are removed");
    QMessageBox::critical( 0, "XCard II",
			   QString(errorMessage));
    return;
  } else if ( clickedReader->tokenState & MSC_STATE_UNKNOWN ) {
    snprintf(errorMessage, sizeof(errorMessage), "%s", 
	     "The token and slot are removed");
    QMessageBox::critical( 0, "XCard II",
			   QString(errorMessage));
    return;
  }


  theRoot    = new QListViewItem(g_browseBox1, clickedReader->tokenName);
  keyRoot    = new QListViewItem(theRoot, "KEYS");
  objectRoot = new QListViewItem(theRoot, "OBJECTS");
  pinRoot    = new QListViewItem(theRoot, "PINS");

  g_browseBox1->ensureItemVisible(keyRoot); 
  g_browseBox1->ensureItemVisible(objectRoot);
  g_browseBox1->ensureItemVisible(pinRoot);  

  XCHandleLongOperation();

  rv = MSCListObjects( &pConnection, MSC_SEQUENCE_RESET, &objInfo );
  p=0;
  
  while ( rv == MSC_SUCCESS ) {

    snprintf(idString, sizeof(idString), "Object  %s", objInfo.objectID);

    objectList[p] = new QListViewItem(objectRoot, idString);
    objectRoot->insertItem(objectList[p]);
    g_browseBox1->ensureItemVisible(objectList[p]);
    
    rv = MSCListObjects( &pConnection, MSC_SEQUENCE_NEXT, &objInfo );
    XCHandleLongOperation();
    ++p;
  }
  
  rv = MSCListKeys( &pConnection, MSC_SEQUENCE_RESET, &keyInf );
  p=0;
  
  while ( rv == MSC_SUCCESS ) {
    snprintf(idString, sizeof(idString), "Key #%d", keyInf.keyNum);
      keyList[p] = new QListViewItem(keyRoot, idString, "Unverified");
      keyRoot->insertItem(keyList[p]);
      g_browseBox1->ensureItemVisible(keyList[p]);
      
      rv = MSCListKeys( &pConnection, MSC_SEQUENCE_NEXT, &keyInf );
      XCHandleLongOperation();
      ++p;
  }    
  
    pinInf = 0;
    
    rv = MSCGetStatus( &pConnection, &statusInf );
    XCHandleLongOperation();

    rv = MSCListPINs( &pConnection, &pinInf );
    p=0;
    
    for (i=1; i < 0xFFFF; i <<= 1) {
      if (pinInf & i) {
	snprintf(idString, sizeof(idString), "Pin #%d", p);

	if ( i & statusInf.loggedID ) {
	  if ( idStateMain ) {
	    pinList[p] = new QListViewItem(pinRoot, idString, "Verified");
	  } else {
	    pinList[p] = new QListViewItem(pinRoot, idString, "Unreported");
	  }
	} else {
	  if ( idStateMain ) {
	    pinList[p] = new QListViewItem(pinRoot, idString, "Unverified");
	  } else {
	    pinList[p] = new QListViewItem(pinRoot, idString, "Unreported");
	  }
	}
	pinRoot->insertItem(pinList[p]);
	g_browseBox1->ensureItemVisible(pinList[p]);	
      }
      ++p;
    }

    XCHandleLongOperation();

    g_browseBox1->insertItem(theRoot); 

    XCUpdateMemory();

    XCCancelLongOperation();
}


int xcardMain::XCGetFreeMemory() {

  return g_memoryLCD1->intValue();
}

 int xcardMain::XCHECK_RESP(int rv, char *message) {

  char errorMessage[200];

  if ( rv != MSC_SUCCESS ) {
    sprintf(errorMessage, "%s%s", message,
	    msc_error(rv));
    QMessageBox::critical( 0, "XCard II",
			   QString(errorMessage));
    return -1;
  } else {
    return 0;
  }
}

void xcardMain::XCOpenObjectConfig() {

  MSCLong32 rv;
  char errorMessage[200];
  //MSCObjectInfo objInfo;
  QString importFile;
  MSCObjectACL objACL;
  char objectID[MSC_MAXSIZE_OBJID];
  int temp_perm;
  MSCULong32 objectSize;
  FILE *fp;

  do {    
    importFile = QFileDialog::getOpenFileName(".", "*");
    
    if ( importFile.isNull() ) {
      printf("NULL Filename\n");
      break;
    } 
    
    fp = fopen(importFile.latin1(), "r");

    if ( fp == NULL ) {
      printf("Can't open file\n");
      break;
    } 
        
    do {
      rv = fscanf(fp, "OBJECT_ID=%s\n", objectID);
      if (rv == EOF) break;
      rv = fscanf(fp, "OBJECT_SIZE=%ld\n", &objectSize);
      if (rv == EOF) break;

      rv = fscanf(fp, "OBJECT_RPERM=%d\n", &temp_perm);
      if (rv == EOF)
		  break;
	  else
		  objACL.readPermission = temp_perm;

      rv = fscanf(fp, "OBJECT_WPERM=%d\n", &temp_perm);
      if (rv == EOF)
		  break;
	  else
		  objACL.writePermission = temp_perm;

      rv = fscanf(fp, "OBJECT_DPERM=%d\n", &temp_perm);    
      if (rv == EOF)
		  break;
	  else
		  objACL.deletePermission = temp_perm;

      rv = MSCCreateObject(&pConnection, objectID, objectSize, &objACL);

      XCHandleLongOperation();

      if ( rv != MSC_SUCCESS ) {
	snprintf(errorMessage, sizeof(errorMessage), "%s%s", 
		 "Create File Failed (Aborted)\nCode: ",
		 msc_error(rv));
	QMessageBox::critical( 0, "XCard II",
			       QString(errorMessage));
	break;
      }	
      
    } while (1);
 
    XCUpdateList();
    XCCancelLongOperation();   
    fclose(fp);
    
  } while (0);

}


void xcardMain::XCSaveObjectConfig() {

  MSCLong32 rv;
  MSCObjectInfo objInfo;
  QString exportFile;
  FILE *fp;

  do {    
    exportFile = QFileDialog::getSaveFileName(".", "*");
    
    if ( exportFile.isNull() ) {
      printf("NULL Filename\n");
      break;
    } 
    
    fp = fopen(exportFile.latin1(), "w");

    if ( fp == NULL ) {
      printf("Can't open file\n");
      break;
    } 
        
    rv = MSCListObjects( &pConnection, MSC_SEQUENCE_RESET, &objInfo );
    
    while ( rv == MSC_SUCCESS ) {
      fprintf(fp, "OBJECT_ID=%s\n", objInfo.objectID);
      fprintf(fp, "OBJECT_SIZE=%ld\n", objInfo.objectSize);
      fprintf(fp, "OBJECT_RPERM=%d\n", objInfo.objectACL.readPermission);
      fprintf(fp, "OBJECT_WPERM=%d\n", objInfo.objectACL.writePermission);
      fprintf(fp, "OBJECT_DPERM=%d\n", objInfo.objectACL.deletePermission);      
      rv = MSCListObjects( &pConnection, MSC_SEQUENCE_NEXT, &objInfo );
      XCHandleLongOperation();
    }

    XCCancelLongOperation();    

    fclose(fp);
    
  } while (0);

}


void xcardMain::XCDeleteAll() {

  MSCLong32 rv;
  MSCObjectInfo objInfo;

  do {    

    int ret = QMessageBox::warning( 0, "XCard II",
				    "Are you sure you want to Delete All ?", 
				    "OK", "Cancel", 0, 0, 1);     
    if ( ret == 1 ) {
      break;
    }

    rv = MSCListObjects( &pConnection, MSC_SEQUENCE_RESET, &objInfo );
    
    while ( rv == MSC_SUCCESS ) {
      rv = MSCDeleteObject( &pConnection, objInfo.objectID, MSC_ZF_WRITE_ZERO );
      rv = MSCListObjects( &pConnection, MSC_SEQUENCE_NEXT, &objInfo );
      XCHandleLongOperation();
    }
    
    XCUpdateList();

    XCCancelLongOperation();

  } while (0);

}

void xcardMain::XCMenuExit() {
  qApp->exit(0);
  exit(0);
}

void xcardMain::dragEnterEvent(QDragEnterEvent* event)
{
  event->accept();
}

void xcardMain::dropEvent(QDropEvent* event)
{
  QStrList fileList;
  QStringList strList;
  QString myFile;

  if ( QUriDrag::decode(event, fileList) == TRUE ) {
    strList = QStringList::fromStrList(fileList);

	myFile = strList.first();

#ifdef MSC_TARGET_UNIX

#else
	QUrl myURL(myFile);
	myFile = myURL.path();

#endif

#ifdef MSC_TARGET_UNIX
    myFile.remove(0, 5);
#else 
	myFile.remove(0,2);
	myFile.replace(1, 1, ":");
#endif

    xcardCreateObj *createObjWindow;
    createObjWindow = new xcardCreateObj(&pConnection, this, 
					 (char *)myFile.latin1());
    if (createObjWindow->isFileOK == TRUE) {
      createObjWindow->show();
    }
  }

}

void xcardMain::XCCancelLongOperation() {
  char currString[200];

  numStars = 0;

  snprintf(currString, sizeof(currString), "Connected to: %s",
	   clickedReader -> tokenName);

  g_statusLabel1->setText(currString);
}

void xcardMain::XCHandleLongOperation() {

  char myStarString[50];
  int i;

  numStars += 1;

  if ( numStars == 20 ) {
    numStars = 1;
  }

  strcpy(myStarString, "Processing: ");

  for (i=11; i < numStars+11; i++) {
    myStarString[i] = '-';
  }
  
  myStarString[i] = 0;
  g_statusLabel1->setText(myStarString);

  qApp->processEvents();

}

void xcardMain::XCChangeView() {

  if ( objectViewHex == TRUE ) {
    objectViewHex = FALSE;
  } else {
    objectViewHex = TRUE;
  }

  XCUpdateList();
}

void xcardMain::XCSetViewer() {
  getViewerDialog->show();
}

void xcardMain::XCAboutXCardII() {

  QMessageBox::critical( 0, "XCard II",
			 "A smartcard formatting and viewing\nutility designed for the musclecard and pcsc-lite.\n\nWritten by David Corcoran\nSee file LICENSE");
}

void xcardMain::XCInitCard() {

  //MSCLong32 rv;
  //char myErrorString[200];
  //MSCInitTokenParams initParams;
  xcardPersonalize *persoCard;

  persoCard = new xcardPersonalize(clickedReader, this, this, 0);

  persoCard->show();

}

void xcardMain::XCReGenerateKey() {
  putKeyDialog->XCInitializeKeyDialog(TRUE, TRUE, -1);
  putKeyDialog->show();
}

void xcardMain::XCRefreshContents() {
  XCUpdateList();
}

void xcardMain::XCRefreshReaders() {

  unsigned int i;
  int p;
  MSCLong32 rv;

  do {
    arrayLength = 0;
    rv = MSCListTokens(MSC_LIST_ALL, 0, &arrayLength);
    if (tokenArray) { free(tokenArray); }
    tokenArray = (MSCLPTokenInfo)malloc(sizeof(MSCTokenInfo)*arrayLength);
    rv = MSCListTokens(MSC_LIST_ALL, tokenArray, &arrayLength);
    
    if ( rv != MSC_SUCCESS ) {
      sprintf(errorMessage, "%s%s", "MSCListTokens failed: ",
	      msc_error(rv));
      g_statusLabel1->setText(errorMessage);
      break;
    }
    
    g_browseReaders1->clear();
    
    readerRoot =  new QListViewItem(g_browseReaders1, "Tokens");
    
    QToolTip::add(g_browseReaders1, 
		  "Double click an available\ntoken to begin");
    
    p = 0;
    
    for ( i=0; i < arrayLength; i++ ) {
      readerList[p] = new QListViewItem(readerRoot, 
					tokenArray[i].tokenName);
      tokenArray[i].addParams = (void *)readerList[p];
      readerRoot->insertItem(readerList[p]);
      g_browseReaders1->ensureItemVisible(readerList[p]);
      
      /* Now set up the event handler */
      
      /*
	cardEventInf[p] = 
	(LPCardEventInfo)malloc(sizeof(CardEventInfo));
	
	SCardEstablishEventCallback(hContext, &readerChars[i],
	cardEventInf[p], 
	(LPCardEventCallback)xcardMainReaderEvent,
	this);
      */
      
      ++p;
    }
    
    mainBox = g_browseBox1;
    
    g_browseReaders1->insertItem(readerRoot); 
    g_statusLabel1->setText("Select a token ...");
    
  } while (0);
  
}

/*
MSCString pcsc_stringify_error ( MSCLong32 Error ) {

  static char strError[75];

  switch ( Error ) {
  case SCARD_S_SUCCESS:
    strcpy(strError, "Command successful.");
    break;
  case SCARD_E_CANCELLED:
    strcpy(strError, "Command cancelled.");
    break;
  case SCARD_E_CANT_DISPOSE:
   strcpy(strError, "Cannot dispose handle.");
    break;
  case SCARD_E_INSUFFICIENT_BUFFER:
   strcpy(strError, "Insufficient buffer.");
    break;
  case SCARD_E_INVALID_ATR:
   strcpy(strError, "Invalid ATR.");
    break;
  case SCARD_E_INVALID_HANDLE:
   strcpy(strError, "Invalid handle.");
    break;
  case SCARD_E_INVALID_PARAMETER:
   strcpy(strError, "Invalid parameter given.");
    break;
  case SCARD_E_INVALID_TARGET:
   strcpy(strError, "Invalid target given.");
    break;
  case SCARD_E_INVALID_VALUE:
   strcpy(strError, "Invalid value given.");
    break;
  case SCARD_E_NO_MEMORY:
   strcpy(strError, "Not enough memory.");
    break;
  case SCARD_F_COMM_ERROR:
   strcpy(strError, "RPC transport error.");
    break;
  case SCARD_F_INTERNAL_ERROR:
   strcpy(strError, "Unknown internal error.");
    break;
  case SCARD_F_UNKNOWN_ERROR:
   strcpy(strError, "Unknown internal error.");
    break;
  case SCARD_F_WAITED_TOO_LONG:
   strcpy(strError, "Waited too long.");
    break;
  case SCARD_E_UNKNOWN_READER:
   strcpy(strError, "Unknown reader specified.");
    break;
  case SCARD_E_TIMEOUT:
   strcpy(strError, "Command timeout.");
    break;
  case SCARD_E_SHARING_VIOLATION:
   strcpy(strError, "Sharing violation.");
    break;
  case SCARD_E_NO_SMARTCARD:
   strcpy(strError, "No smartcard inserted.");
    break;
  case SCARD_E_UNKNOWN_CARD:
   strcpy(strError, "Unknown card.");
    break;
  case SCARD_E_PROTO_MISMATCH:
   strcpy(strError, "Card protocol mismatch.");
    break;
  case SCARD_E_NOT_READY:
   strcpy(strError, "Subsystem not ready.");
    break;
  case SCARD_E_SYSTEM_CANCELLED:
   strcpy(strError, "System cancelled.");
    break;
  case SCARD_E_NOT_TRANSACTED:
   strcpy(strError, "Transaction failed.");
    break;
  case SCARD_E_READER_UNAVAILABLE:
   strcpy(strError, "Reader/s is unavailable.");
    break;
  case SCARD_W_UNSUPPORTED_CARD:
   strcpy(strError, "Card is not supported.");
    break;
  case SCARD_W_UNRESPONSIVE_CARD:
   strcpy(strError, "Card is unresponsive.");
    break;
  case SCARD_W_UNPOWERED_CARD:
   strcpy(strError, "Card is unpowered.");
    break;
  case SCARD_W_RESET_CARD:
   strcpy(strError, "Card was reset.");
    break;
  case SCARD_W_REMOVED_CARD:
   strcpy(strError, "Card was removed.");
    break;
  case SCARD_E_UNSUPPORTED_FEATURE:
    strcpy(strError, "Feature not supported.");
    break;
  case SCARD_E_PCI_TOO_SMALL:
    strcpy(strError, "PCI struct too small.");
    break;
  case SCARD_E_READER_UNSUPPORTED:
   strcpy(strError, "Reader is unsupported.");
    break;
  case SCARD_E_DUPLICATE_READER:
   strcpy(strError, "Reader already exists.");
    break;
  case SCARD_E_CARD_UNSUPPORTED:
   strcpy(strError, "Card is unsupported.");
    break;
  case SCARD_E_NO_SERVICE:
   strcpy(strError, "Service not available.");
    break;
  case SCARD_E_SERVICE_STOPPED:
    strcpy(strError, "Service was stopped.");
    break;

  default:
    sprintf(strError, "Some strange error %d.", Error);
    break;

  };

  return strError;
}
*/
