/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h>
#include <unistd.h> /* for getpid() */
#include <errno.h>

#ifdef HAVE_SYSLOG_H
# include <syslog.h>
# ifndef LOG_DAEMON
#  define LOG_DAEMON LOG_USER
# endif
# ifndef SYSLOG_FAC
#  define SYSLOG_FAC LOG_DAEMON
# endif
#endif

#include "string_printf.h"
#include "defs.h"
#include "rvlog.h"

rvlog* rvlog_loghandle = NULL;


rvlog::rvlog() {
  open();
}

rvlog::rvlog(int how, const string& name) {
  open(how, name);
}

rvlog::rvlog(int how, int opts, const string& name) {
  open(how, opts, name);
}

rvlog::~rvlog() {
  close();
}

int
rvlog::open(const string& name) {
  int how = isatty(0) ? RVLOG_STDERR : RVLOG_SYSLOG;
  return open(how, name);
}

int
rvlog::open(int how, const string& name) {
  return open(how, 0, name);
}

int
rvlog::open(int how, int opts, const string& name) {
  if (how_set(how) == -1)
    return -1;

  name_set(name);

  if (_how & RVLOG_SYSLOG) {
#ifdef HAVE_SYSLOG_H
    openlog(_name.c_str(), LOG_PID, SYSLOG_FAC);
#else
    return -1;
#endif
  }

  _level = RVLOG_MAX;
  _opts = opts;

  return 0;
}

int
rvlog::close() {
#ifdef HAVE_SYSLOG_H
  if (_how & RVLOG_SYSLOG)
    closelog();
#endif
  return 0;
}


int
rvlog::how_set(int how) {
  if (how < -1)
    return -1;
  if (how == -1) {
    _how = 0;
    if (getenv("RVLOG_SYSLOG"))
      _how |= RVLOG_SYSLOG;
    if (getenv("RVLOG_STDERR"))
      _how |= RVLOG_STDERR;
  }
  else 
    _how = how;
  return 0;
}

int
rvlog::how_get() const {
  return _how;
}

int
rvlog::opts_set(int opts) {
  if (opts < 0)
    return -1;
  _opts = opts;
  return 0;
}

int
rvlog::opts_get() const {
  return _opts;
}

int
rvlog::level_set(int level) {
  if (level < -1)
    return -1;
  if (level == -1) {
    char* tmp;
    if ((tmp = getenv("RVLOG_DEBUG"))) {
      if (isdigit((int)*tmp))
	_level = RVLOG_DEBUG(atoi(tmp));
      else
	_level = RVLOG_DEBUG(25);
    }
    else
      //  _level = RVLOG_DEBUG(25); /* by default, log everything */
      _level = -1;
  }
  else
    _level = level;
  return 0;
}  

int
rvlog::level_get() const {
  return _level;
}

int
rvlog::name_set(const string& name) { /* keep only basename */
  _name.assign(name, name.rfind('/') + 1, string::npos);
  return 0;
}

string
rvlog::name_get() const {
  return _name;
}

int
rvlog::logv(int level, const char* format, va_list pvar) const {
  int saved_errno = errno;

  if (level == 0 || format == NULL)
    return -1;

  if (_how == RVLOG_NONE || level > _level)
    return 0; /* Don't log if the maximum log level is too low */

  /* Copy format (char*) to format_percentm (string). */
  string format_percentm = format;

  /* Replace %m in format_percentm string. */
  unsigned int pos = format_percentm.find("%m");
  if (pos != string::npos)
    format_percentm.replace(pos, 2 /* lenght of "%m" */,
			    strerror(saved_errno));

  /* Expand % directives a la printf. */
  string str = string_vprintf(format_percentm.c_str(), pvar);

  /* Log to stderr */
  if (_how & RVLOG_STDERR) {
    /* Add the application name if required. */
    if (_opts & RVLOG_NAME)
      cerr << _name;

    /* Add the pid and maybe the thread Id if required. */
    if (_opts & RVLOG_PID)
      cerr << '[' << getpid() << ']';

    if (_opts & (RVLOG_NAME|RVLOG_PID))
      cerr << ": ";
    cerr << str << endl;
  }

#ifdef HAVE_SYSLOG_H
  /* Log to syslog */
  if (_how & RVLOG_SYSLOG) {
    int slfac;
    if (level == RVLOG_ERROR)
      slfac = LOG_ERR;
    else if (level == RVLOG_WARNING)
      slfac = LOG_WARNING;
    else if (level == RVLOG_INFO)
      slfac = LOG_NOTICE;
    else if (level >= RVLOG_DEBUG(0))
      slfac = LOG_DEBUG;
    else
      slfac = LOG_DEBUG;

    syslog(slfac, str.c_str());
  }
#endif

  errno = saved_errno;
  return 0;
}

int
rvlog::log(int level, const char* format, va_alist) va_dcl const {
  va_list pvar;
  VA_START(pvar, format);
  int ret = logv(level, format, pvar);
  va_end(pvar);
  return ret;
}

int
rvlog_log(int level, const char* format, va_alist) va_dcl {
  if (rvlog_loghandle == NULL) {
    rvlog_loghandle = new rvlog();
    if (rvlog_loghandle == NULL)
      return -1;
  }
  va_list pvar;
  VA_START(pvar, format);
  int ret = rvlog_loghandle->logv(level, format, pvar);
  va_end(pvar);
  return ret;
}
