/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <sstream>

#include <stdio.h>
#include <stdlib.h>
#include <arpa/inet.h>

#include "wfnetwork.h"
#include "wfindent.h"
#include "defs.h"


wf_network::wf_network() :
  name(),
  comment(),
  network(),
  netmask()
{}

wf_network::wf_network(const string& str) :
  name(),
  comment(),
  netmask()
{
  network_set(str);
}

wf_network::wf_network(const string& str, int bitmask) :
  name(),
  comment()
{
  set(str, bitmask);
}

wf_network::wf_network(const wf_ipaddr& ipaddr, int bitmask) :
  name(),
  comment()
{
  set(ipaddr, bitmask);
}


bool
wf_network::set(const string& str, int bitmask) {
  wf_ipaddr netaddr;

  return netaddr.set(str) && netmask_set(bitmask) &&
    network.set(netaddr.get() & netmask.get());
}

bool
wf_network::set(const wf_ipaddr& ipaddr, int bitmask) {
  return ipaddr.isdefined() && netmask_set(bitmask) &&
    network.set(ipaddr.get() & netmask.get());
}

bool
wf_network::network_set(const char* str) {
  const char *delim = ".";
  char *number, *str2, *ptr;
  int slashmask = 0, i;
  int mask = 0, net = 0;
  
  if (str == NULL || *str == '\0')
    return false;
  
  str2 = strdup(str);
  if ((ptr = strrchr(str2, '/')) != NULL) { /* address is IP/mask */
    *ptr++ = '\0';
    if (strrchr(str2, '*') != NULL) /* '*' and '/' are incompatible */
      goto network_set_destr;
    if (netmask_set(ptr) == false)
      goto network_set_destr;
    ++slashmask;
  }

  ptr = str2;

  /* warning: we should consider '*' only at the end of the address RV@@4 */
  for (i = 0; i < 4; i++) {
    if ((number = strsep(&ptr, delim)) == NULL)
      goto network_set_destr;
    
    mask <<= 8;
    net <<= 8;
    if (!strcmp(number, "*"))
      mask |= 0xFF;
    else {
      int num;
      char *p;
      num = strtol(number, &p, 10);
      if (*p != '\0' || num < 0 || num > 255)
	goto network_set_destr;
      net |= num;
    }
  }

  if (ptr != NULL)
    goto network_set_destr;
  free(str2);
  
  if (slashmask)
    return network.set(ntohl(net) & netmask.get());

  if (network.set(ntohl(net)) == false)
    return false;
  if (mask)  /* if netmask was set */
    if (netmask.set(ntohl(~mask)) == false)
      return false;

  return true;

 network_set_destr:
  free(str2);
  return false;
}

#if 0 // not finished RV@@3
bool
wf_network::network_set(const string& str) {
  const char *delim = ".";
  char *number, *ptr;
  string str2;
  int slashmask = 0, i;
  int mask = 0, net = 0;
  unsigned int loc;
  
  if (str.empty())
    return false;
  
  loc = str.rfind(str, '/');
  if (loc != string::npos) {  /* address is IP/mask */
    if (str.rfind(str, '*') != string::npos)
      return false;  /* '*' and '/' are incompatible */
    
  }
  
  // not finished
  {
    *ptr++ = '\0';
    if (strrchr(str2, '*') != NULL) // '*' and '/' are incompatible
      goto network_set_destr;
    if (netmask_set(ptr) == false)
      goto network_set_destr;
    ++slashmask;
  }
  
  ptr = str2;
  
  /* warning: we should consider '*' only at the end of the address RV@@3 */
  for (i = 0; i < 4; i++) {
    if ((number = strsep(&ptr, delim)) == NULL)
      goto network_set_destr;
    
    mask <<= 8;
    net <<= 8;
    if (!strcmp(number, "*"))
      mask |= 0xFF;
    else {
      int num;
      char *p;
      num = strtol(number, &p, 10);
      if (*p != '\0' || num < 0 || num > 255)
	goto network_set_destr;
      net |= num;
    }
  }
  
  if (ptr != NULL)
    goto network_set_destr;
  free(str2);
  
  if (slashmask)
    return network.set(ntohl(net) & netmask.get());
  
  if (network.set(ntohl(net)) == false)
    return false;
  if (mask)  /* if netmask was set */
    if (netmask.set(ntohl(~mask)) == false)
      return false;

  return true;

 network_set_destr:
  free(str2);
  return false;
}
#else
bool
wf_network::network_set(const string& str) {
  return network_set(str.c_str());
}
#endif

bool
wf_network::netmask_set(const char* str) {
  if (str == NULL || *str == '\0')
    return false;

  if (strchr(str, '.') == NULL) { /* bitmask ? */
    char *p;
    int num;

    num = strtol(str, &p, 10);
    if (*p != '\0' || netmask_set(num) == false)
      return false;
  }
  else { /* netmask ? */
    if (netmask.set(str) == false)
      return false;
  }
  
  if (netmask_check() == false)
    return false;

  return true;
}

bool
wf_network::netmask_set(const string& str) {
  return netmask_set(str.c_str());
}

bool
wf_network::netmask_set(int bitmask) {
  int mask;

  return wf_bitmask_tonetmask(bitmask, mask) &&
    netmask.set(mask);
}

bool
wf_network::setnull() {
  return (network.set(0) && netmask.set(0));
}

int
wf_network::netmask_tobitmask() const {
  if (netmask.isdefined() == false)
    return -1;
  return wf_netmask_tobitmask(netmask.get());
}

bool
wf_network::netmask_check() const {
  return netmask.isdefined() && wf_netmask_tobitmask(netmask.get()) != -1;
}

bool
wf_network::isnull() const {
  return network.isdefined() && netmask.isdefined() &&
    network.get() == 0 && netmask.get() == 0;
}

bool
wf_network::ishost() const {
  return network.isdefined() && netmask.isdefined() &&
    netmask.get() == 0xFFFFFFFF && network.get() != 0;
}

bool
wf_network::belong(const wf_ipaddr& ipaddr) const {
  return network.isdefined() && netmask.isdefined() && ipaddr.isdefined() &&
    ((ipaddr.get() & netmask.get()) == (network.get() & netmask.get()));
}

wf_ipaddr
wf_network::broadcast() const {
  return wf_ipaddr(network.get() | ~netmask.get());
}

bool
wf_network::isdefined() const {
  /* name and comment are optional */
  return network.isdefined() && netmask.isdefined();
}

string
wf_network::tostr() const {
  if (name.empty() == false)
    return '$' + name;
  return tostr_value();
}

string
wf_network::tostr_value() const {
  if (isnull() == true)
    return "0/0";

  int bitmask = netmask_tobitmask();
  /*
    if (bitmask == -1)
    error; // RV@@3
  */
  ostringstream os;
  os << network << '/' << bitmask;
  return os.str();
}

ostream&
wf_network::print(ostream& os) const {
  return os << tostr();
}

ostream&
wf_network::print_value(ostream& os) const {
  return os << tostr_value();
}

ostream&
wf_network::output_xml(ostream& os, unsigned int indent_level) const {
  os << wf_indent(indent_level) << "<network>\n";
  if (name.empty() == false)
    os << wf_indent(indent_level + 1) << "<name>" << name << "</name>\n";
  if (network.isdefined())
    os << wf_indent(indent_level + 1) << "<network>" << network <<
      "</network>\n";
  if (netmask.isdefined())
    os << wf_indent(indent_level + 1) << "<netmask>" << netmask <<
      "</netmask>\n";
  if (comment.empty() == false)
    os << wf_indent(indent_level + 1) << "<comment>" << comment << "</comment>\n";
  os << wf_indent(indent_level) << "</network>" << endl;
  return os;
}

ostream&
wf_network::debugprint(ostream& os) const {
  os << _("Network:") << endl;
  os << _("name:\t\t") << name << endl;
  os << _("network:\t") << network << endl;
  os << _("netmask:\t") << netmask;
  if (netmask.isdefined())
    os << " (/" << wf_netmask_tobitmask(netmask.get()) << ')';
  os << endl;
  os << _("comment:\t") << comment << endl;
  return os;
}

ostream&
operator<<(ostream& os, const wf_network& network) {
  return network.print(os);
}

bool
operator==(const wf_network& network1, const wf_network& network2) {
  /* we'll ignore 'name' and 'comment' members during comparison */
  return (network1.network == network2.network &&
	  network1.netmask == network2.netmask);
}

int
wf_netmask_tobitmask(int netmask) {
  int i;
  int bitmask = 0;
  int bit = 1;
  int mask = htonl(netmask);

  for (i = 32; i > 0; i--, mask >>= 1) {
    if (bit) {
      if ((mask & 1)) {
	bitmask = i;
	bit = 0;
      }
    }
    else
      if (!(mask & 1))
	return -1;
  }

  return bitmask;
}

bool
wf_bitmask_check(int bitmask) {
  return (bitmask >= 0 && bitmask <= 32);
}

bool
wf_bitmask_tonetmask(int bitmask, int& netmask) {
  int mask;

  if (wf_bitmask_check(bitmask) == false)
    return false;

  mask = htonl(0xFFFFFFFF) << (32 - bitmask);
  netmask = ntohl(mask);

  return true;
}

/* Warning: a mask cannot reliably be infered from a net */
/* This function is wrong! ALL@@3 */
bool
wf_network_tonetmask(int network, int* netmask) {
  int net;
  int mask = 1;
  
  net = htonl(network);
  while (!(net % 2)) {
    net >>= 1;
    mask <<= 1;
  }

  mask >>= 1;
  mask = -mask;

  *netmask = ntohl(mask);
  return true;
}
