/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdio.h> /* for sprintf */
#if defined(__FreeBSD__) || defined(__NetBSD__) || defined(__OpenBSD__)
//#include <net/if_ether.h> /* for ether_aton */
#endif
#ifdef linux
#include <netinet/ether.h> /* for ether_aton */
#endif

#include "wfmacaddr.h"
#include "defs.h"


wf_macaddr::wf_macaddr() :
  _defined(false)
{
}

wf_macaddr::wf_macaddr(const string& str) {
  set(str);
}

bool
wf_macaddr::set(const string& str) {
#if 0 /* ALL@@1 in case  ether_aton wouldn't exist */
  if (strlen(str) != ETHER_ADDR_LEN * 3 - 1)
    return false;
  
  unsigned int i = 0;
  for (i = 0; i < ETHER_ADDR_LEN; i++) {
    long number;
    char *end;
    
    number = strtol(str + i * 3, &end, 16);
    if (end != str + i * 3 + 2 || number < 0 || number > 255)
      return false;

    info->srcaddr[i] = number;
  }
#endif

  struct ether_addr* mac;
#if defined(__OpenBSD__) /* missing const in ether_aton(char*) prototype */
  mac = ether_aton((char*)str.c_str());
#else
  mac = ether_aton(str.c_str());
#endif
  if (mac == NULL)
    return false;
#if defined(__FreeBSD__)
  memcpy(&_addr, &(mac->octet), ETHER_ADDR_LEN);
#else
  memcpy(&_addr, &(mac->ether_addr_octet), ETHER_ADDR_LEN);
#endif
  _defined = true;
  return true;
}

bool
wf_macaddr::set(const uint8_t* address) {
  if (address == NULL)
    return false;
  memcpy(&_addr, address, ETHER_ADDR_LEN);
  _defined = true;
  return true;
}

bool
wf_macaddr::set(const sockaddr* sockaddr) {
  if (sockaddr == NULL)
    return false;
  memcpy(&_addr, &(sockaddr->sa_data), ETHER_ADDR_LEN);
  _defined = true;
  return true;
}

const uint8_t*
wf_macaddr::get() const {
  return _addr;
}

bool
wf_macaddr::isdefined() const {
  return _defined;
}

bool
wf_macaddr::isnull() const {
  int i;
  for (i = 0; i < ETHER_ADDR_LEN; i++)
    if (_addr[i] != 0)
      return false;
  return true;
}

bool
wf_macaddr::isbroadcast() const {
  int i;
  for (i = 0; i < ETHER_ADDR_LEN; i++)
    if (_addr[i] != 0xFF)
      return false;
  return true;
}

string
wf_macaddr::tostr(unsigned char flags) const {
  string str;

  if (_defined) {
    char buf[20];
    char* format;

    if (flags & WF_MACADDR_UPPERCASE) {
      if (flags & WF_MACADDR_ONEDIGIT)
	format = "%X:%X:%X:%X:%X:%X";
      else
	format = "%02X:%02X:%02X:%02X:%02X:%02X";
    }
    else {
      if (flags & WF_MACADDR_ONEDIGIT)
	format = "%x:%x:%x:%x:%x:%x";
      else
	format = "%02x:%02x:%02x:%02x:%02x:%02x";
    }

    sprintf(buf, format,
	    _addr[0], _addr[1], _addr[2], _addr[3], _addr[4], _addr[5]);
    str = buf;
  }
  return str;
}

#ifdef NO_MACVENDOR_DB

string
wf_macaddr::vendor() const {
  return ""; /* always return an empty string */
}

#else /* NO_MACVENDOR_DB undefined */

struct wf_mac_vendor {
   uint8_t pref1, pref2, pref3;
  char* name;
};

#include "macvendor.cc"

string
wf_macaddr::vendor() const {
  if (_defined == false)
    return "";

  int size = sizeof(mac_vendor) / sizeof(struct wf_mac_vendor);
  int i;

  for (i = 0; i < size; i++)
    if (mac_vendor[i].pref1 == _addr[0] &&
	mac_vendor[i].pref2 == _addr[1] &&
	mac_vendor[i].pref3 == _addr[2])
      break;
  return (i != size) ? mac_vendor[i].name : "";
}

#endif

ostream&
wf_macaddr::print(ostream& os) const {
  string str = tostr();
  if (str.empty())
    os << _("(undefined)");
  else
    os << str;
  return os;
}

ostream&
wf_macaddr::debugprint(ostream& os) const {
  os << _("MAC address:\t") << *this;
  if (_defined == false)
    return os << endl;

  os << " (";
  string str = vendor();
  if (str.empty() == false)
    os << str;
  else
    os << "unknown";
  return os << ')' << endl;
}

ostream&
operator<<(ostream& os, const wf_macaddr& macaddr) {
  return macaddr.print(os);
}

bool
operator==(const wf_macaddr& macaddr1, const wf_macaddr& macaddr2) {
  if (macaddr1._defined != macaddr2._defined || macaddr1._defined == false) {
    return false;
    throw "error"; // RV@@6
  }
  return wf_macaddr::compare(macaddr1._addr, macaddr2._addr) == 0;
}

bool
operator!=(const wf_macaddr& macaddr1, const wf_macaddr& macaddr2) {
  if (macaddr1._defined != macaddr2._defined || macaddr1._defined == false) {
    return true;
    throw "error"; // RV@@6
  }
  return wf_macaddr::compare(macaddr1._addr, macaddr2._addr) != 0;
}

bool
operator<(const wf_macaddr& macaddr1, const wf_macaddr& macaddr2) {
  if (macaddr1._defined == false || macaddr2._defined == false) {
    return false;
    throw "error"; // RV@@6
  }
  return wf_macaddr::compare(macaddr1._addr, macaddr2._addr) < 0;
}

bool
operator>(const wf_macaddr& macaddr1, const wf_macaddr& macaddr2) {
  if (macaddr1._defined == false || macaddr2._defined == false) {
    return false;
    throw "error"; // RV@@6
  }
  return wf_macaddr::compare(macaddr1._addr, macaddr2._addr) > 0;
}

bool
wf_macaddr::check(const string& str) {
#if defined(__OpenBSD__) /* missing const in ether_aton(char*) prototype */
  return ether_aton((char*)str.c_str()) != NULL;
#else
  return ether_aton(str.c_str()) != NULL;
#endif
}

int
wf_macaddr::compare(const uint8_t* s1, const uint8_t* s2) {
  int n = ETHER_ADDR_LEN;
  while (--n >= 0 && *s1++ == *s2++);
  return (n < 0 ? 0 : (*--s1 - *--s2));
}
