/*
 * WallFire -- a comprehensive firewall administration tool.
 * 
 * Copyright (C) 2001 Herv Eychenne <rv@wallfire.org>
 * 
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU General Public License
 * as published by the Free Software Foundation; either version 2
 * of the License, or (at your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 * 
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.
 * 
 */

using namespace std;

#ifdef HAVE_CONFIG_H
#include <config.h>
#endif

#include <stdlib.h>
#include <arpa/inet.h>
#include <sys/stat.h> /* for stat */

#include "wffirewall.h"
#include "defs.h"


wf_firewall::wf_firewall() :
  _fwtool()
{
}

wf_firewall::~wf_firewall() {}

// default wf_firewall(const wf_firewall&)


/* we should define fwtools as macros RV@@2 */

void
wf_firewall::fwtool_set(const string& fwtool) {
  _fwtool = fwtool;
}

const string&
wf_firewall::fwtool_get() const {
  return _fwtool;
}

#ifdef linux
static bool
fwtool_guess_linux(wf_firewall& firewall) {
  if (firewall.os_version_major != 2) /* 2.x kernel only */
    return false;

  if (firewall.os_version_minor == 0) { /* 2.0 kernel */
    firewall.fwtool_set("ipfwadm");
    /* verify that it is operationnal ALL@@2 */
    return true;
  }
  if (firewall.os_version_minor == 2) { /* 2.2 kernel */
    firewall.fwtool_set("ipchains");
    /* verify that it is operationnal ALL@@2 */
    return true;
  }
  if (firewall.os_version_minor > 2) { /* 2.x kernel, where x>2 */
    struct stat sts;
    if (!stat("/proc/net/ip_tables_names", &sts))
      firewall.fwtool_set("iptables");
    return true;
  }
  
  return false;
}
#endif /* linux */

/* return true when there is no problem. That does not mean that a firewalling
 * tool was found.
 * Typically, when no firewalling tool was found but the kernel version is
 * perfectly valid, fwtool is an empty string and true is returned.
 */
bool
wf_firewall::fwtool_guess() {
  fwtool_set("");

  switch (os) {
#ifdef linux
  case WF_OS_LINUX:
    return fwtool_guess_linux(*this);
#endif
  case WF_OS_FREEBSD:
  case WF_OS_NETBSD:
  case WF_OS_OPENBSD:
    {
      struct stat sts;
      if (!stat("/sbin/ipf", &sts)) // check that it is an executable ALL@@2
	fwtool_set("ipfilter");
    }
    return true;
  default:
    return false;
  }
}

/* We consider that no field is already defined. */
bool
wf_firewall::guess(bool skip_useless) {
  return wf_host::guess(skip_useless) && fwtool_guess();
}

ostream&
wf_firewall::debugprint(ostream& outs) const {
  outs << _("Firewall:") << endl;

  outs << _("comment:\t") << comment << endl;

  outs << _("name:\t\t");
  if (name.empty() == false)
    outs << name;
  else
    outs << _("(undefined)");
  outs << endl;

  wf_host::debugprint(outs) << endl;

  outs << _("os:\t\t") << os_tostr() << ' ' << os_version_tostr() << endl;

  if (os == WF_OS_LINUX)
    outs << _("distribution:\t") << linux_distribution_tostr() << ' ' <<
      linux_distribution_version_tostr() << endl;

  outs << _("fwtool:\t\t");
  if (fwtool_get().empty() == false)
    outs << fwtool_get();
  else
    outs << _("(undefined)");
  outs << endl;

  return outs;
}
