/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
*
* This code is partially based on qprintpreviewdialog.cpp from Qt 4.4,
* licensed unsed the GPL license.
**************************************************************************/

#include "previewdialog.h"

#include <QLayout>
#include <QLabel>
#include <QLineEdit>
#include <QComboBox>
#include <QToolButton>
#include <QPushButton>
#include <QDoubleValidator>
#include <QIntValidator>
#include <QPageSetupDialog>
#include <QPrintDialog>
#include <QTextDocument>
#include <QApplication>
#include <QDesktopWidget>

#include "widgets/previewwidget.h"
#include "iconloader.h"

class ZoomFactorValidator : public QDoubleValidator
{
public:
    ZoomFactorValidator( QObject* parent ) : QDoubleValidator( parent )
    {
    }

    ZoomFactorValidator( qreal bottom, qreal top, int decimals, QObject* parent ) :
        QDoubleValidator( bottom, top, decimals, parent )
    {
    }

    State validate( QString& input, int& pos ) const
    {
        bool replacePercent = false;
        if ( input.endsWith( '%' ) ) {
            input = input.left( input.length() - 1 );
            replacePercent = true;
        }
        State state = QDoubleValidator::validate( input, pos );
        if ( replacePercent )
            input += QLatin1Char( '%' );
        if (state == Intermediate) {
            int i = input.indexOf( QLocale::system().decimalPoint() );
            if ( ( i == -1 && input.size() > 3 ) || ( i != -1 && i > 3 ) )
                return Invalid;
        }
        return state;
    }
};

class PreviewLineEdit : public QLineEdit
{
public:
    PreviewLineEdit( QWidget* parent = NULL ) : QLineEdit(parent)
    {
        setContextMenuPolicy( Qt::NoContextMenu );
    }

public:
    void setText( const QString& text )
    {
        QLineEdit::setText( text );
        m_oldText = text;
    }

protected:
    void focusOutEvent( QFocusEvent *e )
    {
        if ( isModified() && !hasAcceptableInput() )
            setText( m_oldText );
        QLineEdit::focusOutEvent( e );
    }

private:
    QString m_oldText;
};

PreviewDialog::PreviewDialog( PreviewPrinter* printer, QTextDocument* document, QWidget* parent ) : QDialog( parent ),
    m_printer( printer ),
    m_document( document )
{
    setWindowTitle( tr( "Print Preview" ) );

    QVBoxLayout* mainLayout = new QVBoxLayout( this );
    mainLayout->setMargin( 0 );
    mainLayout->setSpacing( 0 );

    QHBoxLayout* toolLayout = new QHBoxLayout();
    toolLayout->setMargin( 6 );
    toolLayout->setSpacing( 3 );

    m_firstButton = new QToolButton( this );
    m_firstButton->setToolTip( tr( "First Page" ) );
    m_firstButton->setIcon( IconLoader::icon( "go-start" ) );
    toolLayout->addWidget( m_firstButton );

    m_previousButton = new QToolButton( this );
    m_previousButton->setToolTip( tr( "Previous Page" ) );
    m_previousButton->setIcon( IconLoader::icon( "go-previous" ) );
    toolLayout->addWidget( m_previousButton );

    toolLayout->addSpacing( 10 );

    m_pageEdit = new PreviewLineEdit( this );
    m_pageEdit->setAlignment( Qt::AlignRight );
    m_pageEdit->setSizePolicy( QSizePolicy( QSizePolicy::Fixed, QSizePolicy::MinimumExpanding ) );
    m_pageEdit->setFixedWidth( 40 );
    toolLayout->addWidget( m_pageEdit );

    m_pageLabel = new QLabel( this );
    m_pageLabel->setFixedWidth( 30 );
    toolLayout->addWidget( m_pageLabel );

    toolLayout->addSpacing( 10 );

    m_nextButton = new QToolButton( this );
    m_nextButton->setToolTip( tr( "Next Page" ) );
    m_nextButton->setIcon( IconLoader::icon( "go-next" ) );
    toolLayout->addWidget( m_nextButton );

    m_lastButton = new QToolButton( this );
    m_lastButton->setToolTip( tr( "Last Page" ) );
    m_lastButton->setIcon( IconLoader::icon( "go-end" ) );
    toolLayout->addWidget( m_lastButton );

    toolLayout->addSpacing( 30 );

    m_singleButton = new QToolButton( this );
    m_singleButton->setToolTip( tr( "Single Page" ) );
    m_singleButton->setIcon( IconLoader::icon( "page-single" ) );
    m_singleButton->setCheckable( true );
    toolLayout->addWidget( m_singleButton );

    m_facingButton = new QToolButton( this );
    m_facingButton->setToolTip( tr( "Facing Pages" ) );
    m_facingButton->setIcon( IconLoader::icon( "page-facing" ) );
    m_facingButton->setCheckable( true );
    toolLayout->addWidget( m_facingButton );

    m_allButton = new QToolButton( this );
    m_allButton->setToolTip( tr( "All Pages" ) );
    m_allButton->setIcon( IconLoader::icon( "page-all" ) );
    m_allButton->setCheckable( true );
    toolLayout->addWidget( m_allButton );

    toolLayout->addSpacing( 30 );

    m_zoomInButton = new QToolButton( this );
    m_zoomInButton->setToolTip( tr( "Zoom In" ) );
    m_zoomInButton->setIcon( IconLoader::icon( "view-zoom-in" ) );
    m_zoomInButton->setAutoRepeat( true );
    toolLayout->addWidget( m_zoomInButton );

    m_zoomOutButton = new QToolButton( this );
    m_zoomOutButton->setToolTip( tr( "Zoom Out" ) );
    m_zoomOutButton->setIcon( IconLoader::icon( "view-zoom-out" ) );
    m_zoomOutButton->setAutoRepeat( true );
    toolLayout->addWidget( m_zoomOutButton );

    toolLayout->addSpacing( 10 );

    m_zoomCombo = new QComboBox( this );
    m_zoomCombo->setEditable( true );
    m_zoomCombo->setInsertPolicy( QComboBox::NoInsert );
    m_zoomEdit = new PreviewLineEdit();
    m_zoomEdit->setValidator( new ZoomFactorValidator( 10.0, 800.0, 1, m_zoomEdit ) );
    m_zoomCombo->setLineEdit( m_zoomEdit );
    m_zoomCombo->addItem( "12.5%" );
    m_zoomCombo->addItem( "25%" );
    m_zoomCombo->addItem( "50%" );
    m_zoomCombo->addItem( "100%" );
    m_zoomCombo->addItem( "125%" );
    m_zoomCombo->addItem( "150%" );
    m_zoomCombo->addItem( "200%" );
    m_zoomCombo->addItem( "400%" );
    m_zoomCombo->addItem( "800%" );
    m_zoomCombo->addItem( tr( "Fit Width" ) );
    m_zoomCombo->addItem( tr( "Fit Page" ) );
    m_zoomCombo->setMaxVisibleItems( m_zoomCombo->count() );
    toolLayout->addWidget( m_zoomCombo );

    toolLayout->addSpacing( 10 );

    m_fitWidthButton = new QToolButton( this );
    m_fitWidthButton->setToolTip( tr( "Fit Width" ) );
    m_fitWidthButton->setIcon( IconLoader::icon( "view-fit-width" ) );
    m_fitWidthButton->setCheckable( true );
    toolLayout->addWidget( m_fitWidthButton );

    m_fitPageButton = new QToolButton( this );
    m_fitPageButton->setToolTip( tr( "Fit Page" ) );
    m_fitPageButton->setIcon( IconLoader::icon( "view-fit-page" ) );
    m_fitPageButton->setCheckable( true );
    toolLayout->addWidget( m_fitPageButton );

    toolLayout->addSpacing( 30 );

    m_portraitButton = new QToolButton( this );
    m_portraitButton->setToolTip( tr( "Portrait Orientation" ) );
    m_portraitButton->setIcon( IconLoader::icon( "page-portrait" ) );
    m_portraitButton->setCheckable( true );
    toolLayout->addWidget( m_portraitButton );

    m_landscapeButton = new QToolButton( this );
    m_landscapeButton->setToolTip( tr( "Landscape Orientation" ) );
    m_landscapeButton->setIcon( IconLoader::icon( "page-landscape" ) );
    m_landscapeButton->setCheckable( true );
    toolLayout->addWidget( m_landscapeButton );

    toolLayout->addStretch( 1 );
    mainLayout->addLayout( toolLayout, 0 );

    m_preview = new PreviewWidget( m_printer, this );
    mainLayout->addWidget( m_preview, 1 );

    connect( m_preview, SIGNAL( paintRequested( QPrinter* ) ), this, SLOT( paintRequested( QPrinter* ) ) );
    connect( m_preview, SIGNAL( previewChanged() ), this, SLOT( previewChanged() ) );

    connect( m_firstButton, SIGNAL( clicked() ), this, SLOT( pageFirst() ) );
    connect( m_previousButton, SIGNAL( clicked() ), this, SLOT( pagePrevious() ) );

    connect( m_pageEdit, SIGNAL( editingFinished() ), SLOT( pageEdited() ) );

    connect( m_nextButton, SIGNAL( clicked() ), this, SLOT( pageNext() ) );
    connect( m_lastButton, SIGNAL( clicked() ), this, SLOT( pageLast() ) );

    connect( m_singleButton, SIGNAL( clicked() ), m_preview, SLOT( setSinglePageViewMode() ) );
    connect( m_facingButton, SIGNAL( clicked() ), m_preview, SLOT( setFacingPagesViewMode() ) );
    connect( m_allButton, SIGNAL( clicked() ), m_preview, SLOT( setAllPagesViewMode() ) );

    connect( m_zoomInButton, SIGNAL( clicked() ), this, SLOT( zoomIn() ) );
    connect( m_zoomOutButton, SIGNAL( clicked() ), this, SLOT( zoomOut() ) );

    connect( m_zoomEdit, SIGNAL( editingFinished() ), this, SLOT( zoomEdited() ) );
    connect( m_zoomCombo, SIGNAL( activated(int) ), this, SLOT( zoomEdited() ) );

    connect( m_fitWidthButton, SIGNAL( clicked() ), this, SLOT( fitWidth() ) );
    connect( m_fitPageButton, SIGNAL( clicked() ), this, SLOT( fitPage() ) );

    connect( m_portraitButton, SIGNAL( clicked() ), m_preview, SLOT( setPortraitOrientation() ) );
    connect( m_landscapeButton, SIGNAL( clicked() ), m_preview, SLOT( setLandscapeOrientation() ) );

    QHBoxLayout* buttonLayout = new QHBoxLayout();
    buttonLayout->setMargin( 9 );
    buttonLayout->setSpacing( 6 );

    QPushButton* setupButton = new QPushButton( tr( "Page &Setup..." ), this );
    buttonLayout->addWidget( setupButton );

    buttonLayout->addStretch( 1 );

    QPushButton* printButton = new QPushButton( tr( "&Print" ), this );
    buttonLayout->addWidget( printButton );

    QPushButton* closeButton = new QPushButton( tr( "&Close" ), this );
    buttonLayout->addWidget( closeButton );

    mainLayout->addLayout( buttonLayout, 0 );

    connect( setupButton, SIGNAL( clicked() ), this, SLOT( pageSetup() ) );
    connect( printButton, SIGNAL( clicked() ), this, SLOT( print() ) );
    connect( closeButton, SIGNAL( clicked() ), this, SLOT( close() ) );

    resize( QApplication::desktop()->screenGeometry( parent ).size() * 4 / 5 );
}

PreviewDialog::~PreviewDialog()
{
}

void PreviewDialog::paintRequested( QPrinter* printer )
{
    m_document->print( printer );
}

void PreviewDialog::previewChanged()
{
    int page = m_preview->currentPage();
    int numPages = m_preview->numPages();

    m_firstButton->setEnabled( page > 1 );
    m_previousButton->setEnabled( page > 1 );
    m_nextButton->setEnabled( page < numPages );
    m_lastButton->setEnabled( page < numPages );

    m_pageEdit->setText( QString::number( page ) );
    m_pageLabel->setText( QString( "/ %1" ).arg( numPages ) );
    m_pageEdit->setValidator( new QIntValidator( 1, numPages, m_pageEdit ) );

    m_singleButton->setChecked( m_preview->viewMode() == PreviewWidget::SinglePageView );
    m_facingButton->setChecked( m_preview->viewMode() == PreviewWidget::FacingPagesView );
    m_allButton->setChecked( m_preview->viewMode() == PreviewWidget::AllPagesView );

    m_zoomInButton->setEnabled( m_preview->zoomFactor() < 8.0 );
    m_zoomOutButton->setEnabled( m_preview->zoomFactor() > 0.1 );

    m_zoomEdit->setText( QString::number( m_preview->zoomFactor() * 100, 'g', 3 ) + '%' );

    m_fitWidthButton->setChecked( m_preview->zoomMode() == PreviewWidget::FitToWidth );
    m_fitPageButton->setChecked( m_preview->zoomMode() == PreviewWidget::FitInView );

    m_portraitButton->setChecked( m_preview->orientation() == QPrinter::Portrait );
    m_landscapeButton->setChecked( m_preview->orientation() == QPrinter::Landscape );
}

void PreviewDialog::pageEdited()
{
    bool ok = false;
    int page = m_pageEdit->text().toInt( &ok );
    if ( ok ) {
        m_preview->setCurrentPage( page );
        previewChanged();
    }
}

void PreviewDialog::zoomEdited()
{
    QString text = m_zoomEdit->text();

    if ( text == tr( "Fit Width" ) ) {
        m_preview->fitToWidth();
        previewChanged();
    } else if ( text == tr( "Fit Page" ) ) {
        m_preview->fitInView();
        previewChanged();
    } else {
        bool ok = false;
        qreal zoom = text.remove( '%' ).toFloat( &ok );
        if ( ok ) {
            m_preview->setZoomFactor( zoom / 100.0 );
            previewChanged();
        }
    }
}

void PreviewDialog::pageFirst()
{
    m_preview->setCurrentPage( 1 );
    previewChanged();
}

void PreviewDialog::pagePrevious()
{
    m_preview->setCurrentPage( m_preview->currentPage() - 1 );
    previewChanged();
}

void PreviewDialog::pageNext()
{
    m_preview->setCurrentPage( m_preview->currentPage() + 1 );
    previewChanged();
}

void PreviewDialog::pageLast()
{
    m_preview->setCurrentPage( m_preview->numPages() );
    previewChanged();
}

void PreviewDialog::zoomIn()
{
    qreal zoom = qMax( 0.1, m_preview->zoomFactor() * 1.25 );
    m_preview->setZoomFactor( zoom );
    previewChanged();
}

void PreviewDialog::zoomOut()
{
    qreal zoom = qMax( 0.1, m_preview->zoomFactor() * 0.8 );
    m_preview->setZoomFactor( zoom );
    previewChanged();
}

void PreviewDialog::fitWidth()
{
    if ( m_preview->zoomMode() != PreviewWidget::FitToWidth )
        m_preview->fitToWidth();
    else
        m_preview->setZoomMode( PreviewWidget::CustomZoom );
    previewChanged();
}

void PreviewDialog::fitPage()
{
    if ( m_preview->zoomMode() != PreviewWidget::FitInView )
        m_preview->fitInView();
    else
        m_preview->setZoomMode( PreviewWidget::CustomZoom );
    previewChanged();
}

void PreviewDialog::pageSetup()
{
    QPageSetupDialog dialog( m_printer, this );

    if ( dialog.exec() == QDialog::Accepted )
        m_preview->updatePreview();
}

void PreviewDialog::print()
{
    QPrintDialog dialog( m_printer, this );

    if ( dialog.exec() == QDialog::Accepted ) {
        m_preview->print();
        accept();
    }
}
