/**************************************************************************
* This file is part of the WebIssues program
* Copyright (C) 2006 Michał Męciński
* Copyright (C) 2007-2009 WebIssues Team
*
* This program is free software; you can redistribute it and/or modify
* it under the terms of the GNU General Public License as published by
* the Free Software Foundation; either version 2 of the License, or
* (at your option) any later version.
**************************************************************************/

#include "folderview.h"

#include <QTreeView>
#include <QHeaderView>
#include <QLayout>
#include <QLabel>
#include <QAction>
#include <QMenu>
#include <QMessageBox>
#include <QKeyEvent>

#include "models/tablemodels.h"
#include "models/rowfilters.h"
#include "models/columnconditionsettings.h"
#include "models/treeviewsettings.h"
#include "models/treeviewhelper.h"
#include "models/tablemodelshelper.h"
#include "data/datamanager.h"
#include "data/updateevent.h"
#include "data/updatebatch.h"
#include "data/connectioninfo.h"
#include "widgets/searcheditbox.h"
#include "widgets/separatorcombobox.h"
#include "dialogs/issuedialogs.h"
#include "dialogs/filterdialog.h"
#include "dialogs/columnsettingsdialog.h"
#include "dialogs/managefiltersdialog.h"
#include "dialogs/savefilterdialog.h"
#include "dialogs/reportdialog.h"
#include "dialogs/watchesdialog.h"
#include "dialogs/notifydialog.h"
#include "xmlui/builder.h"
#include "connectionmanager.h"
#include "viewmanager.h"
#include "iconloader.h"

FolderView::FolderView( QObject* parent, QWidget* parentWidget ) : View( parent ),
    m_model( NULL ),
    m_filter( NULL ),
    m_filterType( NoFilter ),
    m_searchColumn( Column_Name ),
    m_gotoIssueId( 0 ),
    m_gotoItemId( 0 ),
    m_activateReason( 0 )
{
    QAction* action;

    action = new QAction( IconLoader::icon( "file-reload" ), tr( "&Update Folder" ), this );
    action->setShortcut( QKeySequence::Refresh );
    connect( action, SIGNAL( triggered() ), this, SLOT( updateFolder() ) );
    setAction( "updateFolder", action );

    action = new QAction( IconLoader::icon( "issue-open" ), tr( "&Open Issue" ), this );
    action->setShortcut( QKeySequence::Open );
    connect( action, SIGNAL( triggered() ), this, SLOT( openIssue() ) );
    setAction( "openIssue", action );

    action = new QAction( IconLoader::icon( "issue-new" ), tr( "&Add Issue..." ), this );
    action->setShortcut( QKeySequence::New );
    connect( action, SIGNAL( triggered() ), this, SLOT( addIssue() ) );
    setAction( "addIssue", action );

    action = new QAction( IconLoader::icon( "folder-watch" ), tr( "Folder &Watches..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( editWatches() ) );
    setAction( "editWatches", action );

    action = new QAction( IconLoader::overlayedIcon( "issue", "overlay-yellow" ), tr( "Ma&rk Issue as New" ), this );
    action->setCheckable( true );
    connect( action, SIGNAL( triggered() ), this, SLOT( toggleMark() ) );
    setAction( "toggleMark", action );

    action = new QAction( IconLoader::overlayedIcon( "folder", "overlay-yellow" ), tr( "Mark All Issues as N&ew" ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( markAllNew() ) );
    setAction( "markAllNew", action );

    action = new QAction( IconLoader::overlayedIcon( "folder", "overlay-gray" ), tr( "Mark &All Issues as Read" ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( markAllRead() ) );
    setAction( "markAllRead", action );

    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) ) {
        action = new QAction( IconLoader::icon( "notify" ), tr( "Enable &Notification" ), this );
        action->setCheckable( true );
        connect( action, SIGNAL( triggered() ), this, SLOT( toggleNotify() ) );
        setAction( "toggleNotify", action );

        action = new QAction( IconLoader::icon( "folder-notify" ), tr( "Enable &Automatic Notifications" ), this );
        action->setCheckable( true );
        connect( action, SIGNAL( triggered() ), this, SLOT( toggleAutoNotify() ) );
        setAction( "toggleAutoNotify", action );
    }

    action = new QAction( IconLoader::icon( "filter" ), tr( "&Change Filter..." ), this );
    action->setShortcut( tr( "Ctrl+F" ) );
    connect( action, SIGNAL( triggered() ), this, SLOT( changeFilter() ) );
    setAction( "changeFilter", action );

    action = new QAction( IconLoader::icon( "filter-save" ), tr( "&Save Filter As..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( saveFilterAs() ) );
    setAction( "saveFilterAs", action );

    action = new QAction( IconLoader::icon( "folder-filter" ), tr( "&Manage Filters..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( manageFilters() ) );
    setAction( "manageFilters", action );

    action = new QAction( IconLoader::icon( "configure-columns" ), tr( "C&onfigure Columns..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( configureColumns() ) );
    setAction( "configureColumns", action );

    action = new QAction( IconLoader::icon( "file-print" ), tr( "&Print List..." ), this );
    action->setShortcut( QKeySequence::Print );
    connect( action, SIGNAL( triggered() ), this, SLOT( printReport() ) );
    setAction( "printReport", action );

    action = new QAction( IconLoader::icon( "export-csv" ), tr( "To &CSV..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( exportCsv() ) );
    setAction( "exportCsv", action );

    action = new QAction( IconLoader::icon( "export-html" ), tr( "To &HTML..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( exportHtml() ) );
    setAction( "exportHtml", action );

    action = new QAction( IconLoader::icon( "export-pdf" ), tr( "To &PDF..." ), this );
    connect( action, SIGNAL( triggered() ), this, SLOT( exportPdf() ) );
    setAction( "exportPdf", action );

    setTitle( "menuMain", tr( "&Folder" ) );
    setTitle( "menuEdit", tr( "&Edit" ) );
    setTitle( "menuExport", tr( "&Export List" ) );

    setButtonStyle( "addIssue", Qt::ToolButtonTextBesideIcon );
    setButtonStyle( "updateFolder", Qt::ToolButtonTextBesideIcon );
    setButtonStyle( "changeFilter", Qt::ToolButtonTextBesideIcon );

    loadXmlUiFile( ":/resources/folderview.xml" );

    QWidget* main = new QWidget( parentWidget );

    QVBoxLayout* mainLayout = new QVBoxLayout( main );
    mainLayout->setMargin( 0 );
    mainLayout->setSpacing( 0 );

    QHBoxLayout* filterLayout = new QHBoxLayout();
    filterLayout->setMargin( 3 );
    filterLayout->setSpacing( 5 );

    mainLayout->addLayout( filterLayout );

    QLabel* filterLabel = new QLabel( tr( "Filte&r:" ), main );
    filterLayout->addWidget( filterLabel );

    m_filterCombo = new SeparatorComboBox( main );
    m_filterCombo->setMaximumWidth( 200 );
    m_filterCombo->setMinimumWidth( 150 );

    connect( m_filterCombo, SIGNAL( activated( int ) ), this, SLOT( filterActivated( int ) ) );

    filterLayout->addWidget( m_filterCombo, 1 );

    filterLabel->setBuddy( m_filterCombo );

    filterLayout->addStretch( 0 );

    QLabel* searchLabel = new QLabel( tr( "&Quick Search:" ), main );
    filterLayout->addWidget( searchLabel );

    m_searchBox = new SearchEditBox( main );
    m_searchBox->setMaximumWidth( 200 );
    m_searchBox->setMinimumWidth( 100 );

    connect( m_searchBox, SIGNAL( textChanged( const QString& ) ), this, SLOT( quickSearchChanged( const QString& ) ) );

    filterLayout->addWidget( m_searchBox, 1 );

    searchLabel->setBuddy( m_searchBox );

    m_searchMenu = new QMenu( m_searchBox );
    m_searchBox->setOptionsMenu( m_searchMenu );

    m_searchActionGroup = new QActionGroup( this );

    connect( m_searchActionGroup, SIGNAL( triggered( QAction* ) ), this, SLOT( searchActionTriggered( QAction* ) ) );

    m_list = new QTreeView( main );
    m_list->setSortingEnabled( true );
    m_list->setRootIsDecorated( false );
    m_list->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_list, SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( listContextMenu( const QPoint& ) ) );
    connect( m_list, SIGNAL( activated( const QModelIndex& ) ),
        this, SLOT( activated( const QModelIndex& ) ) );

    mainLayout->addWidget( m_list );

    m_list->header()->setContextMenuPolicy( Qt::CustomContextMenu );

    connect( m_list->header(), SIGNAL( customContextMenuRequested( const QPoint& ) ),
        this, SLOT( headerContextMenu( const QPoint& ) ) );

    connect( m_searchBox, SIGNAL( deactivate() ), m_list, SLOT( setFocus() ) );

    main->installEventFilter( this );
    m_list->installEventFilter( this );
    m_list->viewport()->installEventFilter( this );

    setMainWidget( main );

    setViewerSizeHint( QSize( 700, 500 ) );

    m_list->setFocus();
}

FolderView::~FolderView()
{
    cleanUp();
}

void FolderView::cleanUp()
{
    if ( isEnabled() ) {
        TreeViewSettings settings;
        settings.openIssuesList( m_typeId );

        settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );
    }

    delete m_filter;
    m_filter = NULL;

    delete m_model;
    m_model = NULL;
}

void FolderView::initialUpdate()
{
    cleanUp();

    m_filter = new IssueRowFilter( this );

    m_model = new RDB::TableItemModel( this );
    m_model->setRowFilter( m_filter );
    m_model->setRootTableModel( new IssuesTableModel( id(), m_model ),
        dataManager->issues()->index(), dataManager->issues()->parentIndex(), id() );

    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateActions() ) );
    connect( m_model, SIGNAL( layoutChanged() ), this, SLOT( updateSummary() ) );
    connect( m_model, SIGNAL( modelReset() ), this, SLOT( updateActions() ) );
    connect( m_model, SIGNAL( modelReset() ), this, SLOT( updateSummary() ) );

    setAccess( checkDataAccess(), true );

    if ( id() != 0 )
        updateFolderIfNeeded();
}

Access FolderView::checkDataAccess()
{
    const FolderRow* folder = dataManager->folders()->find( id() );
    if ( !folder )
        return NoAccess;

    m_projectId = folder->projectId();
    m_typeId = folder->typeId();

    if ( connectionManager->connectionInfo()->access() != AdminAccess ) {
        int userId = connectionManager->connectionInfo()->userId();
        const MemberRow* member = dataManager->members()->find( userId, m_projectId );
        if ( !member )
            return NoAccess;
    }

    const TypeRow* type = dataManager->types()->find( m_typeId );
    if ( !type )
        return UnknownAccess;

    return NormalAccess;
}

void FolderView::enableView()
{
    updateFilters();

    TreeViewSettings settings;
    settings.openIssuesList( m_typeId );

    m_model->setColumns( settings.loadColumns() );

    m_list->setModel( m_model );

    TreeViewHelper::applySortOrder( m_list, settings.loadSortOrder() );
    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );

    connect( m_list->selectionModel(), SIGNAL( selectionChanged( const QItemSelection&, const QItemSelection& ) ),
        this, SLOT( updateActions() ) );

    updateSearchOptions();

    updateCaption();
    updateActions();
}

void FolderView::disableView()
{
    m_list->setModel( NULL );

    updateCaption();
}

void FolderView::updateCaption()
{
    QString name = tr( "Unknown Folder" );
    if ( isEnabled() ) {
        const FolderRow* row = dataManager->folders()->find( id() );
        if ( row )
            name = row->name();
    }
    setCaption( name );
}

void FolderView::updateActions()
{
    m_selectedIssueId = 0;

    QModelIndex index = selectedIndex();
    if ( index.isValid() )
        m_selectedIssueId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();

    action( "openIssue" )->setEnabled( m_selectedIssueId != 0 );
    action( "saveFilterAs" )->setEnabled( m_filterType == NewFilter );

    action( "editWatches" )->setEnabled( true );
    if ( dataManager->folderWatchState( id() ) != NotWatched ) {
        action( "toggleMark" )->setEnabled( m_selectedIssueId != 0 );
        if ( m_selectedIssueId != 0 ) {
            if ( dataManager->issueWatchState( m_selectedIssueId ) == WatchNormal ) {
                action( "toggleMark" )->setText( tr( "Ma&rk Issue as New" ) );
                action( "toggleMark" )->setChecked( false );
            } else {
                action( "toggleMark" )->setText( tr( "Mark Issue as &Read" ) );
                action( "toggleMark" )->setChecked( true );
            }
        }
        int items = m_model->rowCount();
        action( "markAllNew" )->setEnabled( items > 0 );
        action( "markAllRead" )->setEnabled( items > 0 );
    } else {
        action( "toggleMark" )->setEnabled( false );
        action( "markAllNew" )->setEnabled( false );
        action( "markAllRead" )->setEnabled( false );
    }

    if ( connectionManager->connectionInfo()->checkFeature( "notify" ) ) {
        action( "toggleNotify" )->setEnabled( m_selectedIssueId != 0 );
        if ( m_selectedIssueId != 0 ) {
            if ( dataManager->isIssueNotify( m_selectedIssueId ) ) {
                action( "toggleNotify" )->setText( tr( "Disable &Notification" ) );
                action( "toggleNotify" )->setChecked( true );
            } else {
                action( "toggleNotify" )->setText( tr( "Enable &Notification" ) );
                action( "toggleNotify" )->setChecked( false );
            }
        }

        if ( m_autoNotify ) {
            action( "toggleAutoNotify" )->setText( tr( "Disable &Automatic Notifications" ) );
            action( "toggleAutoNotify" )->setChecked( true );
        } else {
            action( "toggleAutoNotify" )->setText( tr( "Enable &Automatic Notifications" ) );
            action( "toggleAutoNotify" )->setChecked( false );
        }
    }

    emit selectedIssueChanged( m_selectedIssueId );
}

void FolderView::updateSummary()
{
    int items = m_model->rowCount();
    int total = m_model->totalCount();

    if ( items == total )
        showSummary( QPixmap(), tr( "%1 issues" ).arg( items ) );
    else
        showSummary( QPixmap(), tr( "%1 issues (of %2 total)" ).arg( items ).arg( total ) );
}

void FolderView::updateFolder()
{
    if ( isEnabled() && !isUpdating() ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->updateFolder( id() );

        executeUpdate( batch );
    }
}

void FolderView::updateFolderIfNeeded()
{
    if ( isEnabled() && !isUpdating() && dataManager->folderUpdateNeeded( id() ) ) {
        UpdateBatch* batch = new UpdateBatch();
        batch->setIfNeeded( true );
        batch->updateFolder( id() );

        executeUpdate( batch );
    }
}

void FolderView::openIssue()
{
    if ( isEnabled() && m_selectedIssueId != 0 )
        viewManager->openIssueView( m_selectedIssueId );
}

void FolderView::addIssue()
{
    if ( isEnabled() ) {
        AddIssueDialog dialog( id(), mainWidget() );
        if ( dialog.exec() == QDialog::Accepted ) {
            int issueId = dialog.issueId();

            setSelectedIssueId( issueId );

            if ( viewManager->isStandAlone( this ) )
                viewManager->openIssueView( issueId );
        }
    }
}

void FolderView::editWatches()
{
    WatchesDialog dialog( id(), mainWidget() );
    dialog.exec();
}

void FolderView::toggleMark()
{
    if ( isEnabled() && m_selectedIssueId != 0 ) {
        if ( dataManager->issueWatchState( m_selectedIssueId ) == WatchNormal )
            dataManager->resetIssueWatchStamp( m_selectedIssueId );
        else
            dataManager->updateIssueWatchStamp( m_selectedIssueId );

        viewManager->postViewEvent( NULL, ViewEvent::RecalculateWatches, id() );
    }
}

void FolderView::markAllNew()
{
    markAllIssues( false );
}

void FolderView::markAllRead()
{
    markAllIssues( true );
}

void FolderView::markAllIssues( bool read )
{
    QList<int> issues = visibleIssues();
    if ( issues.isEmpty() )
        return;

    if ( QMessageBox::warning( mainWidget(), tr( "Warning" ),
        tr( "All currently displayed issues will be marked as %1. Do you want to continue?" )
            .arg( read ? tr( "read" ) : tr( "new" ) ),
        QMessageBox::Ok | QMessageBox::Cancel ) != QMessageBox::Ok )
            return;

    for ( int i = 0; i < issues.count(); i++ ) {
        if ( read )
            dataManager->updateIssueWatchStamp( issues.at( i ) );
        else
            dataManager->resetIssueWatchStamp( issues.at( i ) );
    }

    viewManager->postViewEvent( NULL, ViewEvent::RecalculateWatches, id() );
}

void FolderView::toggleNotify()
{
    if ( m_selectedIssueId != 0 ) {
        NotifyDialog dialog( mainWidget() );
        if ( dialog.notifyIssue( m_selectedIssueId, !dataManager->isIssueNotify( m_selectedIssueId ) ) )
            dialog.exec();
    }
}

void FolderView::toggleAutoNotify()
{
    if ( isEnabled() ) {
        if ( !m_autoNotify && !NotifyDialog::checkPreferences( mainWidget() ) )
            return;

        ColumnConditionSettings settings;
        settings.openIssueFilters( m_typeId );

        settings.setAutoNotify( id(), !m_autoNotify );

        viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, m_typeId );
    }
}

void FolderView::changeFilter()
{
    if ( isEnabled() ) {
        FilterDialog dialog( m_projectId, mainWidget() );

        dialog.setPrompt( tr( "Change filter settings for the current view:" ) );

        TreeViewSettings settings;
        settings.openIssuesList( m_typeId );

        dialog.setAvailableColumns( settings.availableColumns() );
        dialog.setConditions( m_filter->conditions() );

        connect( &dialog, SIGNAL( settingsApplied() ), this, SLOT( applyFilter() ) );

        dialog.exec();
    }
}

void FolderView::saveFilterAs()
{
    if ( isEnabled() && m_filterType == NewFilter ) {
        ColumnConditionSettings settings;
        settings.openIssueFilters( m_typeId );

        SaveFilterDialog dialog( mainWidget() );
        dialog.setExistingFilters( settings.filterNames() );

        if ( dialog.exec() == QDialog::Accepted ) {
            settings.saveFilter( dialog.filterName(), m_filter->conditions() );

            m_filterType = SettingsFilter;
            m_filterName = dialog.filterName();

            m_newConditions.clear();

            viewManager->postViewEvent( NULL, ViewEvent::UpdateFilters, m_typeId );
        }
    }
}

void FolderView::manageFilters()
{
    if ( isEnabled() ) {
        ManageFiltersDialog dialog( id(), mainWidget() );
        dialog.exec();
    }
}

void FolderView::configureColumns()
{
    if ( isEnabled() ) {
        ColumnSettingsDialog dialog( mainWidget() );

        const TypeRow* type = dataManager->types()->find( m_typeId );
        QString name = type ? type->name() : QString();

        dialog.setPrompt( tr( "Default columns settings for issues of type <b>%1</b>:" ).arg( name ) );

        TreeViewSettings settings;
        settings.openIssuesList( m_typeId );

        dialog.setAvailableColumns( settings.availableColumns() );
        dialog.setDefaultColumns( settings.defaultColumns() );
        dialog.setFixedColumns( settings.fixedColumns() );
        dialog.setDefaultSortOrder( settings.defaultSortOrder() );

        dialog.setColumns( settings.loadColumns() );
        dialog.setSortOrder( settings.loadSortOrder() );

        connect( &dialog, SIGNAL( settingsApplied() ), this, SLOT( applyColumns() ) );

        dialog.exec();
    }
}

void FolderView::applyColumns()
{
    if ( isEnabled() ) {
        TreeViewSettings settings;
        settings.openIssuesList( m_typeId );

        ColumnSettingsDialog* dialog = (ColumnSettingsDialog*)sender();
        settings.saveColumns( dialog->columns() );
        settings.saveSortOrder( dialog->sortOrder() );

        viewManager->postViewEvent( metaObject()->className(), ViewEvent::UpdateColumnList, m_typeId );
    }
}

void FolderView::printReport()
{
    ReportDialog dialog( HtmlReportGenerator::Folder, ReportDialog::Print, mainWidget() );
    dialog.setFolder( id(), visibleIssues(), m_filterCombo->currentText() );
    dialog.exec();
}

void FolderView::exportCsv()
{
    ReportDialog dialog( HtmlReportGenerator::Folder, ReportDialog::ExportCsv, mainWidget() );
    dialog.setFolder( id(), visibleIssues(), m_filterCombo->currentText() );
    dialog.exec();
}

void FolderView::exportHtml()
{
    ReportDialog dialog( HtmlReportGenerator::Folder, ReportDialog::ExportHtml, mainWidget() );
    dialog.setFolder( id(), visibleIssues(), m_filterCombo->currentText() );
    dialog.exec();
}

void FolderView::exportPdf()
{
    ReportDialog dialog( HtmlReportGenerator::Folder, ReportDialog::ExportPdf, mainWidget() );
    dialog.setFolder( id(), visibleIssues(), m_filterCombo->currentText() );
    dialog.exec();
}

void FolderView::updateEvent( UpdateEvent* e )
{
    setAccess( checkDataAccess() );

    if ( isEnabled() ) {
        if ( e->unit() == UpdateEvent::Projects )
            updateCaption();

        if ( e->unit() == UpdateEvent::Types )
            updateColumnList( false );
    }

    if ( isEnabled() && m_gotoIssueId != 0 && e->unit() == UpdateEvent::Folder && e->id() == id() ) {
        gotoIssue( m_gotoIssueId, m_gotoItemId );
        m_gotoIssueId = 0;
        m_gotoItemId = 0;
    }

    if ( id() != 0 && e->unit() == UpdateEvent::Projects )
        updateFolderIfNeeded();
}

void FolderView::viewEvent( ViewEvent* e )
{
    if ( isEnabled() ) {
        switch ( e->action() ) {
            case ViewEvent::UpdateColumnList:
                if ( e->id() == m_typeId )
                    updateColumnList( true );
                break;

            case ViewEvent::UpdateFilters:
                if ( e->id() == m_typeId )
                    updateFilters();
                break;

            case ViewEvent::UpdateSettings:
                updateColumnList( true );
                updateFilters();
                break;

            case ViewEvent::RecalculateWatches:
            case ViewEvent::UpdateWatches:
                if ( e->id() == id() )
                    m_model->updateData();
                break;

            default:
                break;
        }
    }
    View::viewEvent( e );
}

void FolderView::headerContextMenu( const QPoint& pos )
{
    QMenu* menu = builder()->contextMenu( "contextHeader" );
    if ( menu )
        menu->exec( m_list->header()->mapToGlobal( pos ) );
}

void FolderView::listContextMenu( const QPoint& pos )
{
    QModelIndex index = m_list->indexAt( pos );

    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
    }

    QString menuName;
    if ( index.isValid() )
        menuName = "contextIssue";
    else
        menuName = "contextNull";

    QMenu* menu = builder()->contextMenu( menuName );
    if ( menu )
        menu->exec( m_list->viewport()->mapToGlobal( pos ) );
}

void FolderView::activated( const QModelIndex& index )
{
    if ( index.isValid() && ( m_activateReason != QEvent::MouseButtonPress || viewManager->isStandAlone( this ) ) ) {
        int issueId = m_model->data( index, RDB::TableItemModel::RowIdRole ).toInt();
        viewManager->openIssueView( issueId );
    }
}

void FolderView::setSelectedIssueId( int issueId )
{
    QModelIndex index = m_model->findCell( 0, issueId, Column_Name );
    if ( !index.isValid() ) {
        setSelectedFilter( QString() );
        index = m_model->findCell( 0, issueId, Column_Name );
    }

    if ( index.isValid() ) {
        m_list->selectionModel()->setCurrentIndex( index, QItemSelectionModel::Current );
        m_list->selectionModel()->select( index, QItemSelectionModel::ClearAndSelect | QItemSelectionModel::Rows );
    }
}

void FolderView::gotoIssue( int issueId, int itemId )
{
    if ( issueId != m_selectedIssueId ) {
        if ( isUpdating() ) {
            m_gotoIssueId = issueId;
            m_gotoItemId = itemId;
        }

        setSelectedIssueId( issueId );
    }

    if ( issueId == m_selectedIssueId )
        emit gotoItem( itemId );
}

void FolderView::updateColumnList( bool resort )
{
    TreeViewSettings settings;
    settings.openIssuesList( m_typeId );

    settings.saveColumnWidths( TreeViewHelper::readColumnWidths( m_list ) );

    QPair<int, Qt::SortOrder> order;
    if ( resort )
        order = settings.loadSortOrder();
    else
        order = TreeViewHelper::readSortOrder( m_list );

    m_model->setColumns( settings.loadColumns() );

    TreeViewHelper::applySortOrder( m_list, order );
    TreeViewHelper::applyColumnWidths( m_list, settings.loadColumnWidths() );

    updateSearchOptions();
}

void FolderView::updateSearchOptions()
{
    if ( !m_model->columns().contains( m_searchColumn ) ) {
        m_searchColumn = Column_Name;
        m_filter->setQuickSearch( m_searchColumn, m_searchBox->text() );
    }

    m_searchMenu->clear();

    for ( int i = 0; i < m_model->columns().count(); i++ )
    {
        int column = m_model->columns().at( i );
        QString name = TableModelsHelper::columnName( column );
        QAction* action = m_searchMenu->addAction( name );
        action->setData( column );
        action->setCheckable( true );
        if ( column == m_searchColumn )
            action->setChecked( true );
        action->setActionGroup( m_searchActionGroup );
    }

    QString prompt = TableModelsHelper::columnName( m_searchColumn );
    m_searchBox->setPromptText( prompt );
}

void FolderView::quickSearchChanged( const QString& text )
{
    m_filter->setQuickSearch( m_searchColumn, text );
}

void FolderView::searchActionTriggered( QAction* action )
{
    m_searchColumn = (Column)action->data().toInt();

    QString prompt = TableModelsHelper::columnName( m_searchColumn );
    m_searchBox->setPromptText( prompt );

    m_filter->setQuickSearch( m_searchColumn, m_searchBox->text() );
}

bool FolderView::eventFilter( QObject* obj, QEvent* e )
{
    if ( obj == mainWidget() ) {
        if ( e->type() == QEvent::ShortcutOverride ) {
            QKeyEvent* ke = (QKeyEvent*)e;
            if ( ke->key() == Qt::Key_F3 && ( ke->modifiers() & ~Qt::ShiftModifier ) == 0 ) {
                if ( isEnabled() ) {
                    m_searchBox->setFocus();
                    ke->accept();
                    return true;
                }
            }
        }
    }

    if ( obj == m_list || obj == m_list->viewport() ) {
        QEvent::Type type = e->type();
        if ( type == QEvent::MouseButtonPress || type == QEvent::MouseButtonDblClick || type == QEvent::KeyPress )
            m_activateReason = type;
    }

    return View::eventFilter( obj, e );
}

QString FolderView::selectedFilter() const
{
    return ( m_filterType == SettingsFilter ) ? m_filterName : QString();
}

void FolderView::setSelectedFilter( const QString& filter )
{
    m_filterType = filter.isEmpty() ? NoFilter : SettingsFilter;
    m_filterName = filter;

    m_searchBox->clear();

    updateFilters();
}

void FolderView::updateFilters()
{
    m_filterCombo->clear();

    m_filterCombo->addItem( tr( "(All Issues)" ) );

    ColumnConditionSettings settings;
    settings.openIssueFilters( m_typeId );

    QStringList filters = settings.filterNames();

    if ( filters.count() > 0 ) {
        filters.sort();

        m_filterCombo->addSeparator();
        m_filterCombo->addItems( filters );

        if ( m_filterType == SettingsFilter ) {
            for ( int i = 0; i < filters.count(); i++ ) {
                if ( filters[ i ] == m_filterName ) {
                    m_filterCombo->setCurrentIndex( i + 2 );
                    break;
                }
            }
        }
    }

    if (  m_newConditions.count() > 0 ) {
        m_filterCombo->addSeparator();
        m_filterCombo->addItem( tr( "(Unnamed Filter)" ) );
        if ( m_filterType == NewFilter )
            m_filterCombo->setCurrentIndex( m_filterCombo->count() - 1 );
    }

    if ( m_filterCombo->currentIndex() == 0 )
        m_filterType = NoFilter;

    loadCurrentFilter();

    m_autoNotify = settings.autoNotify( id() );

    updateActions();
}

void FolderView::filterActivated( int index )
{
    if ( index == 0 )
        m_filterType = NoFilter;
    else if ( index == m_filterCombo->count() - 1 && m_newConditions.count() > 0 )
        m_filterType = NewFilter;
    else {
        m_filterType = SettingsFilter;
        m_filterName = m_filterCombo->currentText();
    }

    loadCurrentFilter();

    updateActions();
}

void FolderView::loadCurrentFilter()
{
    switch ( m_filterType ) {
        case NoFilter:
            m_filter->setConditions( QList<ColumnCondition>() );
            break;

        case SettingsFilter: {
            ColumnConditionSettings settings;
            settings.openIssueFilters( m_typeId );

            QList<ColumnCondition> conditions = settings.loadFilter( m_filterName );
            m_filter->setConditions( conditions );
            break;
        }

        case NewFilter:
            m_filter->setConditions( m_newConditions );
            break;
    }

    emit selectedFilterChanged( selectedFilter() );
}

void FolderView::applyFilter()
{
    if ( isEnabled() ) {
        m_filterType = NewFilter;

        FilterDialog* dialog = (FilterDialog*)sender();
        m_newConditions = dialog->conditions();

        updateFilters();
    }
}

QModelIndex FolderView::selectedIndex()
{
    if ( !m_list->selectionModel() )
        return QModelIndex();

    QModelIndexList selection = m_list->selectionModel()->selectedRows();
    if ( selection.isEmpty() )
        return QModelIndex();

    return selection.at( 0 );
}

QList<int> FolderView::visibleIssues()
{
    QList<int> list;

    int rows = m_model->rowCount();
    for ( int i = 0; i < rows; i++ ) {
        int issueId = m_model->data( m_model->index( i, 0 ), RDB::TableItemModel::RowIdRole ).toInt();
        list.append( issueId );
    }

    return list;
}
