
// vbsingle.cpp
// single-user run-a-job code
// Copyright (c) 2005-2007 by The VoxBo Development Team

// This file is part of VoxBo
// 
// VoxBo is free software: you can redistribute it and/or modify it
// under the terms of the GNU General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// VoxBo is distributed in the hope that it will be useful, but
// WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// General Public License for more details.
// 
// You should have received a copy of the GNU General Public License
// along with VoxBo.  If not, see <http://www.gnu.org/licenses/>.
// 
// For general information on VoxBo, including the latest complete
// source code and binary distributions, manual, and associated files,
// see the VoxBo home page at: http://www.voxbo.org/
//
// original version written by Dan Kimberg


// usage:

// vbx -e <jobtype> <arg> <arg> <arg> [run fake job right now]
// vbx -q jobtype [get jobtype arguments]
// vbx -r <seq> [run seq from private queue]

using namespace std;

#include <stdio.h>
#include <signal.h>
#include <readline/readline.h>
#include <readline/history.h>
#include "vbx.h"
#include "voxbo.h"

int handle_line(string &str);
string ask_next();
void do_assignment(string &str);
void do_list(tokenlist &args);
void do_run(tokenlist &args,int max=-1);
void do_kill(tokenlist &args);
void do_debug(tokenlist &args);
void do_exec(tokenlist &args);
void do_exec_interactive(tokenlist &args);
void do_exec_help(tokenlist &args);
void do_whynot(tokenlist &args);
void do_retry(tokenlist &args);
void do_doover(tokenlist &args);
void do_leapfrog(tokenlist &args);
void do_setqueue(tokenlist &args);
void do_help();
int ask_stop();

//VBJobSpec *find_next_job(VBSequence *s);
//VBSequence *find_sequence(int seqnum);
VBPrefs vbp;
vector<VBHost> hostlist;

pthread_mutex_t mutex_seqlist=PTHREAD_MUTEX_INITIALIZER;
vector<VBSequence> seqlist;
vector<VBJobSpec> speclist;

int
main(int argc,char **argv)
{
  string cmd;
  int ret;

  vbp.init();
  printf("\nWelcome to vbsingle\n");
  printf("A component of VoxBo (v%s)\n",vbversion.c_str());
  printf("Your queue is at: %s\n\n",vbp.queuelist[0].dir.c_str());
  chdir(vbp.queuelist[0].dir.c_str());

  hostlist.push_back(vbp.thishost);
  read_queue(vbp.queuelist[0].dir,seqlist);
  populate_running_jobs(seqlist,hostlist);

  // one-shot usage
  if (argc>1) {
    for (int i=1; i<argc; i++)
      cmd+=(string)argv[i]+" ";
    handle_line(cmd);
    exit(0);
  }
  // interactive usage
  while(1) {
    cmd=ask_next();

    update_queue(vbp.queuelist[0].dir,seqlist);
    process_vbx();
    cleanupqueue(seqlist,vbp.queuelist[0].dir);
    // if there are available CPUs
    prune_queue(seqlist,speclist);

    for(JI j=speclist.begin(); j != speclist.end(); j++) {
      run_job(*j);
    }

    if (!cmd.size())
      continue;
    ret=handle_line(cmd);
    if (ret) {
      printf("\n\nThank you for using vbsingle!\n\n");
      break;  
    }
  }
}

void *
queue_thread()
{
  
}

int
handle_line(string &str)
{
  tokenlist args;
  string cmd;
  args.ParseLine(str);
  cmd=vb_tolower(args[0]);
  if (cmd.size()==0)
    return 0;
  if (cmd.find('=')!=string::npos)
    do_assignment(str);
  else if (cmd=="list" || cmd=="l" || cmd=="ls")
    do_list(args);
  else if (cmd=="run" || cmd=="r")
    do_run(args);
  else if (cmd=="run1" || cmd=="r1")
    do_run(args,1);
  else if (cmd=="kill" || cmd=="k")
    do_kill(args);
  else if (cmd=="debug")
    do_debug(args);
  else if (cmd=="exec")
    do_exec(args);
  else if (cmd=="whynot")
    do_whynot(args);
  else if (cmd=="retry")
    do_retry(args);
  else if (cmd=="doover")
    do_doover(args);
  else if (cmd=="leap")
    do_leapfrog(args);
  else if (cmd=="setqueue")
    do_setqueue(args);
  else if (cmd=="help" || cmd=="h" || cmd=="?")
    do_help();
  else if (cmd=="quit" || cmd=="q" || cmd=="exit")
    return 1;
  else
    printf("[E] sorry, didn't understand command %s\n",cmd.c_str());
  return 0;
}

void
do_assignment(const string &str)
{
  tokenlist args;
  args.SetSeparator("= ");
  args.ParseLine(str);
  if (args.size()<2) {
    printf("[E] sorry, didn't understand your command\n");
    return;
  }
  string var=args[0];
  string val=xstripwhitespace(args.Tail());
  printf("[I] not setting |%s| to |%s|\n",var.c_str(),val.c_str());
}

void
do_kill(tokenlist &args)
{
  if (args.size()!=2) {
    printf("[E] usage: kill <sequence number(s)>\n");
    return;
  }
  vector<int> seqs=numberlist(args[1]);
  for (vector<int>::iterator i=seqs.begin(); i!=seqs.end(); i++) {
    for (SI s=seqlist.begin(); s!= seqlist.end(); s++) {
      if (s->seqnum==*i) {
        // delete_sequencedir(s->seqdir);
        seqlist.erase(s);
        break;
      }
    }
  }
}

void
do_debug(tokenlist &args)
{
  printf("[E] debugging only makes sense when there are bugs\n");
}

void
do_setqueue(tokenlist &args)
{
  if (args.size()!=2) {
    printf("[E] usage: setqueue <queue directory>\n");
    return;
  }
  ino_t ii=vb_direxists(args[1]);
  if (!ii) {
    printf("[E] directory \"%s\" does not exist\n",args(1));
    return;
  }
  vbp.queuelist.clear();
  seqlist.clear();
  vbp.add_queue("VoxBo Queue",args[1],ii);
  chdir(vbp.queuelist[0].dir.c_str());
  read_queue(vbp.queuelist[0].dir,seqlist);
  populate_running_jobs(seqlist,hostlist);
}

void
do_exec(tokenlist &args)
{
  if (vb_tolower(args[1])=="-h") {
    do_exec_help(args);
    return;
  }
  if (vb_tolower(args[1])=="-i") {
    do_exec_interactive(args);
    return;
  }
  vbp.read_jobtypes();
  VBJobSpec js;
  js.jobtype=args[1];

  for (int i=0; i<(int)vbp.jobtypelist.size(); i++)
    if (js.jobtype==vbp.jobtypelist[i].shortname)
      js.jt = &(vbp.jobtypelist[i]);
  if (!js.jt) {
    printf("[E] bad jobtype '%s'\n",js.jobtype.c_str());
    return;
  }
  if (args.size()-2 != (int)js.jt->arguments.size()) {
    // FIXME show argument structure
    printf("[E] wrong number of arguments\n");
    return;
  }
  char dd[STRINGLEN];
  if (getcwd(dd,STRINGLEN))
    js.dirname=dd;
  js.error=-9999;
  js.errorstring="no status code reported";
  for (int i=2; i<args.size(); i++) {
    js.arguments.Add(js.jt->arguments[i-2].name+" "+args[i]);
  }
  run_voxbo_job(js);
}

void
do_exec_interactive(tokenlist &args)
{
  vbp.read_jobtypes();
  VBJobSpec js;
  js.jobtype=args[2];

  for (int i=0; i<(int)vbp.jobtypelist.size(); i++)
    if (js.jobtype==vbp.jobtypelist[i].shortname)
      js.jt = &(vbp.jobtypelist[i]);
  if (!js.jt) {
    printf("[E] bad jobtype '%s'\n",js.jobtype.c_str());
    return;
  }
  char *ret;
  string prompt;
  for (vector<VBArgument>::iterator vi=js.jt->arguments.begin();
       vi != js.jt->arguments.end(); vi++) {
    printf("argument \"%s\" - %s\n",vi->name.c_str(),vi->description.c_str());
    prompt="    "+vi->name+": ";
    ret=readline(prompt.c_str());
    if (ret==NULL) {
      printf("\n");
      return;
    }
    js.arguments.Add(vi->name+" "+ret);
  }
  char dd[STRINGLEN];
  if (getcwd(dd,STRINGLEN))
    js.dirname=dd;
  js.error=-9999;
  js.errorstring="no status code reported";
  run_voxbo_job(js);
}

void
do_exec_help(tokenlist &args)
{
  vbp.read_jobtypes();
  VBJobSpec js;
  js.jobtype=args[2];

  for (int i=0; i<(int)vbp.jobtypelist.size(); i++)
    if (js.jobtype==vbp.jobtypelist[i].shortname)
      js.jt = &(vbp.jobtypelist[i]);
  if (!js.jt) {
    printf("[E] bad jobtype '%s'\n",js.jobtype.c_str());
    return;
  }
  for (vector<VBArgument>::iterator vi=js.jt->arguments.begin();
       vi != js.jt->arguments.end(); vi++) {
    printf("       argument name: %s\n",vi->name.c_str());
    printf("argument description: %s\n",vi->description.c_str());
  }
}

void
do_whynot(tokenlist &args)
{
}

void
do_retry(tokenlist &args)
{
  if (!seqlist.size())
    read_queue(vbp.queuelist[0].dir,seqlist);
  int seqnum;
  if (args.size()==2)
    seqnum=strtol(args[1]);
  else if (args.size()==1)
    seqnum=seqlist[0].seqnum;
  else {
    printf("[E] usage: retry <sequence number>\n");
    return;
  }
  VBSequence *seq=find_sequence(seqnum);
  if (!seq) {
    printf("[E] sequence %d not found\n",seqnum);
    return;
  }
  char jobfile[STRINGLEN];
  for(JI j=seq->speclist.begin(); j!=seq->speclist.end(); j++) {
    if (j->status=='B') {
      j->status='W';
      sprintf(jobfile,"%s/%08d/%05d.job",vbp.queuelist[0].dir.c_str(),seqnum,j->jnum);
      appendline(jobfile,"status W");
    }
  }
  seq->updatecounts();
}

void
do_doover(tokenlist &args)
{
  if (!seqlist.size())
    read_queue(vbp.queuelist[0].dir,seqlist);
  int seqnum;
  if (args.size()==2)
    seqnum=strtol(args[1]);
  else if (args.size()==1)
    seqnum=seqlist[0].seqnum;
  else {
    printf("[E] usage: doover <sequence number>\n");
    return;
  }
  VBSequence *seq=find_sequence(seqnum);
  if (!seq) {
    printf("[E] sequence %d not found\n",seqnum);
    return;
  }
  char jobfile[STRINGLEN];
  for(JI j=seq->speclist.begin(); j!=seq->speclist.end(); j++) {
    j->status='W';
    sprintf(jobfile,"%s/%08d/%05d.job",vbp.queuelist[0].dir.c_str(),seqnum,j->jnum);
    appendline(jobfile,"status W");
  }
  seq->updatecounts();
}

void
do_leapfrog(tokenlist &args)
{
  if (!seqlist.size())
    read_queue(vbp.queuelist[0].dir,seqlist);
  if (args.size()!=2) {
    printf("[E] usage: leapfrog <sequence number>\n");
    return;
  }
  int seqnum=strtol(args[1]);
  VBSequence *seq=find_sequence(seqnum);
  if (!seq) {
    printf("[E] sequence %d not found\n",seqnum);
    return;
  }
  char jobfile[STRINGLEN];
  for(JI j=seq->speclist.begin(); j!=seq->speclist.end(); j++) {
    if (j->status=='B') {
      j->status='D';
      sprintf(jobfile,"%s/%08d/%05d.job",vbp.queuelist[0].dir.c_str(),seqnum,j->jnum);
      appendline(jobfile,"status D");
    }
  }
}

void
do_list(tokenlist &args)
{
  if (!seqlist.size())
    read_queue(vbp.queuelist[0].dir,seqlist);
  if (!seqlist.size()) {
    printf("The queue is empty.\n");
    return;
  }

  printf("\n");
  printf("Sequence Name        Num  Pri Owner    ");
  printf("Wait  Run   Bad   Done Total\n");
  int diecnt=0,postcnt=0;
  for(SI s=seqlist.begin(); s!=seqlist.end(); s++) {
    if (s->name.size())
      printf("%-20.20s ",s->name.c_str());
    else
      printf("%-20.20s ","<unnamed sequence>");
    printf("%-5d",s->seqnum);
    printf(" %-3d",s->priority);
    printf("%-10.10s",s->owner.c_str());
    printf("%-6d",s->waitcnt);
    printf("%-6d",s->runcnt);
    printf("%-6d",s->badcnt);
    printf("%-5d",s->donecnt);
    printf("%-6d",s->jobcnt);
    if (s->status == 'K') {
      printf("[*]");
      diecnt++;
    }
    else if (s->status == 'P') {
      printf("[P]");
      postcnt++;
    }
    printf("\n");
  }
  if (diecnt)
    printf("  [*] this sequence has been killed\n");
  if (postcnt)
    printf("  [P] this sequence has been postponed\n");
}

void
do_run(tokenlist &args,int max)
{
  if (!seqlist.size())
    read_queue(vbp.queuelist[0].dir,seqlist);
  int interactivemode=0;
  vector<int> seqnums,tmpnums;
  for (int i=1; i<args.size(); i++) {
    if (args[i]=="-i")
      interactivemode=1;
    else {
      tmpnums=numberlist(args[i]);
      if (tmpnums.size()==0) {
        printf("[E] invalid sequence number(s)\n");
        return;
      }
      for (int j=0; j<(int)tmpnums.size(); j++)
        seqnums.push_back(tmpnums[j]);
    }
  }
  if (seqnums.size()==0)
    seqnums.push_back(seqlist[0].seqnum);
  vbp.read_jobtypes();
  for (int i=0; i<(int)seqnums.size(); i++) {
    int seqnum=seqnums[i];
    int cnt=0;
    VBSequence *seq=find_sequence(seqnum);
    if (!seq) {
      printf("[E] sequence %d not found\n",seqnum);
      continue;
    }
    char jobfile[STRINGLEN];
    while(1) {
      // FIXME FIXME FIXME VBJobSpec *j=find_next_job(seq);
      VBJobSpec *j=NULL;
      if (j==NULL) {
        printf("[I] No jobs in sequence %d ready to run.\n",seqnum);
        break;
      }
      j->jt=NULL;
      for (int i=0; i<(int)vbp.jobtypelist.size(); i++)
        if (j->jobtype==vbp.jobtypelist[i].shortname)
          j->jt = &(vbp.jobtypelist[i]);
      if (j->jt==NULL) {
        printf("[E] jobtype %s not found\n",j->jobtype.c_str());
        return;
      }
      run_voxbo_job(*j);
      printf("[I] job %08d-%05d returned error code %d\n",j->snum,j->jnum,j->error);
      sprintf(jobfile,"%s/%08d/%05d.job",vbp.queuelist[0].dir.c_str(),seqnum,j->jnum);
      if (j->GetState()==XGood || j->GetState()==XWarn) {
        appendline(jobfile,"status D");
        j->status='D';
      }
      else if (j->GetState()==XRetry) {
        appendline(jobfile,"status W");
        j->status='W';
      }
      else {
        appendline(jobfile,"status B");
        j->status='B';
      }
      seq->updatecounts();
      if (seq->donecnt==seq->jobcnt) {
        printf("[I] Sequence %d all done!\n",seqnum);
        break;
      }
      if (max>0 && ++cnt>=max)
        break;
      if (interactivemode) {
        if (ask_stop()) {
          printf("[I] Okay, stopping.\n");
          break;
        }
      }
    }
  }
}

int
ask_stop()
{
  while (TRUE) {
    termios tsave,tnew;
    tcgetattr(0,&tsave);
    tcgetattr(0,&tnew);
    tnew.c_lflag&=~(ICANON|ECHO);
    tnew.c_cc[VMIN]=0;
    tnew.c_cc[VTIME]=30;  // FIXME hardcoded 3s delay!
    tcsetattr(0,TCSADRAIN,&tnew);
    string str;
    cout << "Continue? [Yes/No/Wait] " << flush;
    str=cin.get();
    cin.clear();
    tcsetattr(0,TCSADRAIN,&tsave);
    str=vb_tolower(str);
    cout << str << endl;
    if (str.size()==0)
      return 0;
    if ((int)str[0]==-1)
      return 0;
    if (str[0]=='y')
      return 0;
    if (str[0]=='n')
      return 1;
    if (str[0]=='w')
      sleep(10);
  }
}

void
do_help()
{
  printf("\n");
  printf("VoxBo vbsingle (v%s) commands:\n",vbversion.c_str());
  printf(" list            list what's in the queue\n");
  printf(" run <seq>       run seq number\n");
  printf(" run1 <seq>      run one job from seq\n");
  printf(" kill <seq>      \n");
  printf(" debug <seq>     \n");
  printf(" exec <cmd>      cmd is a jobtype followed by its arguments\n");
  printf(" retry <seq>     mark bad as good, try again\n");
  printf(" doover <seq>   start from scratch\n");
  printf(" leapfrog <seq>  mark bad as done, keep going\n");
  printf(" help            this screen\n");
  printf("\n");
}

// VBJobSpec *
// find_next_job(VBSequence *s)
// {
//   for (JI j=s->speclist.begin(); j!=s->speclist.end(); j++) {
//     if (j->status!='W')
//       continue;
//     if (has_unsatisfied_dependencies(&(*s),*j))
//       continue;
//     return &(*j);
//   }
//   return NULL;
// }

// VBSequence *
// find_sequence(int seqnum)
// {
//   for (SI s=seqlist.begin(); s!=seqlist.end(); s++) {
//     if (s->seqnum==seqnum)
//       return &(*s);
//   }
//   return NULL;
// }

string
ask_next()
{
  char *ret=readline("vbsingle> ");
  if (ret) add_history(ret);
  if (ret)
    return ret;
  else
    return "quit";
}
