/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <ug_utils.h>
#include <ug_list_view.h>
#include <ug_data_app.h>
#include <ug_item.h>

#include <glib/gi18n.h>

// ----------------------------------------------------------------------------
// Icons

static gchar* list_icon_name [UG_LIST_N_ICON] =
{
	GTK_STOCK_FILE,				// UG_LIST_ICON_FILE
	GTK_STOCK_MEDIA_PAUSE,		// UG_LIST_ICON_PAUSED
	GTK_STOCK_MEDIA_PLAY,		// UG_LIST_ICON_RUNNING
	GTK_STOCK_EXECUTE,			// UG_LIST_ICON_EXECUTE
	GTK_STOCK_REFRESH,			// UG_LIST_ICON_REFRESH
	GTK_STOCK_YES,				// UG_LIST_ICON_COMPLETED
	GTK_STOCK_DELETE,			// UG_LIST_ICON_RECYCLED
	GTK_STOCK_DND_MULTIPLE,		// UG_LIST_ICON_CATEGORY
	GTK_STOCK_DIRECTORY,		// UG_LIST_ICON_FOLDER
	GTK_STOCK_NETWORK,			// UG_LIST_ICON_NETWORK
	GTK_STOCK_INFO,				// UG_LIST_ICON_INFO
	GTK_STOCK_DIALOG_WARNING,	// UG_LIST_ICON_WARNING
	GTK_STOCK_DIALOG_ERROR,		// UG_LIST_ICON_ERROR
};

static GdkPixbuf* list_icons [UG_LIST_N_ICON]      = {0};

static void widget_render_icons (GtkWidget* widget)
{
	guint index;

	if (list_icons[0])
		return;

	// render list_icons for GtkTreeView
	for(index=0; index < UG_LIST_N_ICON; ++index) {
		if (list_icon_name[index] == NULL)
			continue;

		list_icons[index] = gtk_widget_render_icon (widget,
		                                            list_icon_name[index],
		                                            GTK_ICON_SIZE_MENU,
		                                            NULL);
	}
}


// ----------------------------------------------------------------------------
// Category List View

static void category_col_set_icon (GtkTreeViewColumn *tree_column,
                                   GtkCellRenderer   *cell,
                                   GtkTreeModel      *model,
                                   GtkTreeIter       *iter,
                                   gpointer           data)
{
	UgCategory*	category;
	UgListIcon	icon_index;

	gtk_tree_model_get (model, iter, 0, &category, -1);
	icon_index = category->list_icon;
	if (icon_index >= UG_LIST_N_ICON)
		icon_index = UG_LIST_ICON_CATEGORY;

	g_object_set (cell, "pixbuf", list_icons[icon_index], NULL);
}

static void category_col_set_name (GtkTreeViewColumn *tree_column,
                                   GtkCellRenderer   *cell,
                                   GtkTreeModel      *model,
                                   GtkTreeIter       *iter,
                                   gpointer           data)
{
	UgCategory*	category;

	gtk_tree_model_get (model, iter, 0, &category, -1);
	g_object_set (cell, "text", category->name, NULL);
}

static void category_col_set_quantity (GtkTreeViewColumn *tree_column,
                                       GtkCellRenderer   *cell,
                                       GtkTreeModel      *model,
                                       GtkTreeIter       *iter,
                                       gpointer           data)
{
	UgCategory*	category;
	gchar*		string;
	gint		quantity;

	gtk_tree_model_get (model, iter, 0, &category, -1);
	quantity = gtk_tree_model_iter_n_children ((GtkTreeModel*) category->download_store, NULL);
	string = g_strdup_printf ("%u", quantity);
	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

GtkTreeView* ug_category_view_new (void)
{
	GtkTreeView*       tview;
	GtkCellRenderer*   renderer;
	GtkTreeViewColumn* column;

	tview = (GtkTreeView*)gtk_tree_view_new ();
	gtk_tree_view_set_headers_visible (tview, FALSE);

	widget_render_icons ((GtkWidget*)tview);

	// column name
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Category"));
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         category_col_set_icon,
	                                         NULL, NULL);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         category_col_set_name,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
//	gtk_tree_view_column_set_min_width (column, 120);
	gtk_tree_view_column_set_expand (column, TRUE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column jobs
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Quantity"));
	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "xalign", 1.0, NULL);
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         category_col_set_quantity,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_expand (column, TRUE);
	gtk_tree_view_column_set_alignment (column, 1.0);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	gtk_widget_show (GTK_WIDGET (tview));
	return tview;
}


// ----------------------------------------------------------------------------
// Download View

// data set function for GtkCellRenderer ----------------------------
static void col_set_list_icon (GtkTreeViewColumn *tree_column,
                                GtkCellRenderer   *cell,
                                GtkTreeModel      *model,
                                GtkTreeIter       *iter,
                                gpointer           data)
{
	UgDataset*	dataset;
	UgListIcon	icon_index;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	if (UG_DATASET_APP (dataset))
		icon_index = UG_DATASET_APP (dataset)->list_icon;
	else
		icon_index = UG_LIST_ICON_ERROR;

	if (icon_index >= UG_LIST_N_ICON)
		icon_index = UG_LIST_ICON_ERROR;

	g_object_set (cell, "pixbuf", list_icons [icon_index], NULL);
}

static void col_set_name (GtkTreeViewColumn *tree_column,
                          GtkCellRenderer   *cell,
                          GtkTreeModel      *model,
                          GtkTreeIter       *iter,
                          gpointer           data)
{
	UgDataset*		dataset;
	UgDataCommon*	common;
	gchar*			string = NULL;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	common = UG_DATASET_COMMON (dataset);

	if (common->name)
		string = common->name;
	else
		string = common->file;

	g_object_set (cell, "text", string, NULL);
}

static void col_set_complete (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer   *cell,
                              GtkTreeModel      *model,
                              GtkTreeIter       *iter,
                              gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_dtoa_unit ((gdouble) progress->complete, 1, NULL);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_total (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_dtoa_unit ((gdouble) progress->total, 1, NULL);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_percent (GtkTreeViewColumn *tree_column,
                             GtkCellRenderer   *cell,
                             GtkTreeModel      *model,
                             GtkTreeIter       *iter,
                             gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress) {
//		string = g_strdup_printf ("%.*f%c", 1, progress->percent, '%');
		string = g_strdup_printf ("%.1f%c", progress->percent, '%');
		g_object_set (cell, "visible", 1, NULL);
		g_object_set (cell, "value", (gint) progress->percent, NULL);
		g_object_set (cell, "text", string, NULL);
		g_free (string);
	}
	else {
		g_object_set (cell, "visible", 0, NULL);
		g_object_set (cell, "value", 0, NULL);
		g_object_set (cell, "text", "", NULL);
	}
}

static void col_set_consume_time (GtkTreeViewColumn *tree_column,
                                  GtkCellRenderer   *cell,
                                  GtkTreeModel      *model,
                                  GtkTreeIter       *iter,
                                  gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress)
		string = ug_str_from_time ((guint) progress->consume_time, TRUE);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_remain_time (GtkTreeViewColumn *tree_column,
                                 GtkCellRenderer   *cell,
                                 GtkTreeModel      *model,
                                 GtkTreeIter       *iter,
                                 gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress && UG_DATASET_APP (dataset) && UG_DATASET_APP (dataset)->plugin)
		string = ug_str_from_time ((guint) progress->remain_time, TRUE);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_speed (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*	dataset;
	UgProgress*	progress;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	progress = UG_DATASET_PROGRESS (dataset);

	if (progress && UG_DATASET_APP (dataset) && UG_DATASET_APP (dataset)->plugin)
		string = ug_str_dtoa_unit (progress->average_speed, 1, "/s");
//		string = ug_str_dtoa_unit (progress->current_speed, 1, "/s");
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_retry (GtkTreeViewColumn *tree_column,
                           GtkCellRenderer   *cell,
                           GtkTreeModel      *model,
                           GtkTreeIter       *iter,
                           gpointer           data)
{
	UgDataset*		dataset;
	UgDataCommon*	common;
	gchar*			string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	common = UG_DATASET_COMMON (dataset);

	if (common->retry_count != 0)
		string = g_strdup_printf ("%d", common->retry_count);
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
	g_free (string);
}

static void col_set_category (GtkTreeViewColumn *tree_column,
                              GtkCellRenderer   *cell,
                              GtkTreeModel      *model,
                              GtkTreeIter       *iter,
                              gpointer           data)
{
	UgDataset*	dataset;
	UgDataApp*	app_data;
	gchar*		string;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);
	app_data = UG_DATASET_APP (dataset);

	if (app_data)
		string = app_data->category_name;
	else
		string = NULL;

	g_object_set (cell, "text", string, NULL);
}

static void col_set_url (GtkTreeViewColumn *tree_column,
                         GtkCellRenderer   *cell,
                         GtkTreeModel      *model,
                         GtkTreeIter       *iter,
                         gpointer           data)
{
	UgDataset*	dataset;

	gtk_tree_model_get (model, iter, 0, &dataset, -1);

	g_object_set (cell, "text", UG_DATASET_COMMON (dataset)->url, NULL);
}

GtkTreeView*	ug_download_view_new (void)
{
	GtkTreeView*       tview;
	GtkTreeSelection*  selection;
	GtkCellRenderer*   renderer;
	GtkCellRenderer*   renderer_progress;
	GtkTreeViewColumn* column;

	tview = (GtkTreeView*)gtk_tree_view_new ();
	gtk_tree_view_set_fixed_height_mode (tview, TRUE);
	selection = gtk_tree_view_get_selection (tview);
	gtk_tree_selection_set_mode (selection, GTK_SELECTION_MULTIPLE);

	widget_render_icons ((GtkWidget*)tview);

	// column name
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Name"));
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_list_icon,
	                                         NULL, NULL);

	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_name,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 180);
	gtk_tree_view_column_set_expand (column, TRUE);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column completed
	column = gtk_tree_view_column_new ();
	renderer = gtk_cell_renderer_text_new ();
	g_object_set (renderer, "xalign", 1.0, NULL);
	gtk_tree_view_column_set_title (column, _("Complete"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_complete,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 80);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column total
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Size"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_total,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 80);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column percent
	column = gtk_tree_view_column_new ();
	renderer_progress = gtk_cell_renderer_progress_new ();
	gtk_tree_view_column_set_title (column, _("%"));
	gtk_tree_view_column_pack_start (column, renderer_progress, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer_progress,
	                                         col_set_percent,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 60);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column "Elapsed" for consuming time
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Elapsed"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_consume_time,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 65);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column "Left" for remaining time
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Left"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_remain_time,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 65);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// columns speed
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Speed"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_speed,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 90);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column retries
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Retry"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_retry,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 45);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_alignment (column, 1.0);
	gtk_tree_view_append_column (tview, column);

	// column category
	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("Category"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_category,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 100);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column url
//	renderer = gtk_cell_renderer_text_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, _("URL"));
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         col_set_url,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 200);
	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	gtk_widget_show (GTK_WIDGET (tview));
	return tview;
}

// ----------------------------------------------------------------------------
// Item View

static void item_col_set_toggle (GtkTreeViewColumn *tree_column,
                                 GtkCellRenderer   *cell,
                                 GtkTreeModel      *model,
                                 GtkTreeIter       *iter,
                                 gpointer          *data)
{
	UgItem*		item;

	gtk_tree_model_get (model, iter, 0, &item, -1);
	g_object_set (cell, "active", item->mark, NULL);
}

static void item_on_toggled (GtkCellRendererToggle* cell,
                             gchar*                 path_str,
                             GtkTreeView*           view)
{
	GtkTreeIter		iter;
	GtkTreePath*	path  = gtk_tree_path_new_from_string (path_str);
	GtkTreeModel*	model = gtk_tree_view_get_model (view);
	UgItem*			item;

	gtk_tree_model_get_iter (model, &iter, path);
	gtk_tree_model_get (model, &iter, 0, &item, -1);
	item->mark ^= 1;

	gtk_tree_path_free (path);
}

static void item_col_set_icon (GtkTreeViewColumn *tree_column,
                               GtkCellRenderer   *cell,
                               GtkTreeModel      *model,
                               GtkTreeIter       *iter,
                               gpointer           data)
{
	UgItem*		item;
	UgListIcon	icon_index;

	gtk_tree_model_get (model, iter, 0, &item, -1);
	icon_index = item->list_icon;
	if (icon_index >= UG_LIST_N_ICON)
		g_object_set (cell, "pixbuf", NULL, NULL);
	else
		g_object_set (cell, "pixbuf", list_icons[icon_index], NULL);
}

static void item_col_set_name (GtkTreeViewColumn *tree_column,
                               GtkCellRenderer   *cell,
                               GtkTreeModel      *model,
                               GtkTreeIter       *iter,
                               gpointer           data)
{
	UgItem*	item;

	gtk_tree_model_get (model, iter, 0, &item, -1);
	g_object_set (cell, "text", item->name, NULL);
}

static void item_col_set_value (GtkTreeViewColumn *tree_column,
                                GtkCellRenderer   *cell,
                                GtkTreeModel      *model,
                                GtkTreeIter       *iter,
                                gpointer           data)
{
	UgItem*	item;

	gtk_tree_model_get (model, iter, 0, &item, -1);
	g_object_set (cell, "text", item->value, NULL);
}

GtkTreeView* ug_item_view_new (const gchar* name_title, const gchar* value_title)
{
	GtkTreeView*		tview;
	GtkCellRenderer*	renderer;
	GtkTreeViewColumn*	column;

	tview = (GtkTreeView*) gtk_tree_view_new ();
//	gtk_tree_view_set_fixed_height_mode (tview, TRUE);
//	gtk_tree_view_set_headers_visible (tview, FALSE);

	widget_render_icons ((GtkWidget*)tview);

	// Toggle
	renderer = gtk_cell_renderer_toggle_new ();
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, "M");
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         (GtkTreeCellDataFunc)item_col_set_toggle,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, FALSE);
	gtk_tree_view_column_set_alignment (column, 0.5);
	gtk_tree_view_column_set_min_width (column, 15);
	gtk_tree_view_append_column (tview, column);
	g_signal_connect (renderer, "toggled", G_CALLBACK (item_on_toggled), tview);

	// column name
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, name_title);
	renderer = gtk_cell_renderer_pixbuf_new ();
	gtk_tree_view_column_pack_start (column, renderer, FALSE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         item_col_set_icon,
	                                         NULL, NULL);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         item_col_set_name,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
//	gtk_tree_view_column_set_min_width (column, 120);
//	gtk_tree_view_column_set_expand (column, TRUE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	// column value
	column = gtk_tree_view_column_new ();
	gtk_tree_view_column_set_title (column, value_title);
	renderer = gtk_cell_renderer_text_new ();
	gtk_tree_view_column_pack_start (column, renderer, TRUE);
	gtk_tree_view_column_set_cell_data_func (column,
	                                         renderer,
	                                         item_col_set_value,
	                                         NULL, NULL);
	gtk_tree_view_column_set_resizable (column, TRUE);
	gtk_tree_view_column_set_alignment (column, 0.0);
//	gtk_tree_view_column_set_expand (column, TRUE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_append_column (tview, column);

	gtk_widget_show (GTK_WIDGET (tview));
	return tview;
}

void	ug_item_view_for_summary (GtkTreeView* item_view)
{
	GtkTreeViewColumn*	column;

	gtk_tree_view_set_headers_visible (item_view, FALSE);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_TOGGLE);
	gtk_tree_view_column_set_visible (column, FALSE);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_NAME);
	gtk_tree_view_column_set_visible (column, TRUE);
	gtk_tree_view_column_set_min_width (column, 50);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_VALUE);
	gtk_tree_view_column_set_visible (column, TRUE);
	gtk_tree_view_column_set_expand (column, TRUE);
}

void	ug_item_view_for_selector (GtkTreeView* item_view)
{
	GtkTreeViewColumn*	column;

//	gtk_tree_view_set_fixed_height_mode (item_view, TRUE);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_TOGGLE);
	gtk_tree_view_column_set_visible (column, TRUE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_NAME);
	gtk_tree_view_column_set_visible (column, FALSE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);

	column = gtk_tree_view_get_column (item_view, UG_ITEM_VIEW_COLUMN_VALUE);
//	gtk_tree_view_column_set_sizing (column, GTK_TREE_VIEW_COLUMN_FIXED);
	gtk_tree_view_column_set_resizable (column, FALSE);
	gtk_tree_view_column_set_visible (column, TRUE);
	gtk_tree_view_column_set_expand (column, TRUE);
}

