/*
 *
 *   Copyright (C) 2005-2010 by Raymond Huang
 *   plushuang at users.sourceforge.net
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License, or (at your option) any later version.
 *
 *  This library is distributed in the hope that it will be useful,
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 51 Franklin St, Fifth Floor, Boston, MA  02110-1301  USA
 *
 *  ---
 *
 *  In addition, as a special exception, the copyright holders give
 *  permission to link the code of portions of this program with the
 *  OpenSSL library under certain conditions as described in each
 *  individual source file, and distribute linked combinations
 *  including the two.
 *  You must obey the GNU Lesser General Public License in all respects
 *  for all of the code used other than OpenSSL.  If you modify
 *  file(s) with this exception, you may extend this exception to your
 *  version of the file(s), but you are not obligated to do so.  If you
 *  do not wish to do so, delete this exception statement from your
 *  version.  If you delete this exception statement from all source
 *  files in the program, then also delete it here.
 *
 */

#include <memory.h>
#include <ug_option.h>
#include <ug_data_download.h>

void ug_option_set_group (const UgOption* option, GOptionGroup* group)
{
	g_option_group_add_entries (group, option->option_entry);
}

void ug_option_get_dataset (const UgOption* option, UgDataset* dataset)
{
	option->get_dataset (option, dataset);
}

void ug_option_init_data (const UgOption* option)
{
	memset (option->option_data, 0, option->option_data_size);
	if (option->init)
		option->init (option->option_data);
}

// ---------------------------------------------------------------------------
// UgOptionMainData

static void		ug_option_main_init	(UgOptionMainData* option_data);
static gboolean	ug_option_main_get_dataset (UgOption* option, UgDataset* dataset);

static UgOptionMainData		option_main_data;
static GOptionEntry			option_main_entry[] =
{
	{"quiet",			0,	0,	G_OPTION_ARG_NONE,		&option_main_data.quiet,			"add download directly. Don't show dialog.",	NULL},
	{"category-index",	0,	0,	G_OPTION_ARG_INT,		&option_main_data.category_index,	"add download to Nth category. (default -1)",	"N"},
	{"input-file",	  'i',	0,	G_OPTION_ARG_STRING,	&option_main_data.input_file,		"add URLs found in FILE.",						"FILE"},

	{"folder",			0,	0,	G_OPTION_ARG_STRING,	&option_main_data.folder,			"placed download file in FOLDER.",		"FOLDER"},
	{"filename",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.filename,			"set download filename to FILE.",		"FILE"},

	{"user",			0,	0,	G_OPTION_ARG_STRING,	&option_main_data.user,				"set both ftp and http user to USER.",		"USER"},
	{"password",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.password,			"set both ftp and http password to PASS.",	"PASS"},

	{"proxy-type",		0,	0,	G_OPTION_ARG_INT,		&option_main_data.proxy_type,		"set proxy type to N. (0=Don't use)",	"N"},
	{"proxy-host",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.proxy_host,		"set proxy host to HOST.",				"HOST"},
	{"proxy-port",		0,	0,	G_OPTION_ARG_INT,		&option_main_data.proxy_port,		"set proxy port to PORT.",				"PORT"},
	{"proxy-user",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.proxy_user,		"set USER as proxy username.",			"USER"},
	{"proxy-password",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.proxy_password,	"set PASS as proxy password.",			"PASS"},

	{"http-user",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_user,		"set http user to USER.",				"USER"},
	{"http-password",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_password,	"set http password to PASS.",			"PASS"},
	{"http-referer",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_referer,		"include `Referer: URL' header in HTTP request.",	"URL"},
	{"http-cookie-data",0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_cookie_data,	"load cookies from STRING.",						"STRING"},
	{"http-cookie-file",0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_cookie_file,	"load cookies from FILE.",							"FILE"},
	{"http-post-data",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_post_data,	"use the POST method; send STRING as the data.",	"STRING"},
	{"http-post-file",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.http_post_file,	"use the POST method; send contents of FILE",		"FILE"},

	{"ftp-user",		0,	0,	G_OPTION_ARG_STRING,	&option_main_data.ftp_user,			"set ftp user to USER.",				"USER"},
	{"ftp-password",	0,	0,	G_OPTION_ARG_STRING,	&option_main_data.ftp_password,		"set ftp password to PASS.",			"PASS"},

	{NULL}
};

static	UgOption	option_main =
{
	"main",
	NULL,
	option_main_entry,
	&option_main_data,
	sizeof (option_main_data),

	(UgInitFunc)			ug_option_main_init,
	(UgGetDatasetFunc)		ug_option_main_get_dataset,
};

// extern
const	UgOption*	UgOptionMain = &option_main;

static void	ug_option_main_init	(UgOptionMainData* option_data)
{
	option_data->category_index = -1;
}

static gboolean	ug_option_main_get_dataset (UgOption* option, UgDataset* dataset)
{
	UgOptionMainData*	option_data;
	gboolean		result = FALSE;
	UgDataCommon*	common;
	UgDataProxy*	proxy;
	UgDataHttp*		http;
	UgDataFtp*		ftp;

	option_data = option->option_data;
	// common
	if (option_data->folder) {
		common = ug_dataset_realloc (dataset, UgDataCommonClass, 0);
		g_free (common->folder);
		common->folder = g_strdup (option_data->folder);
		result = TRUE;
	}

	if (option_data->filename) {
		common = ug_dataset_realloc (dataset, UgDataCommonClass, 0);
		g_free (common->file);
		common->file = g_strdup (option_data->filename);
		result = TRUE;
	}

	if (option_data->user) {
		common = ug_dataset_realloc (dataset, UgDataCommonClass, 0);
		g_free (common->user);
		common->user = g_strdup (option_data->user);
		result = TRUE;
	}

	if (option_data->password) {
		common = ug_dataset_realloc (dataset, UgDataCommonClass, 0);
		g_free (common->password);
		common->password = g_strdup (option_data->password);
		result = TRUE;
	}

	// proxy
	if (option_data->proxy_type) {
		proxy = ug_dataset_realloc (dataset, UgDataProxyClass, 0);
		proxy->type = option_data->proxy_type;
		result = TRUE;
	}

	if (option_data->proxy_host) {
		proxy = ug_dataset_realloc (dataset, UgDataProxyClass, 0);
		g_free (proxy->host);
		proxy->host = g_strdup (option_data->proxy_host);
		result = TRUE;
	}

	if (option_data->proxy_port) {
		proxy = ug_dataset_realloc (dataset, UgDataProxyClass, 0);
		proxy->port = option_data->proxy_port;
		result = TRUE;
	}

	if (option_data->proxy_user) {
		proxy = ug_dataset_realloc (dataset, UgDataProxyClass, 0);
		g_free (proxy->user);
		proxy->user = g_strdup (option_data->proxy_user);
		result = TRUE;
	}

	if (option_data->proxy_password) {
		proxy = ug_dataset_realloc (dataset, UgDataProxyClass, 0);
		g_free (proxy->password);
		proxy->password = g_strdup (option_data->proxy_password);
		result = TRUE;
	}

	// http
	if (option_data->http_user) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->user);
		http->user = g_strdup (option_data->http_user);
		result = TRUE;
	}

	if (option_data->proxy_password) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->password);
		http->password = g_strdup (option_data->proxy_password);
		result = TRUE;
	}

	if (option_data->http_referer) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->referer);
		http->referer = g_strdup (option_data->http_referer);
		result = TRUE;
	}

	if (option_data->http_cookie_data) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->cookie_data);
		http->cookie_data = g_strdup (option_data->http_cookie_data);
		result = TRUE;
	}

	if (option_data->http_cookie_file) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->cookie_file);
		http->cookie_file = g_strdup (option_data->http_cookie_file);
		result = TRUE;
	}

	if (option_data->http_post_data) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->post_data);
		http->post_data = g_strdup (option_data->http_post_data);
		result = TRUE;
	}

	if (option_data->http_post_file) {
		http = ug_dataset_realloc (dataset, UgDataHttpClass, 0);
		g_free (http->post_file);
		http->post_file = g_strdup (option_data->http_post_file);
		result = TRUE;
	}

	// ftp
	if (option_data->ftp_user) {
		ftp = ug_dataset_realloc (dataset, UgDataFtpClass, 0);
		g_free (ftp->user);
		ftp->user = g_strdup (option_data->ftp_user);
		result = TRUE;
	}

	if (option_data->ftp_password) {
		ftp = ug_dataset_realloc (dataset, UgDataFtpClass, 0);
		g_free (ftp->password);
		ftp->password = g_strdup (option_data->ftp_password);
		result = TRUE;
	}

	return result;
}

