
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: utils_files.h 2565 2007-07-21 12:02:03Z mschwerin $
 *
 */

#include "types.h"

#include <sys/types.h>

#ifndef HAVE_UTILS_FILES_H
#define HAVE_UTILS_FILES_H


/// Resolves a relative to an absolute filename.
/**
 * The returned string must be freed.
 */
char *relative_to_absolute (const char *filename);


/// Resolves a softlink.
/**
 * If the passed MRL is a softlink the target is returned. If it is not a
 * softlink the MRL itself is returned.
 *
 * The returned string must be freed.
 */
char *resolve_softlink (const char *mrl);


/// Returns the directory of the passed MRL.
/**
 * The returned string must be freed.
 */
char *get_dirname (const char *mrl);


/// Returns the base name of the passed MRL.
/**
 * The returned string must be freed.
 */
char *get_basename (const char *mrl);


/// Returns the extension of the passed MRL.
const char *get_extension (const char *mrl);


/// Returns the size of a file.
off_t get_filesize (const char *mrl);


/// Returns TRUE if the passed filename matches the passed extension.
bool has_extension (const char *filename, const char *extension);


/// Returns the MRL of a thumbnail for the specified MRL.
/**
 * The returned string must be freed.
 */
char *get_thumbnail (const char *mrl);


/// Recursively creates a directory.
bool mkdir_recursive (const char *mrl, mode_t mode);


/// Returns TRUE if the passed MRL exists.
bool file_exists (const char *mrl);


/// Escapes a filename to be used in a URI.
/**
 * The returned string must be freed.
 */
char *filename_escape_to_uri (const char *filename);


/// Escapes a filename to be used in a shell.
/**
 * The returned string must be freed.
 */
char *filename_escape_for_shell (const char *filename);


/// Read the complete file into a buffer which is returned.
/**
 * The returned buffer must be freed.
 */
char *read_entire_file (const char *mrl, int *size);


/// Returns TRUE if the MRL is an oxine playlist.
bool is_playlist_oxp (const char *mrl);


/// Returns TRUE if the MRL is an M3U playlist.
bool is_playlist_m3u (const char *mrl);


/// Returns TRUE if the MRL is an PLS playlist.
bool is_playlist_pls (const char *mrl);


/// Returns TRUE if the MRL is a directory.
bool is_directory (const char *mrl);


/// Returns TRUE if the MRL is DVD copied to disc.
bool is_directory_dvd (const char *mrl);


/// Returns TRUE if the MRL is VCD copied to disc.
bool is_directory_vcd (const char *mrl);


/// Returns TRUE if the MRL is an ISO image.
bool is_iso_image (const char *mrl);


/// Returns TRUE if the MRL is a hidden file.
bool is_file_hidden (const char *mrl);


/// Returns TRUE if the MRL is an image file.
bool is_file_image (const char *mrl);


#endif /* HAVE_UTILS_FILES_H */
