
/*
 * Copyright (C) 2004-2005 Maximilian Schwerin
 *
 * This file is part of oxine a free media player.
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.
 *
 * $Id: list.h 2411 2007-07-03 07:49:11Z mschwerin $
 *
 */

#ifndef HAVE_LIST_H
#define HAVE_LIST_H

#include "types.h"


/**
 * Callback functions of this type are used to determin if two entries of a
 * list must be swapped.
 *
 * @param data1                 A pointer to user data.
 * @param data2                 Another pointer to user data.
 */
typedef bool (*l_swap_cb_t) (void *data1, void *data2);


/**
 * Callback functions of this type are used to free the user memory of a
 * list entry.
 *
 * @param data                  A pointer to user data.
 */
typedef void (*l_free_cb_t) (void *data);


/**
 * The list.
 */
typedef struct l_list_s l_list_t;


/**
 * Creates and returns a new empty list.
 */
l_list_t *l_list_new (void);


/**
 * Clears and frees the list.
 *
 * @param list                  This is a pointer to the list.
 * @param free_cb               This is a pointer to a callback function that
 *                              can be used to free the user memory of a list
 *                              entry. This may be NULL.
 */
void l_list_free (l_list_t * list, l_free_cb_t free_cb);


/**
 * Removes all entries from the list. 
 *
 * @param list                  This is a pointer to the list.
 * @param free_cb               This is a pointer to a callback function that
 *                              can be used to free the user memory of a list
 *                              entry. This may be NULL.
 */
void l_list_clear (l_list_t * list, l_free_cb_t free_cb);


/**
 * Sorts the list.
 *
 * @param list                  This is a pointer to the list.
 * @param swap_cb               This is a pointer to a callback function that
 *                              is used to determin if two entries must be
 *                              swapped. This may not be NULL.
 */
void l_list_sort (l_list_t * list, l_swap_cb_t swap_cb);


/**
 * Adds list l2 to the end of list l1 and frees l2.
 *
 * @param l1                    This is a pointer to the list.
 * @param l2                    This list is concatenated to the end of
 *                              <code>l1</code> and freed afterwards.
 */
void l_list_concat (l_list_t * l1, l_list_t * l2);


/**
 * Inserts a new entry with data at the given position.
 *
 * @param list                  This is a pointer to the list.
 * @param position              The position to insert the new entry.
 * @param data                  A pointer to user data.
 *
 * @return                      The user data.
 */
void *l_list_insert (l_list_t * list, int position, void *data);


/**
 * Prepends a new entry to the beginning of the list.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer to user data.
 *
 * @return                      The user data.
 */
void *l_list_prepend (l_list_t * list, void *data);


/**
 * Appends a new entry to the end of the list.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer to user data.
 *
 * @return                      The user data.
 */
void *l_list_append (l_list_t * list, void *data);


/**
 * Removes the entry identified by <code>data</code> from the list.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer to user data.
 *
 * @return                      The user data of the removed entry.
 */
void *l_list_remove (l_list_t * list, void *data);


/**
 * Returns the user data of the first entry of the list.
 *
 * @param list                  This is a pointer to the list.
 */
void *l_list_first (l_list_t * list);


/**
 * Returns the user data of the last entry of the list.
 *
 * @param list                  This is a pointer to the list.
 */
void *l_list_last (l_list_t * list);


/**
 * Returns the user data of the next entry. The current entry is identified
 * by <code>data</code>.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer the user data of the previous item.
 */
void *l_list_next (l_list_t * list, void *data);


/**
 * Returns the user data of the previous entry. The current entry is
 * identified by <code>data</code>.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer the user data of the next item.
 */
void *l_list_prev (l_list_t * list, void *data);


/**
 * Returns the number of entries in the list.
 *
 * @param list                  This is a pointer to the list.
 */
int l_list_length (l_list_t * list);


/**
 * Moves an entry one position towards the front of the list. The entry is
 * identified by <code>data</code>.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer to user data.
 */
void l_list_move_up (l_list_t * list, void *data);


/**
 * Moves an entry one position towards the end of the list. The entry is
 * identified by <code>data</code>.
 *
 * @param list                  This is a pointer to the list.
 * @param data                  A pointer to user data.
 */
void l_list_move_down (l_list_t * list, void *data);


#endif /* HAVE_LIST_H */
