//                                               -*- C++ -*-
/**
 *  @file  NormalFactory.cxx
 *  @brief Factory for Normal distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: NormalFactory.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>
#include "NormalFactory.hxx"
#include "Normal.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      typedef Normal::NumericalPoint    NumericalPoint;
      typedef Normal::CorrelationMatrix CorrelationMatrix;
      typedef Normal::CovarianceMatrix  CovarianceMatrix;

      /* Default constructor */
      NormalFactory::NormalFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      NormalFactory * NormalFactory::clone() const
      {
	return new NormalFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      NormalFactory::DistributionImplementation * NormalFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Normal distribution from an empty sample";
	const NumericalPoint mean(sample.computeMean());
	const CovarianceMatrix covariance(sample.computeCovariance());
	if (!covariance.isPositiveDefinite()) throw InternalException(HERE) << "Error: can build a Normal distribution only from a sample with a positive definite covariance matrix, here covariance=" << covariance;
	return Normal(mean, covariance).clone();
      }

      NormalFactory::DistributionImplementation * NormalFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  Normal distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Normal distribution from the given parameters";
	  }
      }

      NormalFactory::DistributionImplementation * NormalFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return Normal().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
