//                                               -*- C++ -*-
/**
 *  @file  LaplaceFactory.cxx
 *  @brief Factory for Laplace distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2007-05-10 16:43:31 +0200 (jeu. 10 mai 2007) $
 *  Id:      $Id: LaplaceFactory.cxx 434 2007-05-10 14:43:31Z dutka $
 */
#include "LaplaceFactory.hxx"
#include "Laplace.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {

      /* Default constructor */
      LaplaceFactory::LaplaceFactory():
	DistributionImplementationFactory()
      {
	// Nothing to do
      }

      /* Virtual constructor */
      LaplaceFactory * LaplaceFactory::clone() const
      {
	return new LaplaceFactory(*this);
      }

      /* Here is the interface that all derived class must implement */

      LaplaceFactory::DistributionImplementation * LaplaceFactory::buildImplementation(const NumericalSample & sample) const /* throw(InvalidArgumentException, InternalException) */
      {
	if (sample.getSize() == 0) throw InvalidArgumentException(HERE) << "Error: cannot build a Laplace distribution from an empty sample";
	if (sample.getDimension() != 1) throw InvalidArgumentException(HERE) << "Error: can build an Laplace distribution only from a sample of dimension 1, here dimension=" << sample.getDimension();
	const UnsignedLong size(sample.getSize());
	const NumericalScalar mu(sample.computeMedianPerComponent()[0]);
	NumericalScalar lambda(0.0);
	for (UnsignedLong i = 0; i < size; ++i) lambda += fabs(sample[i][0] - mu);
	return Laplace(lambda / size, mu).clone();
      }

      LaplaceFactory::DistributionImplementation * LaplaceFactory::buildImplementation(const NumericalPointCollection & parameters) const /* throw(InvalidArgumentException, InternalException) */
      {
	try {
	  Laplace distribution;
	  distribution.setParametersCollection(parameters);
	  return distribution.clone();
	}
	catch (InvalidArgumentException & ex)
	  {
	    throw InternalException(HERE) << "Error: cannot build a Laplace distribution from the given parameters";
	  }
      }

      LaplaceFactory::DistributionImplementation * LaplaceFactory::buildImplementation() const /* throw(InvalidArgumentException, InternalException) */
      {
	return Laplace().clone();
      }


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
