//                                               -*- C++ -*-
/**
 *  @file  Histogram.hxx
 *  @brief The Histogram distribution
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: Histogram.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_HISTOGRAM_HXX
#define OPENTURNS_HISTOGRAM_HXX

#include "OTprivate.hxx"
#include "NonEllipticalDistribution.hxx"
#include "Collection.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Distribution {


      /**
       * @class Histogram
       *
       * The Histogram distribution.
       */
      class Histogram
        : public Model::NonEllipticalDistribution
      {
        CLASSNAME;
      public:

#ifndef SWIG
        // TODO : To be documented
        /** A couple (? , ?) */
        struct Pair {
          NumericalScalar h_, l_, s_;
          Pair() : h_(), l_(), s_() {}
          Pair(NumericalScalar h, NumericalScalar l) : h_(h), l_(l), s_(h * l) {}
          ~Pair() {}
	  Bool operator == (const Pair & other) const { return (h_ == other.h_) && (l_ == other.l_) && (s_ == other.s_); }
        }; /* end struct Pair */

#else
        typedef OT::Uncertainty::Distribution::HistogramPair Pair;
#endif

        typedef Base::Common::InvalidArgumentException               InvalidArgumentException;
        typedef Model::NonEllipticalDistribution                     NonEllipticalDistribution;  // required by SWIG
        typedef NonEllipticalDistribution::NumericalPoint            NumericalPoint;
        typedef NonEllipticalDistribution::NumericalSample           NumericalSample;
        typedef Base::Type::Collection<Pair>                         PairCollection;
        typedef NonEllipticalDistribution::CovarianceMatrix          CovarianceMatrix;
        typedef NonEllipticalDistribution::NumericalPointWithDescriptionCollection  NumericalPointWithDescriptionCollection;
        typedef NonEllipticalDistribution::NotDefinedException       NotDefinedException;


        /** Default constructor */
        Histogram();

        /** Parameters constructor */
        Histogram(const NumericalScalar first,
                  const PairCollection & collection);


        /** Comparison operator */
        Bool operator ==(const Histogram & other) const;

        /** String converter */
	String __repr__() const;



        /* Interface inherited from Distribution */

        /** Virtual constructor */
        virtual Histogram * clone() const;

        /** Get one realization of the Histogram distribution */
        NumericalPoint getRealization() const;

        /** Get the DDF of the Histogram distribution */
        using NonEllipticalDistribution::computeDDF;
        NumericalPoint computeDDF(const NumericalPoint & point) const;

        /** Get the PDF of the Histogram distribution */
        using NonEllipticalDistribution::computePDF;
        NumericalScalar computePDF(const NumericalPoint & point) const;

        /** Get the CDF of the Histogram distribution */
        using NonEllipticalDistribution::computeCDF;
        NumericalScalar computeCDF(const NumericalPoint & point, const Bool tail = false) const;

        /** Get the PDFGradient of the Histogram distribution */
        NumericalPoint computePDFGradient(const NumericalPoint & point) const;

        /** Get the CDFGradient of the Histogram distribution */
        NumericalPoint computeCDFGradient(const NumericalPoint & point) const;

        /** Get the quantile of the Histogram distribution */
        NumericalScalar computeScalarQuantile(const NumericalScalar prob,
					      const Bool tail = false,
                                              const NumericalScalar precision = DefaultQuantileEpsilon) const;

        /** Get the mean of the distribution */
        NumericalPoint getMean() const /* throw(NotDefinedException) */;

        /** Get the covariance of the distribution */
        CovarianceMatrix getCovariance() const /* throw(NotDefinedException) */;

        /** Parameters value and description accessor */
        NumericalPointWithDescriptionCollection getParametersCollection() const;

        /* Interface specific to Histogram */

        /** First point accessor */
        void setFirst(const NumericalScalar first);
        NumericalScalar getFirst() const;

        /** Collection accessor */
        void setPairCollection(const PairCollection & collection);
        PairCollection getPairCollection() const;



      protected:


      private:

        /** Compute the numerical range of the distribution given the parameters values */
        void computeRange();

        /** The first point of the collection */
        NumericalScalar firstX_;

        /** The collection of pair describing the distribution */
        PairCollection collection_;

        /** The cumulated bin widths */
        NumericalPoint cumulatedWidth_;

        /** The cumulated bin surface */
        NumericalPoint cumulatedSurface_;

        /** The total surface */
        NumericalScalar surface_;

      }; /* class Histogram */



      typedef OpenTURNS::Uncertainty::Distribution::Histogram::Pair           HistogramPair;
      typedef OpenTURNS::Uncertainty::Distribution::Histogram::PairCollection HistogramPairCollection;

#ifndef SWIG
      /** A stream operator */
      inline std::ostream & operator << (std::ostream & os, const Histogram::Pair & pair)
      {
        return os << "(" << pair.h_ << "," << pair.l_ << ")";
      }
      inline Base::Common::OStream & operator << (Base::Common::OStream & OS, const Histogram::Pair & pair)
      {
	return OS << "(" << pair.h_ << "," << pair.l_ << ")";
      }
#endif


    } /* namespace Distribution */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */

#endif /* OPENTURNS_HISTOGRAM_HXX */
