//                                               -*- C++ -*-
/**
 *  @file  InverseNatafEllipticalCopulaEvaluation.cxx
 *  @brief Class for the InverseNataf transformation evaluation for elliptical
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: InverseNatafEllipticalCopulaEvaluation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include "InverseNatafEllipticalCopulaEvaluation.hxx"

namespace OpenTURNS {

  namespace Uncertainty {

    namespace Algorithm {

      /*
       * @class InverseNatafEllipticalCopulaEvaluation
       *
       * This class offers an interface for the InverseNataf function for elliptical copula
       */

      CLASSNAMEINIT(InverseNatafEllipticalCopulaEvaluation);

      /* Parameter constructor */
      InverseNatafEllipticalCopulaEvaluation::InverseNatafEllipticalCopulaEvaluation(const Distribution & standardDistribution,
										     const SquareMatrix & cholesky):
	NumericalMathEvaluationImplementation(),
	standardDistribution_(standardDistribution),
	cholesky_(cholesky)
      {
	Description description;
	for (UnsignedLong i = 0; i < cholesky_.getNbColumns(); ++i) description.add(OSS() << "x" << i);
	for (UnsignedLong i = 0; i < cholesky_.getNbRows(); ++i) description.add(OSS() << "y" << i);
	setDescription(description);
      }

      /* Virtual constructor */
      InverseNatafEllipticalCopulaEvaluation * InverseNatafEllipticalCopulaEvaluation::clone() const
      {
	return new InverseNatafEllipticalCopulaEvaluation(*this);
      }

      /* String converter */
      String InverseNatafEllipticalCopulaEvaluation::__repr__() const
      {
	OSS oss;
	oss << "class=" << InverseNatafEllipticalCopulaEvaluation::GetClassName()
	    << " description=" << getDescription()
            << " standardDistribution=" << standardDistribution_
            << " cholesky=" << cholesky_;
                   
	return oss;
      }

      /*
       * Evaluation
       * This function transforms a standard elliptical distribution into an elliptical copula of the same kind with a correlation matrix R:
       * Z(u) = L.u, where L is the Cholesky factor of R: L.L^t = R, L is lower triangular
       * Si(u) = F(Zi), where F is the CDF of the standard elliptical distribution
       */
      InverseNatafEllipticalCopulaEvaluation::NumericalPoint InverseNatafEllipticalCopulaEvaluation::operator () (const NumericalPoint & in) const
	/* throw (InvalidArgumentException, InternalException) */
      {
	const UnsignedLong dimension(getInputDimension());
	// First, correlate the components
	NumericalPoint result(cholesky_ * in);
	const Distribution standardMarginal(standardDistribution_.getMarginal(0));
	// Second, apply the commmon marginal distribution
	for (UnsignedLong i = 0; i < dimension; ++i) result[i] = standardMarginal.computeCDF(NumericalPoint(1, result[i]));
	return result;
      }

      /* Gradient according to the marginal parameters. Currently, the dependence parameter are not taken into account. */

      InverseNatafEllipticalCopulaEvaluation::Matrix InverseNatafEllipticalCopulaEvaluation::parametersGradient(const NumericalPoint & in) const
      {
	return Matrix(0, getInputDimension());
      }
	  
      /* Accessor for input point dimension */
      UnsignedLong InverseNatafEllipticalCopulaEvaluation::getInputDimension() const
	/* throw(InternalException) */
      {
	return cholesky_.getNbColumns();
      }

      /* Accessor for output point dimension */
      UnsignedLong InverseNatafEllipticalCopulaEvaluation::getOutputDimension() const
	/* throw(InternalException) */
      {
	return cholesky_.getNbRows();
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
