//                                               -*- C++ -*-
/**
 *  @file  LHSExperiment.cxx
 *  @brief Abstract top-level view of an LHSExperiment plane
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2008-06-26 13:50:17 +0200 (jeu, 26 jun 2008) $
 *  Id:      $Id: LHSExperiment.cxx 862 2008-06-26 11:50:17Z dutka $
 */
#include "LHSExperiment.hxx"
#include "RandomGenerator.hxx"
#include "Collection.hxx"
#include "NumericalPoint.hxx"
#include "Exception.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      typedef Base::Stat::RandomGenerator                         RandomGenerator;
      typedef Base::Type::Collection<LHSExperiment::Distribution> DistributionCollection;
      typedef Base::Type::NumericalPoint                          NumericalPoint;
      typedef Base::Common::InvalidArgumentException              InvalidArgumentException;

      CLASSNAMEINIT(LHSExperiment);

      /* Default constructor */
      LHSExperiment::LHSExperiment(const String & name):
        WeightedExperiment(name),
        shuffle_(0, 0),
        isAlreadyComputedShuffle_(false)
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      LHSExperiment::LHSExperiment(const UnsignedLong size,
                                   const String & name):
        WeightedExperiment(size, name),
        shuffle_(0, 0),
        isAlreadyComputedShuffle_(false)
      {
        // Nothing to do
      }

      /* Constructor with parameters */
      LHSExperiment::LHSExperiment(const Distribution & distribution,
                                   const UnsignedLong size,
                                   const String & name):
        WeightedExperiment(distribution, size, name),
        shuffle_(0, 0),
        isAlreadyComputedShuffle_(false)
      {
        // Check if the distribution has an independent copula
	setDistribution(distribution);
      }

      /* Virtual constructor */
      LHSExperiment * LHSExperiment::clone() const
      {
        return new LHSExperiment(*this);
      }

      /* String converter */
      String LHSExperiment::__repr__() const
      {
        OSS oss;
        oss << "class=" << GetClassName()
            << " name=" << getName()
            << " distribution=" << distribution_
            << " size=" << size_;
        return oss;
      }

      /* Sample generation */
      LHSExperiment::NumericalSample LHSExperiment::generate()
      {
        if (!isAlreadyComputedShuffle_) computeShuffle();
        const UnsignedLong dimension(distribution_.getDimension());
        NumericalSample sample(size_, dimension);
        DistributionCollection marginals(dimension);
        for (UnsignedLong i = 0; i < dimension; ++i) marginals[i] = distribution_.getMarginal(i);
        for(UnsignedLong index = 0; index < size_; ++index)
          {
            const NumericalPoint u(RandomGenerator::Generate(dimension));
            for(UnsignedLong component = 0; component < dimension; ++component)
              {
                const NumericalScalar xi((shuffle_(component, index) + u[component]) / size_);
                sample[index][component] = marginals[component].computeQuantile(xi)[0];
              }
          }
        return sample;
      }

      /* Sample generation */
      LHSExperiment::NumericalSample LHSExperiment::generate(NumericalPoint & weights)
      {
	weights = weights_;
        return generate();
      }

      /* Shuffle accessor */
      LHSExperiment::Matrix LHSExperiment::getShuffle() const
      {
        if (!isAlreadyComputedShuffle_) computeShuffle();
        return shuffle_;
      }

      /* Shuffle the cells. */
      void LHSExperiment::computeShuffle() const
      {
        const UnsignedLong dimension(distribution_.getDimension());
        // Preallocate the shuffle
        shuffle_ = Matrix(dimension, size_);
        // Initialize the cells shuffle to the identity
        for (UnsignedLong j = 0; j < size_; ++j)
          {
            for (UnsignedLong i = 0; i < dimension; ++i)
              {
                shuffle_(i, j) = j;
              }
          }
        // Perform the shuffle. We use the same algorithm than the one used in the STL or in GSL
        for (UnsignedLong index = 0; index < size_ - 1; ++index)
          {
            // For each component of the current realization, compute its new position
            RandomGenerator::UnsignedLongCollection u(RandomGenerator::IntegerGenerate(dimension, size_ - index));
            // Then swap the corresponding components
            for (UnsignedLong component = 0; component < dimension; ++component)
              {
                UnsignedLong newPosition(index + u[component]);
                NumericalScalar newValue(shuffle_(component, newPosition));
                shuffle_(component, newPosition) = shuffle_(component, index);
                shuffle_(component, index) = newValue;
              }
          }
        isAlreadyComputedShuffle_ = true;
      }

      /* Distribution accessor */
      void LHSExperiment::setDistribution(const Distribution & distribution)
      {
	if (!distribution.hasIndependentCopula()) throw InvalidArgumentException(HERE) << "Error: cannot use the LHS experiment with a non-independent copula.";
	WeightedExperiment::setDistribution(distribution);
      }

    } /* namespace Algorithm */
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
