//                                               -*- C++ -*-
/**
 *  @file  SORMResult.cxx
 *  @brief SORMResult implements the results obtained from the Second Order Reliability Method
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: SORMResult.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cmath>

#include "SORM.hxx"
#include "StandardEvent.hxx"
#include "NumericalMathFunction.hxx"
#include "SymmetricTensor.hxx"
#include "IdentityMatrix.hxx"
#include "Matrix.hxx"
#include "Log.hxx"
#include "Normal.hxx"
#include "StandardEvent.hxx"
#include "PersistentObjectFactory.hxx"

namespace OpenTURNS
{

  namespace Uncertainty
  {

    namespace Algorithm
    {

      typedef Base::Type::SymmetricTensor       SymmetricTensor;
      typedef Base::Type::IdentityMatrix        IdentityMatrix;
      typedef Base::Type::Matrix                Matrix;
      typedef Base::Type::SymmetricMatrix       SymmetricMatrix;
      typedef Base::Func::NumericalMathFunction NumericalMathFunction;
      typedef Base::Common::Log                 Log;
      typedef Distribution::Normal              Normal;
      typedef Model::StandardEvent              StandardEvent;

      CLASSNAMEINIT(SORMResult);

      static Base::Common::Factory<SORMResult> RegisteredFactory("SORMResult");

      /*
       * @brief  Standard constructor: the class is defined by an optimisation algorithm, a failure event and a physical starting point
       */
      SORMResult::SORMResult(const NumericalPoint & standardSpaceDesignPoint,
			     const Event & limitStateVariable,
			     const Bool isStandardPointOriginInFailureSpace,
			     const String & name):
	AnalyticalResult(standardSpaceDesignPoint, limitStateVariable, isStandardPointOriginInFailureSpace, name),
	standardDistribution_(limitStateVariable.getImplementation()->getAntecedent().getImplementation()->getDistribution().getStandardDistribution()),
	standardMarginal_(standardDistribution_.getMarginal(0))
      {
	/* get the physical Limite State Function */
	
      	const NumericalMathFunction limitStateFunction(StandardEvent(limitStateVariable).getImplementation()->getFunction());
	/* compute its gradient */
	const Matrix gradient(limitStateFunction.gradient(getStandardSpaceDesignPoint()));
	/* Get the first column */
	gradientLimitStateFunction_ = gradient * NumericalPoint(1, 1.0);
	/* compute its hessian */
	const SymmetricTensor hessian(limitStateFunction.hessian(getStandardSpaceDesignPoint()));
	/* Get the first sheet */
	hessianLimitStateFunction_ = SquareMatrix(hessian.getNbRows());
	for (UnsignedLong i = 0; i < hessian.getNbRows(); i++)
	  {
	    for (UnsignedLong j = 0; j < hessian.getNbColumns(); j++)
	      {
		hessianLimitStateFunction_(i, j) = hessian(i, j, 0);
	      }
	  }
	/* compute its curvatures and the several formulas for the SORM probability */
	computeCurvatures();
	eventProbabilityBreitung_ = -1.0;
	try{
	  computeEventProbabilityBreitung();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.__repr__());
	}


	eventProbabilityHohenBichler_ = -1.0;
	try{
	  computeEventProbabilityHohenBichler();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.__repr__());
	}


	eventProbabilityTvedt_ = -1.0;
	try{
	  computeEventProbabilityTvedt();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.__repr__());
	}


	try{
	  computeGeneralisedReliabilityIndex();
	}
	catch (NotDefinedException & ex) {
	  Log::Info(ex.__repr__());
	}
      } // end SORMResult::Result

      /* Default constructor */
      SORMResult::SORMResult():
	AnalyticalResult(),
	hessianLimitStateFunction_(),
	gradientLimitStateFunction_(),
	sortedCurvatures_(),
	eventProbabilityBreitung_(),
	eventProbabilityHohenBichler_(),
	eventProbabilityTvedt_(),
	generalisedReliabilityIndexBreitung_(),
	generalisedReliabilityIndexHohenBichler_(),
	generalisedReliabilityIndexTvedt_(),
	standardDistribution_(Normal(1)),
	standardMarginal_(Normal(1))
      {
	// Nothing to do
      }

      /* Virtual constructor */
      SORMResult * SORMResult::clone() const
      {
	return new SORMResult(*this);
      }

      /* The function that actually evaluates the event probability with SORM Breitung approximation */
      void SORMResult::computeEventProbabilityBreitung()
      /* throw(NotDefinedException) */
      {
	const NumericalScalar beta(getHasoferReliabilityIndex());
	const NumericalPoint minusBeta(1, -beta);
	const NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	/* test if all curvatures verifie 1 + beta * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + beta * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate Breitung SORM probability, one of the curvatures is < -1/beta";
	  }

	/* PBreitung = E(-beta)/Prod(sqrt(1+beta*curvature[i])) */
	eventProbabilityBreitung_ = standardCDFBeta;
	for (UnsignedLong index = 0 ; index < sortedCurvatures_.getDimension(); index ++)
	  {
	    eventProbabilityBreitung_ /= sqrt(1.0 + beta*sortedCurvatures_[index]);
	  }
	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace())
	  {
	    eventProbabilityBreitung_ = 1.0 - eventProbabilityBreitung_;
	  }
      } // end SORMResult::computeEventProbabilityBreitung

	/* EventProbabilityBreitung accessor */
      NumericalScalar SORMResult::getEventProbabilityBreitung() const
      { 
	return eventProbabilityBreitung_;
      }

      /* The function that actually evaluates the curvatures of the standard limite state function at the standard design point */
      void SORMResult::computeCurvatures()
      {
	/* see Mefisto v3.2 documentation */
	/* we calculate the main curvatures */
	const NumericalScalar inverseGradientNorm(1.0 / gradientLimitStateFunction_.norm());
	const NumericalPoint unitGradientLimitStateFunction(inverseGradientNorm * gradientLimitStateFunction_);
	const UnsignedLong dimension(getStandardSpaceDesignPoint().getDimension());
	SquareMatrix kroneckerUnitGradientLimitStateFunction(dimension);
	for (UnsignedLong i = 0; i < dimension; i++)
	  {
	    for (UnsignedLong j = 0; j < dimension; j++)
	      {
		kroneckerUnitGradientLimitStateFunction(i, j) = unitGradientLimitStateFunction[i] * unitGradientLimitStateFunction[j];
	      }
	  } 
	/* W = (uGrad.uGrad^t -Id) * Hess(g) */
	const SquareMatrix Id = IdentityMatrix(dimension);
	const SquareMatrix::NumericalComplexCollection eigenValues(((kroneckerUnitGradientLimitStateFunction - Id) * hessianLimitStateFunction_).computeEigenValues());
	NumericalPoint realEigenValues(dimension);
	for (UnsignedLong i = 0; i < dimension; ++i)
	  {
	    realEigenValues[i] = eigenValues[i].real();
	  }

	/* The curvatures are proportional to the eigenvalues of W
	   If the normal of the boundary of the failure domain points to the origin at the design point, then we change
           the sign of the curvatures. It insure that a convexe failure domain will have positive curvatures */
	sortedCurvatures_ = ((NumericalPoint::dot(gradientLimitStateFunction_, getStandardSpaceDesignPoint()) > 0.0 ? 1.0 : -1.0) * inverseGradientNorm) * realEigenValues;
	/* we sort the curvatures with increasing order */
	std::sort(sortedCurvatures_.begin(), sortedCurvatures_.end());
      } // end SORMResult::computeCurvatures

      /* SortedCurvatures accessor */
      SORM::NumericalPoint SORMResult::getSortedCurvatures() const
      {
	return sortedCurvatures_;
      }

      /* The function that actually evaluates the event probability with SORM HohenBichler approximation */
      void SORMResult::computeEventProbabilityHohenBichler()
      /* throw(NotDefinedException) */
      {    
	/* this formula is valid only for standard distribution with independent components */
	if (!standardDistribution_.hasIndependentCopula()) throw NotDefinedException(HERE) << "Error: impossible to calculate HohenBichler SORM probability for standard distributions with non independent components.";

	const NumericalPoint minusBeta(1, -getHasoferReliabilityIndex());
	const NumericalScalar standardPDFBeta(standardMarginal_.computePDF(minusBeta));
	const NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	const NumericalScalar rho(standardPDFBeta / standardCDFBeta);

	/* test if all curvatures verifie 1 + rho * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + rho * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate HohenBichler SORM probability, one of the curvatures is < -1/rho";
	  }

	/* Phohenbichler = Phi(-beta)/Prod(sqrt(1+rho*curvature[i])) */
	eventProbabilityHohenBichler_ = standardCDFBeta;
	for (UnsignedLong index = 0 ; index < sortedCurvatures_.getDimension(); index ++)
	  {
	    eventProbabilityHohenBichler_ /= sqrt(1.0 + rho*sortedCurvatures_[index]);
	  }

	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace()) 
	  {
	    eventProbabilityHohenBichler_ = 1.0 - eventProbabilityHohenBichler_;
	  }
      } // SORMResult::computeEventProbabilityHohenBichler

	/* EventProbability HohenBichleraccessor */
      NumericalScalar SORMResult::getEventProbabilityHohenBichler() const
      { 
	return eventProbabilityHohenBichler_;
      }

      /* The function that actually evaluates the event probability with SORM Tvedtapproximation */
      void SORMResult::computeEventProbabilityTvedt()
      /* throw(NotDefinedException) */
      {    
	/* this formula is valid only for standard distribution with independent components */
	if (!standardDistribution_.hasIndependentCopula()) throw NotDefinedException(HERE) << "Error: impossible to calculate Tvedt SORM probability for standard distributions with non independent components.";

	const NumericalScalar beta(getHasoferReliabilityIndex());
	const NumericalPoint minusBeta(1, -beta);

	/* test if all curvatures verifie 1 + (beta+1) * curvature  > 0 */
	/* curvatures are sorted with increasing order and stocked in the attribute SortedCurvatures */
	if (1.0 + (1 + beta) * sortedCurvatures_[0] < 0) 
	  {
	    throw NotDefinedException(HERE) << "Error: impossible to calculate Tvedt SORM probability, one of the curvatures is < -1/(1+beta)";
	  }

	const NumericalScalar standardPDFBeta(standardMarginal_.computePDF(minusBeta));
	const NumericalScalar standardCDFBeta(standardMarginal_.computeCDF(minusBeta));

	/* compute the first term A1 */

	NumericalScalar prod1(1.0);
	const UnsignedLong dimension(sortedCurvatures_.getDimension());
	for (UnsignedLong index = 0 ; index < dimension; index ++)
	  {
	    prod1 /= sqrt(1.0 + beta * sortedCurvatures_[index]);
	  }
	const NumericalScalar termA1(standardCDFBeta * prod1);

	/* compute the second term A2 */

	const NumericalScalar rho(beta * standardCDFBeta - standardPDFBeta);

	NumericalScalar prod2(1.0);
	for (UnsignedLong index = 0; index < dimension; index ++)
	  {
	    prod2 /= sqrt(1.0 + (1.0 + beta) * sortedCurvatures_[index]);
	  }

	const NumericalScalar termA2(rho * (prod1 - prod2));

	/* compute the second term A3 */

	NumericalComplex complexProd3(1.0, 0.0);
	NumericalComplex iPlusBeta(beta, 1.0);

	for (UnsignedLong index = 0; index < dimension; index ++)
	  {
	    complexProd3 /= sqrt(1.0 + iPlusBeta * sortedCurvatures_[index]);
	  }
	const NumericalScalar termA3((beta + 1.0) * rho * (prod1 - complexProd3.real()));

	/* compute tvedt probability */

	eventProbabilityTvedt_ = termA1 + termA2 + termA3;
	/* if the Standard Point Origin is in the failure space, take the complementry probability */
	if (getIsStandardPointOriginInFailureSpace()) 
	  {
	    eventProbabilityTvedt_ = 1.0 - eventProbabilityTvedt_;
	  }
      } // end SORMResult::computeEventProbabilityTvedt

	/* EventProbability accessor */
      NumericalScalar SORMResult::getEventProbabilityTvedt() const
      { 
	return eventProbabilityTvedt_;
      }

      /* The function that actually evaluates the generalised reliability index with SORM BreitungHB and Tvedt  approximations */
      void SORMResult::computeGeneralisedReliabilityIndex()
      {
	/* evaluate the GeneralisedReliabilityIndex */
	//* GeneralisedReliabilityIndex is defined by : - Inverse standard marginal CDF (eventProbability) in usual case or : + Inverse standard marginal CDF (eventProbability) in other case */
	NumericalScalar sign(1.0);
	if (!getIsStandardPointOriginInFailureSpace()) { // StandardPointOriginInFailureSpace is FALSE : usual case
	  sign = -1.0;
	}

	/* generalised reliability index with SORM Breitung approximation */
	generalisedReliabilityIndexBreitung_ = sign * standardMarginal_.computeQuantile(eventProbabilityBreitung_)[0];

	/* generalised reliability index with SORM HohenBichler approximation */
	generalisedReliabilityIndexHohenBichler_ = 0.0;
	if (eventProbabilityHohenBichler_ != -1.0)
	  {
	    generalisedReliabilityIndexHohenBichler_ = sign * standardMarginal_.computeQuantile(eventProbabilityHohenBichler_)[0];
	  }

	/* generalised reliability index with SORM Tvedt approximation */
	generalisedReliabilityIndexTvedt_ = 0.0;
	if (eventProbabilityTvedt_ != -1.0)
	  {
	    generalisedReliabilityIndexTvedt_ = sign * standardMarginal_.computeQuantile(eventProbabilityTvedt_)[0];
	  }

      } // end SORMResult::computeGeneralisedReliabilityIndex

	/* GeneralisedReliabilityIndexBreitung accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexBreitung() const
      {
	return generalisedReliabilityIndexBreitung_;
      }

      /* GeneralisedReliabilityIndex accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexHohenBichler() const
      {
	return generalisedReliabilityIndexHohenBichler_;
      }

      /* GeneralisedReliabilityIndex accessor */
      NumericalScalar SORMResult::getGeneralisedReliabilityIndexTvedt() const
      {
	return generalisedReliabilityIndexTvedt_;
      }

      /* String converter */ 
      String SORMResult::__repr__() const 
      {
	OSS oss;
	oss << "class=" << SORMResult::GetClassName()
	    << " " << AnalyticalResult::__repr__()
	    << " sortedCurvatures=" << sortedCurvatures_
	    << " eventProbabilityBreitung=" << eventProbabilityBreitung_
	    << " eventProbabilityHohenBichler=" << eventProbabilityHohenBichler_
	    << " eventProbabilityTvedt=" << eventProbabilityTvedt_
	    << " generalisedReliabilityIndexBreitung=" << generalisedReliabilityIndexBreitung_
	    << " generalisedReliabilityIndexHohenBichler=" << generalisedReliabilityIndexHohenBichler_
	    << " generalisedReliabilityIndexTvedt=" << generalisedReliabilityIndexTvedt_
	    << " gradientLimitStateFunction_=" << gradientLimitStateFunction_
	    << " hessianLimitStateFunction_=" << hessianLimitStateFunction_;
	return oss;
      }

      /* Method save() stores the object through the StorageManager */
      void SORMResult::save(StorageManager::Advocate & adv) const
      {
	AnalyticalResult::save(adv);
	adv.saveAttribute( "hessianLimitStateFunction_", hessianLimitStateFunction_ );
	adv.saveAttribute( "gradientLimitStateFunction_", gradientLimitStateFunction_ );
	adv.saveAttribute( "sortedCurvatures_", sortedCurvatures_ );
	adv.saveAttribute( "eventProbabilityBreitung_", eventProbabilityBreitung_ );
	adv.saveAttribute( "eventProbabilityHohenBichler_", eventProbabilityHohenBichler_ );
	adv.saveAttribute( "eventProbabilityTvedt_", eventProbabilityTvedt_ );
	adv.saveAttribute( "generalisedReliabilityIndexBreitung_", generalisedReliabilityIndexBreitung_ );
	adv.saveAttribute( "generalisedReliabilityIndexHohenBichler_", generalisedReliabilityIndexHohenBichler_ );
	adv.saveAttribute( "generalisedReliabilityIndexTvedt_", generalisedReliabilityIndexTvedt_ );
	adv.saveAttribute( "standardDistribution_", standardDistribution_ );
	adv.saveAttribute( "standardMarginal_", standardMarginal_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void SORMResult::load(StorageManager::Advocate & adv)
      {
	AnalyticalResult::load(adv);
	adv.loadAttribute( "hessianLimitStateFunction_", hessianLimitStateFunction_ );
	adv.loadAttribute( "gradientLimitStateFunction_", gradientLimitStateFunction_ );
	adv.loadAttribute( "sortedCurvatures_", sortedCurvatures_ );
	adv.loadAttribute( "eventProbabilityBreitung_", eventProbabilityBreitung_ );
	adv.loadAttribute( "eventProbabilityHohenBichler_", eventProbabilityHohenBichler_ );
	adv.loadAttribute( "eventProbabilityTvedt_", eventProbabilityTvedt_ );
	adv.loadAttribute( "generalisedReliabilityIndexBreitung_", generalisedReliabilityIndexBreitung_ );
	adv.loadAttribute( "generalisedReliabilityIndexHohenBichler_", generalisedReliabilityIndexHohenBichler_ );
	adv.loadAttribute( "generalisedReliabilityIndexTvedt_", generalisedReliabilityIndexTvedt_ );
	adv.loadAttribute( "standardDistribution_", standardDistribution_ );
	adv.loadAttribute( "standardMarginal_", standardMarginal_ );
      }

    } /* namespace  Algorithm*/
  } /* namespace Uncertainty */
} /* namespace OpenTURNS */
