//                                               -*- C++ -*-
/**
 *  @file  GraphImplementation.cxx
 * @brief Graph implements graphic devices for plotting through R
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: GraphImplementation.cxx 1473 2010-02-04 15:44:49Z dutka $
 */
#include <cstdlib>
#include <fstream>

#include "GraphImplementation.hxx"
#include "Rfunctions.hxx"
#include "Path.hxx"
#include "PersistentObjectFactory.hxx"
#include "ResourceMap.hxx"
#include "Log.hxx"
#include "OTconfig.hxx"
#include "Os.hxx"

namespace OpenTURNS
{

  namespace Base
  {

    namespace Type
    {

      using Graph::Drawable;

      TEMPLATE_CLASSNAMEINIT(PersistentCollection<Drawable>);

      static Common::Factory<PersistentCollection<Drawable> > RegisteredFactory1("PersistentCollection<Drawable>");
    } /* namespace Type */

    namespace Graph
    {
      CLASSNAMEINIT(GraphImplementation);

      static Common::Factory<GraphImplementation> RegisteredFactory("GraphImplementation");

      typedef Common::Path                  Path;
      typedef Common::Log                   Log;
      typedef Common::ResourceMap           ResourceMap;
      typedef Common::Os                    Os;
      typedef Stat::NumericalSample         NumericalSample;
      typedef Common::InvalidRangeException InvalidRangeException;
      typedef Common::NotYetImplementedException NotYetImplementedException;

      const String GraphImplementation::NoSpecifiedLabel               = "";//ResourceMap::GetInstance().get("GraphImplementation-NoSpecifiedLabel");

      const UnsignedLong GraphImplementation::BoundingBoxSize = 4;

      const UnsignedLong GraphImplementation::DefaultWidth             = 640;//strtoul(ResourceMap::GetInstance().get("DrawableImplementation-NoSpecifiedLabel").c_str(), NULL, 0);
      const UnsignedLong GraphImplementation::DefaultHeight            = 480;//strtoul(ResourceMap::GetInstance().get("DrawableImplementation-NoSpecifiedLabel").c_str(), NULL, 0);
      const NumericalScalar GraphImplementation::DefaultLegendFontSize = 1.0;//strtod(ResourceMap::GetInstance().get("DrawableImplementation-NoSpecifiedLabel").c_str(), NULL);
      Bool GraphImplementation::IsFirstInitialization = true;


      GraphImplementation::Description GraphImplementation::ValidLegendPositions;

      /* Initialize valid legend positions */
      void GraphImplementation::initializeValidLegendPositions()
      {
	ValidLegendPositions.setName("ValidLegendPositions");
	ValidLegendPositions.add(NoSpecifiedLabel);
	ValidLegendPositions.add("bottomright");
	ValidLegendPositions.add("bottom");
	ValidLegendPositions.add("bottomleft");
	ValidLegendPositions.add("left");
	ValidLegendPositions.add("topleft");
	ValidLegendPositions.add("topright");
	ValidLegendPositions.add("right");
	ValidLegendPositions.add("center");
      }

      /* Default constructor */
      GraphImplementation::GraphImplementation():
	PersistentObject(NoSpecifiedLabel),
	title_(NoSpecifiedLabel),
	legendPosition_(NoSpecifiedLabel),
	legendFontSize_(DefaultLegendFontSize),
	xTitle_(NoSpecifiedLabel),
	yTitle_(NoSpecifiedLabel),
	showAxes_(true),
	showGrid_(true),
	automaticBoundingBox_(true),
	boundingBox_(BoundingBoxSize),
	drawablesCollection_(0),
	path_(NoSpecifiedLabel),
	fileName_(NoSpecifiedLabel)
      {
	if(IsFirstInitialization){
	  initializeValidLegendPositions();
	  IsFirstInitialization = false;
	}
      }

      /* Constructor with parameters */
      GraphImplementation::GraphImplementation(const String & title,
					       const String & xTitle,
					       const String & yTitle,
					       const Bool showAxes,
					       const String & legendPosition,
                                               const NumericalScalar legendFontSize) /* throw(InvalidArgumentException) */
	: PersistentObject(title),
	  title_(title),
	  legendFontSize_(legendFontSize),
	  xTitle_(xTitle),
	  yTitle_(yTitle),
	  showAxes_(showAxes),
	  showGrid_(true),
	  automaticBoundingBox_(true),
	  boundingBox_(BoundingBoxSize),
	  drawablesCollection_(0),
	  path_(NoSpecifiedLabel),
	  fileName_(NoSpecifiedLabel)
      {
	if(IsFirstInitialization)
	  {
	    initializeValidLegendPositions();
	    IsFirstInitialization = false;
	  }
	// Check if the given legend position is valid

	if(!isValidLegendPosition(legendPosition)) throw InvalidArgumentException(HERE) << "The given legend position = " << legendPosition << " is invalid";

	legendPosition_ = legendPosition;
      }
      
      /* Virtual constructor */
      GraphImplementation * GraphImplementation::clone() const
      {
	return new GraphImplementation(*this);
      }

      /* String converter */
      String GraphImplementation::__repr__() const
      {
	OSS oss;
	oss << "class=" << GraphImplementation::GetClassName()
	    << " name=" << getName()
	    << " title=" << title_
	    << " xTitle=" << xTitle_
	    << " yTitle=" << yTitle_
	    << " axes=" << (showAxes_?"ON":"OFF")
	    << " grid=" << (showGrid_?"ON":"OFF")
	    << " legendposition="<< legendPosition_
	    << " legendFontSize=" << legendFontSize_
	    << " drawables=" << drawablesCollection_
	    << " fileName=" << getFileName();
	return oss;
      }

      /* Adds a drawable instance to the collection of drawables contained in GraphImplementation */
      void GraphImplementation::addDrawable(const Drawable & aDrawable)
      {
	drawablesCollection_.add(aDrawable);
      }

      /** Drawables accessor */
      GraphImplementation::DrawableCollection GraphImplementation::getDrawables() const
      {
	return drawablesCollection_;
      }

      void GraphImplementation::setDrawables(const DrawableCollection & drawableCollection)
      {
	drawablesCollection_ = drawableCollection;
      }

      /** Individual drawable accessor */
      Drawable GraphImplementation::getDrawable(const UnsignedLong index) const
      {
	if (index >= drawablesCollection_.getSize()) throw InvalidRangeException(HERE) << "Error: trying to get a drawable at position " << index << " from a collection of size " << drawablesCollection_.getSize();
	return drawablesCollection_[index];
      }

      void GraphImplementation::setDrawable(const Drawable & drawable, const UnsignedLong index)
      {
	if (index >= drawablesCollection_.getSize()) throw InvalidRangeException(HERE) << "Error: trying to set a drawable at position " << index << " into a collection of size " << drawablesCollection_.getSize();
	drawablesCollection_[index] = drawable;
      }

      /* Hide or show x and y axes */
      void GraphImplementation::setAxes(const Bool showAxes)
      {
	showAxes_ = showAxes;
      }
      
      /* Accessor for showAxes_ */
      Bool GraphImplementation::getAxes() const
      {
	return showAxes_;
      }
	
      /* Hide or show x grid */
      void GraphImplementation::setGrid(const Bool showGrid)
      {
	showGrid_ = showGrid;
      }
      
      /* Accessor for showGrid_ */
      Bool GraphImplementation::getGrid() const
      {
	return showGrid_;
      }
	
      /* Accesor for xTitle */
      String GraphImplementation::getXTitle() const
      {
	return xTitle_;
      }

      /* Accessor for xTitle */
      void GraphImplementation::setXTitle(const String & title)
      {
	xTitle_ = title;
      }
	
      /* Accessor for yTitle */
      String GraphImplementation::getYTitle() const
      {
	return yTitle_;
      }
	
      /* Accessor for yTitle */
      void GraphImplementation::setYTitle(const String & title)
      {
	yTitle_ = title;
      }

      /* Accesor for title */
      String GraphImplementation::getTitle() const
      {
	return title_;
      }
	
      /* Accesor for title */
      void GraphImplementation::setTitle(const String & title)
      {
	title_ = title;
      }

      /* Accessor for path */
      String GraphImplementation::getPath() const
      {
	return path_;
      }
	
      /* Accessor for file name */
      String GraphImplementation::getFileName() const
      {
	return fileName_;
      }
      
      /* Build the R command corresponding to the legend */
      String GraphImplementation::makeRLegendCommand() const
      {
	OSS labels, colors, lines, points, fill;
	labels << "c(";
	colors << "c(";
	lines << "c(";
	points << "c(";
	fill << "c(";
	
	Bool drawLegend = false;

	for(DrawableCollection::const_iterator it = drawablesCollection_.begin(); it != drawablesCollection_.end(); ++it)
	  {
	    if(it->getLegendName() != NoSpecifiedLabel)
	      {
		drawLegend = true;
		labels << "\"" << it->getLegendName() << "\",";

		if(it->getColor() == NoSpecifiedLabel)
		  colors << "NA,";
		else colors << "\"" << it->getColor() << "\",";

		if(it->getFillStyle() == NoSpecifiedLabel)
		  fill << "NA,";
		else fill << "\"" << it->getFillStyle() << "\",";

		if(it->getPointStyle() == NoSpecifiedLabel || it->getFillStyle() != NoSpecifiedLabel) //cannot merge fill and point symbol
		  points << "NA,";
		else points << it->getPointCode(it->getPointStyle())<< ",";

		if(it->getLineStyle() == NoSpecifiedLabel || it->getFillStyle() != NoSpecifiedLabel ) //cannot merge line and fill symbol
		  lines << "NA,";
		else lines << "\"" << it->getLineStyle() << "\",";
	      }
	  }
	if(drawLegend){
	  String labels_str(labels);
	  labels_str.replace(labels_str.length() - 1, 1, ")");
	  
	  String colors_str(colors);
	  colors_str.replace(colors_str.length() - 1, 1, ")");
	  
	  String lines_str(lines);
	  lines_str.replace(lines_str.length() - 1, 1, ")");
	  
	  String points_str(points);
	  points_str.replace(points_str.length() - 1, 1, ")");
	
	  String fill_str(fill);
	  fill_str.replace(fill_str.length() - 1, 1, ")");

	  OSS rCommand;
	  rCommand << "legend(\"" << legendPosition_ << "\","
		   << "legend=" << labels_str << ","
		   << "col=" << colors_str << ","
		   << "lty=" << lines_str << ","
		   << "pch=" << points_str << ","
		   << "fill="<< fill_str << ","
		   << "cex=" << legendFontSize_ << ","
		   << "bg=\"grey90\",merge=TRUE,density=40)";

	  return rCommand;
	}
	
	else return NoSpecifiedLabel;
      }

      /* Get the R command corresponding to the graph */
      String GraphImplementation::getRCommand() const
      {
	return OSS() << makeRHeaderCommand() << makeRCoreCommand();
      }

      /* Make R header commande */
      String GraphImplementation::makeRHeaderCommand() const
      {
	return OSS() << R_LEGEND << "\n" << "\n" << R_PIE << "\n" << "options(digits=17)" << "\n" << "options(warn=-1)" << "\n";
      }

      /* Make R core command */
      String GraphImplementation::makeRCoreCommand() const
      {
	// get the general bounding box
	BoundingBox boundingBox(getBoundingBox());

	//load the R code attached to the general plot
	OSS graphCommand;
	graphCommand << "plot(c(" << boundingBox[0] << "," << boundingBox[1] << "),"
		     << "c(" << boundingBox[2] << "," << boundingBox[3] << "),"
		     << "type=\"n\",main=\"" << title_ << "\",";
	if (showAxes_)
	  {
	    graphCommand << "xlab=\"" << xTitle_ << "\",ylab=\"" << yTitle_ << "\","
	                 << "axes=TRUE";
	  }
	else
	  {
	    graphCommand << "xlab=\"\",ylab=\"\",axes=FALSE";
          }
	if (showGrid_)
	  {
	    graphCommand << ", panel.first=grid()";
	  }
	graphCommand << ", cex.main=2, cex.axis=1.5, cex.lab=1.5)\n";

	// add the R code attached to each drawable
	UnsignedLong drawablesSize(drawablesCollection_.getSize());
	for(UnsignedLong i = 0; i < drawablesSize; ++i)
	  {
	    if (drawablesCollection_[i].getData().getSize() != 0)
	      graphCommand << drawablesCollection_[i].draw() << "\n";
	  }
	// make the legend command
	graphCommand << (legendPosition_ == NoSpecifiedLabel ? NoSpecifiedLabel : makeRLegendCommand());
	return graphCommand;
      }

      /* The method that generates the graphic files */
      void GraphImplementation::draw(const String & path,
				     const String & file,
				     const NumericalScalar width,
				     const NumericalScalar height,
				     const Format format) /* throw(InternalException) */
      {
	path_ = path;
	fileName_ = file;
	Format drawingFormat(format);
	// Check the needed drawingFormat. If it is invalid, ste it to ALL
	if ((drawingFormat != ALL) && (drawingFormat != PNG) && (drawingFormat != EPS) && (drawingFormat != FIG) && (drawingFormat != PDF))
	  {
	    drawingFormat = ALL;
	  }
	OSS rCommand;
 	rCommand << makeRHeaderCommand();
	String rCoreCommand(makeRCoreCommand());
	if ((drawingFormat == ALL) || (drawingFormat == EPS))
	  {
	    rCommand << "postscript(\"" << getPostscript() << "\", horizontal = FALSE, onefile = FALSE, paper = \"special\", height=" << height / 72. << ", width=" << width / 72. << ")" << "\n" << rCoreCommand << "\n" << "dev.off()" << "\n";
	  }
	if ((drawingFormat == ALL) || (drawingFormat == PDF))
	  {
	    rCommand << "pdf(\"" << getPDF() << "\", onefile = FALSE, paper = \"special\", height=" << height / 72. << ", width=" << width / 72. << ")" << "\n" << rCoreCommand << "\n" << "dev.off()" << "\n";
	  }
	if ((drawingFormat == ALL) || (drawingFormat == PNG))
	  {
#ifndef WIN32
	    rCommand << "bitmap(\"" << getBitmap() << "\",type=\"png16m\",height=" << height / 72. << ", width=" << width / 72. << ", res=72.)" << "\n" << rCoreCommand << "\n" << "dev.off()" << "\n";
#else
	    rCommand << "png(\"" << getBitmap() << "\",height=" << height << ", width=" << width << ",res=56.)" << "\n" << rCoreCommand << "\n" << "dev.off()" << "\n";
#endif
	  }
	if ((drawingFormat == ALL) || (drawingFormat == FIG))
	  {
	    rCommand << "xfig(\"" << getVectorial() << "\", horizontal = FALSE, onefile = FALSE, paper = \"A4\", height=" << height / 72. << ", width=" << width / 72. << ")" << "\n" << rCoreCommand << "\n" << "dev.off()" << "\n";
	  }

	//vsystem commands to write R code in a temporary file
	/* using mkstemp non standard C for temporary file generation */
	String temporaryFileName(Path::BuildTemporaryFileName("tmp_graph.R.XXXXXX"));
	std::ofstream cmdFile(temporaryFileName.c_str(), std::ios::out);
	cmdFile << String(rCommand);
	cmdFile.close();
	
	//execute R and load R script in temporary file
	OSS systemCommand;
#ifdef R_EXECUTABLE_PATH
	systemCommand << Common::ResourceMap::GetInstance().get("R-executable-command") << " --no-save --silent < \"" << temporaryFileName << "\"" << Os::GetDeleteCommandOutput();
#else
	throw NotYetImplementedException(HERE) << "GraphImplementation::draw() need R";
#endif
	if (system(String(systemCommand).c_str())) throw InternalException(HERE) << "GraphImplementation: error trying to execute R command=" << String(systemCommand);
       	if (remove(temporaryFileName.c_str()) == -1) Log::Warn(OSS() << "GraphImplementation: error trying to remove file " << temporaryFileName);
        clean();
      }

      /* Clean temporary files */
      void GraphImplementation::clean()
      {
	UnsignedLong drawableNumber(drawablesCollection_.getSize());
	// Clean all the temporary data created by the drawables during their drawing
	for (UnsignedLong i = 0; i < drawableNumber; ++i)
	  {
	    if (drawablesCollection_[i].getData().getSize() != 0)
	      drawablesCollection_[i].clean();
	  }
      }

      /* The method that generates the graphic files */
      void GraphImplementation::draw(const String & file,
				     const NumericalScalar width,
				     const NumericalScalar height,
				     const Format format) /* throw(InternalException) */
      {
	draw(".", file, width, height, format);
      }

      /* The method returning absolute path of the bitmap graphic file */
      String GraphImplementation::getBitmap() const
      {
	String bitmap(path_);
	if( bitmap.find_last_of('/') != (bitmap.length() - 1) )
	  bitmap += "/";
	bitmap += fileName_ + ".png";
	return bitmap;
      }

      /* The method returning absolute path of the postscript graphic file */
      String GraphImplementation::getPostscript() const
      {
	String postScript(path_);
	if( postScript.find_last_of('/') != (postScript.length() - 1) )
	  postScript += "/";
	postScript += fileName_ + ".eps";
	return postScript;
      }

      /* The method returning absolute path of the PDF graphic file */
      String GraphImplementation::getPDF() const
      {
	String pdf(path_);
	if( pdf.find_last_of('/') != (pdf.length() - 1) )
	  pdf += "/";
	pdf += fileName_ + ".pdf";
	return pdf;
      }

      /* The method returning absolute path of the vectorial graphic file */
      String GraphImplementation::getVectorial() const
      {
	String vectorial(path_);
	if( vectorial.find_last_of('/') != (vectorial.length() - 1) )
	  vectorial += "/";
	vectorial += fileName_ + ".fig";
	return vectorial;
      }

      /* Get the bounding box of the whole plot */
      GraphImplementation::BoundingBox GraphImplementation::getBoundingBox() const
      {
	if (automaticBoundingBox_) computeBoundingBox();
	return boundingBox_;
      }

      /* Set the bounding box of the whole plot */
      void GraphImplementation::setBoundingBox(const BoundingBox & boundingBox)
      {
	if (boundingBox.getDimension() != BoundingBoxSize) throw InvalidArgumentException(HERE) << "Error: the given bounding box must have a dimension equals to " << BoundingBoxSize << ", here boundingBox=" << boundingBox;
	boundingBox_ = boundingBox;
	automaticBoundingBox_ = false;
      }

      void GraphImplementation::setBoundingBox(const Interval & boundingBox)
      {
	if (boundingBox.getDimension() != 2) throw InvalidArgumentException(HERE) << "Error: the given bounding box must have a dimension equals to 2";
	if (!boundingBox.getFiniteLowerBound()[0])  throw InvalidArgumentException(HERE) << "Error: the lower bound of the first component must be finite";
	if (!boundingBox.getFiniteLowerBound()[1])  throw InvalidArgumentException(HERE) << "Error: the lower bound of the second component must be finite";
	if (!boundingBox.getFiniteUpperBound()[0])  throw InvalidArgumentException(HERE) << "Error: the upper bound of the first component must be finite";
	if (!boundingBox.getFiniteUpperBound()[1])  throw InvalidArgumentException(HERE) << "Error: the upper bound of the second component must be finite";
	boundingBox_[0] = boundingBox.getLowerBound()[0];
	boundingBox_[1] = boundingBox.getUpperBound()[0];
	boundingBox_[2] = boundingBox.getLowerBound()[1];
	boundingBox_[3] = boundingBox.getUpperBound()[1];
	automaticBoundingBox_ = false;
      }

      /* Automatic bounding box accessor */
      Bool GraphImplementation::getAutomaticBoundingBox() const
      {
        return automaticBoundingBox_;
      }

      void GraphImplementation::setAutomaticBoundingBox(const Bool automaticBoundingBox)
      {
        automaticBoundingBox_ = automaticBoundingBox;
      }

      /* Compute the best bounding box to enclose all the drawables */
      void GraphImplementation::computeBoundingBox() const
      {
	UnsignedLong size(drawablesCollection_.getSize());
	boundingBox_ = BoundingBox(BoundingBoxSize);
	// First exceptional case: no drawable, we default to default bounding box
	if (size == 0)
	  {
	    Log::Info("Warning: cannot compute the bounding box of a graph with no drawable, switch to [0,1]x[0,1] default bounding box");
	    boundingBox_[0] = 0.0;
	    boundingBox_[1] = 1.0;
	    boundingBox_[2] = 0.0;
	    boundingBox_[3] = 1.0;
	    return;
	  }
	NumericalSample boxes(size, BoundingBoxSize);

	// first, get each Drawable's bounding box and drawing command
	for(UnsignedLong i = 0; i < size; ++i)
	  {
	    if (drawablesCollection_[i].getData().getSize() != 0)
	      boxes[i] = drawablesCollection_[i].getBoundingBox();
	  }

	BoundingBox min(boxes.getMin());
	BoundingBox max(boxes.getMax());
	boundingBox_[0]= min[0];
	boundingBox_[1]= max[1];
	boundingBox_[2]= min[2];
	boundingBox_[3]= max[3];
	// All the bounding boxes are degenerated to a point, we default to a 1x1 box centered at this point
	if ((boundingBox_[0] == boundingBox_[1]) || (boundingBox_[2] == boundingBox_[3]))
	  {
	    Log::Info("Warning: the overall bounding box is degenerated to a point. Switch to a 1x1 box centered at this point");
	    boundingBox_[0] -= 0.5;
	    boundingBox_[1] += 0.5;
	    boundingBox_[2] -= 0.5;
	    boundingBox_[3] += 0.5;
	    return;
	  }

	return;
      }
      
      /* Get the legend position */
      String GraphImplementation::getLegendPosition() const
      {
	return legendPosition_;
      }

      /* Set the legend position */
      void GraphImplementation::setLegendPosition(const String & position) /* throw(InvalidArgumentException) */
      {
	if(!isValidLegendPosition(position)) throw InvalidArgumentException(HERE) << "The given legend position = " << position << " is invalid";
	
	legendPosition_ = position;
      }

      /* Gives all the valid legend positions */
      GraphImplementation::Description GraphImplementation::GetValidLegendPositions()
      {
	if(IsFirstInitialization)
	  {
	    initializeValidLegendPositions();
	    IsFirstInitialization = false;
	  }
	return ValidLegendPositions;
      }

      /* Get the legend font size */
      NumericalScalar GraphImplementation::getLegendFontSize() const
      {
	return legendFontSize_;
      }

      /* Set the legend font size */
      void GraphImplementation::setLegendFontSize(const NumericalScalar legendFontSize) /* throw(InvalidArgumentException) */
      {
	if(legendFontSize <= 0.0) throw InvalidArgumentException(HERE) << "The given legend font size = " << legendFontSize << " is invalid";
	
	legendFontSize_ = legendFontSize;
      }

      /* check for legend position validity */
      Bool GraphImplementation::isValidLegendPosition(const String & position) const
      {
	const Description::const_iterator it = find(ValidLegendPositions.begin(), ValidLegendPositions.end(), position);
 
	return (it != ValidLegendPositions.end());
      }

      /* Method save() stores the object through the StorageManager */
      void GraphImplementation::save(StorageManager::Advocate & adv) const
      {
	PersistentObject::save(adv);
	adv.saveAttribute( "title_", title_ );
	adv.saveAttribute( "legendPosition_", legendPosition_ );
	adv.saveAttribute( "legendFontSize_", legendFontSize_ );
	adv.saveAttribute( "xTitle_", xTitle_ );
	adv.saveAttribute( "yTitle_", yTitle_ );
	adv.saveAttribute( "showAxes_", showAxes_ );
	adv.saveAttribute( "drawablesCollection_", drawablesCollection_ );
	adv.saveAttribute( "path_", path_ );
	adv.saveAttribute( "fileName_", fileName_ );
      }

      /* Method load() reloads the object from the StorageManager */
      void GraphImplementation::load(StorageManager::Advocate & adv)
      {
	PersistentObject::load(adv);
	adv.loadAttribute( "title_", title_ );
	adv.loadAttribute( "legendPosition_", legendPosition_ );
	adv.loadAttribute( "legendFontSize_", legendFontSize_ );
	adv.loadAttribute( "xTitle_", xTitle_ );
	adv.loadAttribute( "yTitle_", yTitle_ );
	adv.loadAttribute( "showAxes_", showAxes_ );
	adv.loadAttribute( "drawablesCollection_", drawablesCollection_ );
	adv.loadAttribute( "path_", path_ );
	adv.loadAttribute( "fileName_", fileName_ );
      }

    } /* namespace Graph */

  }/* namespace Base */

}/* namespace OpenTURNS */
