//                                               -*- C++ -*-
/**
 *  @file  OTtypes.hxx
 *  @brief The header file that declares all general types
 *
 *  (C) Copyright 2005-2010 EDF-EADS-Phimeca
 *
 *  This library is free software; you can redistribute it and/or
 *  modify it under the terms of the GNU Lesser General Public
 *  License as published by the Free Software Foundation; either
 *  version 2.1 of the License.
 *
 *  This library is distributed in the hope that it will be useful
 *  but WITHOUT ANY WARRANTY; without even the implied warranty of
 *  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 *  Lesser General Public License for more details.
 *
 *  You should have received a copy of the GNU Lesser General Public
 *  License along with this library; if not, write to the Free Software
 *  Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307 USA
 *
 *  @author: $LastChangedBy: dutka $
 *  @date:   $LastChangedDate: 2010-02-04 16:44:49 +0100 (jeu. 04 févr. 2010) $
 *  Id:      $Id: OTtypes.hxx 1473 2010-02-04 15:44:49Z dutka $
 */
#ifndef OPENTURNS_OTTYPES_HXX
#define OPENTURNS_OTTYPES_HXX

#include <string>                // for std::string
#include <complex>               // for std::complex

/**
 * @namespace  OpenTURNS
 * @brief      The namespace where all Open TURNS symbols live.
 */

namespace OpenTURNS
{

  /**
   * Type Bool is an alias for the C++ standard bool type
   */
  typedef bool Bool;

  /**
   * Type String should store Unicode strings (for later internationalization)
   */
  typedef std::string String;

  /**
   * Type Id is an unique identifier number for PersistentObjects
   */
  typedef unsigned long Id;

  /**
   * Type NumericalScalar is a floating point number representation
   */
  typedef double NumericalScalar;

  /**
   * Type NumericalComplex is a complex floating point number representation
   */
  typedef std::complex<NumericalScalar> NumericalComplex;

  /**
   * Type FileName stores file names and pathes
   */
  typedef std::string FileName;

  /**
   * Type UnsignedLong stores values for that should be positive integers
   */
  typedef unsigned long UnsignedLong;

  /**
   * Type SizeType stores values for sizes, quantities, etc.
   */
  typedef long SizeType;

  /**
   * Type IndexType stores values for indexes, positions, etc.
   */
  typedef long IndexType;

  /**
   * Type LibraryHandle stores dynamic library pointers
   */
  typedef void * LibraryHandle;

  /**
   * Type LibrarySymbol stores pointers library symbols
   */
  typedef void * LibrarySymbol;

  /**
   * The types for utility functions
   */
  typedef void (*LogFunctionPointer) (const String &);




  /**
   * @fn Bool operator !=(const T & lhs, const T & rhs)
   *
   * @brief Templated inequality comparison operator
   * @param lhs The left hand side of the comparison
   * @param rhs The right hand side of the comparison
   * @return True if \em lhs is different from \em rhs
   */

  template <class T> inline
  Bool operator !=(const T & lhs, const T & rhs)
  {
    return !(lhs == rhs);
  }


  // Here are forward declarations of namespaces for documentation needs

  /** @brief The namespace for the Base part of the library */
  namespace Base {

    /** @brief The namespace for basic objects of general usage */
    namespace Common {}

    /** @brief The namespace for finite differentation algorithms */
    namespace Diff {}

    /** @brief The namespace for mathematical function modelization */
    namespace Func {}

    /** @brief The namespace for graphical outputs */
    namespace Graph {}

    /** @brief The namespace for optimization algorithms */
    namespace Optim {}

    /** @brief The namespace for optimization algorithms */
    namespace Optim {}

    /** @brief The namespace for general purpose solver definitions */
    namespace Solver {}

    /** @brief The namespace for statistical type definitions */
    namespace Stat {}

    /** @brief The namespace for Open TURNS' specific type definitions */
    namespace Type {}

  } /* namespace Base */




  /** @brief The namespace for the Uncertainty part of the library */
  namespace Uncertainty {

    /** @brief The namespace for uncertainty algorithm definitions */
    namespace Algorithm {}

    /** @brief The namespace for actual distribution and copula definitions */
    namespace Distribution {}

    /** @brief The namespace for high level probabilistic object definitions */
    namespace Model {}

    /** @brief The namespace for statistical test definitions */
    namespace StatTest {}

  
  } /* namespace Uncertainty */




} /* namespace OpenTURNS */


/**
 * An alias to namespace OpenTURNS.
 * @sa         OpenTURNS
 */
namespace OT = OpenTURNS;


#endif /* OPENTURNS_OTTYPES_HXX */
