/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdlib.h>
#include <stdio.h>

#include "MRW_Private.h"

char *progname;

void
Usage ()
{
	fprintf (stderr, "Usage: %s < mrwfile\n", progname);
	exit (1);
}

typedef struct {
	unsigned tag;
	char	*name;
} TAG_ENTRY;

TAG_ENTRY ifdtab[] = {
	{ IFDTAG_WIDTH,		"Width" },
	{ IFDTAG_HEIGHT,	"Height" },
	{ IFDTAG_COMPRESS,	"Compression" },
	{ IFDTAG_DCFVER,	"DCF Version" },
	{ IFDTAG_MANUF,		"Manufacturer" },
	{ IFDTAG_CAMERA,	"Camera" },
	{ IFDTAG_FIRMWARE,	"Firmware" },
	{ IFDTAG_DATETIME,	"Date/time" },
	{ IFDTAG_EXIFOFFSET,	"Exif Offset" },
	{ 0, NULL }
};

TAG_ENTRY exiftab[] = {
	{ EXIFTAG_EXPOSURETIME,		"Exposure time" },
	{ EXIFTAG_FNUMBER,		"F Number" },
	{ EXIFTAG_EXPPROG,		"Exposure Program" },
	{ EXIFTAG_ISOSPEED,		"ISO Speed" },
	{ EXIFTAG_VERSION,		"Version" },
	{ EXIFTAG_DATETIMEORIG,		"Date/time Orig" },
	{ EXIFTAG_DATETIMEDIGITIZED,	"Date/time Digitized" },
	{ EXIFTAG_SHUTTERSPEED,		"Shutter Speed Code" },
	{ EXIFTAG_APERTURE,		"Aperture Code" },
	{ EXIFTAG_EXPOSUREBIAS,		"Exposure Bias" },
	{ EXIFTAG_MAXAPERTURE,		"Maximum Aperture" },
	{ EXIFTAG_METERMODE,		"Meter Mode" },
	{ EXIFTAG_LIGHTSOURCE,		"Light Source" },
	{ EXIFTAG_FLASH,		"Flash" },
	{ EXIFTAG_FOCALLEN,		"Focal Length" },
	{ EXIFTAG_MANUFNOTES,		"Manufact. Notes Offset" },
	{ EXIFTAG_COLORSPACE,		"Color Space" },
	{ 0, NULL }
};

TAG_ENTRY manutab[] = {
	{ MANUTAG_BLOCKDESC,		"Block Desc" },
	{ MANUTAG_CAMERASETTINGS,	"Camera Settings1" },
	{ MANUTAG_CAMERASETTINGS2,	"Camera Settings2" },
	{ MANUTAG_AUTOFOCUS,		"Autofocus" },
	{ MANUTAG_IMAGESIZE,		"Image size" },
	{ MANUTAG_THUMBNAIL,		"Thumbnail" },
	{ MANUTAG_PIMINFO,		"PIM Info" },
	{ 0, NULL }
};

static void
PrintEntry (IFD *ifd, IFD_entry *p, TAG_ENTRY *tab)
{
	int	undefined = 0;

	while (tab->name != NULL && tab->tag != p->ifd_tag)
		tab++;
	if (tab->name)
		printf ("%22.22s: ", tab->name);
	else
		printf ("Unknown tag 0x%08x: ", p->ifd_tag);

	switch (p->ifd_type) {
		case IFDTYPE_BYTE:
			printf ("Byte           "); break;
		case IFDTYPE_STRING:
			printf ("String         "); break;
		case IFDTYPE_SHORT:
			printf ("Short          "); break;
		case IFDTYPE_LONG:
			printf ("Long           "); break;
		case IFDTYPE_RATIONAL:
			printf ("Rational       "); break;
		case IFDTYPE_SBYTE:
			printf ("Signed byte    "); break;
		case IFDTYPE_UNDEFINED:
			undefined = 1;
			printf ("Undefined      "); break;
		case IFDTYPE_SSHORT:
			printf ("Signed short   "); break;
		case IFDTYPE_SLONG:
			printf ("Signed long    "); break;
		case IFDTYPE_SRATIONAL:
			printf ("Signed rational"); break;
		case IFDTYPE_FLOAT:
			undefined = 1;
			printf ("Float          "); break;
		case IFDTYPE_DOUBLE:
			undefined = 1;
			printf ("Double         "); break;
		default:
			undefined = 1;
			printf ("Unknown type %d", p->ifd_type);
	}

	printf (". %4d value%s:", p->ifd_count, p->ifd_count == 1 ? "" : "s");
	if ((p->ifd_count > 4 && p->ifd_type != IFDTYPE_STRING) || undefined) {
		printf (" NOT PRINTED");
	}
	else {
		int	i;

		if (p->ifd_type == IFDTYPE_STRING)
			printf (" ");
		for (i = 0; i < p->ifd_count; i++) {
			int num, denom;
			switch (p->ifd_type) {
				case IFDTYPE_BYTE:
				case IFDTYPE_SBYTE:
					printf (" %d", p->ifd_offset >> 24); break;
				case IFDTYPE_STRING:
					printf ("%c", mrw_val (ifd->ifd_data, p->ifd_offset + i, 1)); break;
				case IFDTYPE_SHORT:
				case IFDTYPE_SSHORT:
					printf (" %d", p->ifd_offset >> 16); break;
				case IFDTYPE_LONG:
				case IFDTYPE_SLONG:
					printf (" %d", p->ifd_offset); break;
				case IFDTYPE_RATIONAL:
					IFD_GetRational (ifd, p->ifd_tag, &num, &denom);
					printf (" %d/%d", num, denom);
					if (denom == 0)
						printf (" = NaN");
					else
						printf (" = %g", num/(double)denom);
					break;
				case IFDTYPE_SRATIONAL:
					IFD_GetSRational (ifd, p->ifd_tag, &num, &denom);
					printf (" %d/%d", num, denom);
					if (denom == 0)
						printf (" = NaN");
					else
						printf (" = %g", num/(double)denom);
					break;
			}
		}
	}

	printf ("\n");
}

static void
PrintTable (IFD *ifd, TAG_ENTRY *tab)
{
	int	i;

	for (i = 0; i < ifd->ifd_count; i++)
		PrintEntry (ifd, &ifd->ifd_entry[i], tab);
}

TAG_ENTRY riftab[] = {
	{  1, "Saturation" },
	{  2, "Contrast" },
	{  3, "Sharpness" },
	{  4, "White Balance" },
	{  5, "Subject Program" },
	{  6, "Film Speed" },
	{  7, "Color Mode" },
	{  8, "Preset1R" },
	{ 10, "Preset1B" },
	{ 12, "Preset2R" },
	{ 14, "Preset2B" },
	{ 16, "Preset3R" },
	{ 18, "Preset3B" },
	{ 20, "Preset4R" },
	{ 22, "Preset4B" },
	{ 24, "Preset5R" },
	{ 26, "Preset5B" },
	{ 28, "Preset6R" },
	{ 30, "Preset6B" },
	{ 56, "Color Filter" },
	{ 57, "B&W Filter" },
	{ -1, NULL }
};

void
PrintRIFByte (int offset, int value)
{
	TAG_ENTRY	*tab = riftab;

	while (tab->name != NULL && tab->tag != offset)
		tab++;
	printf (" RIF ");
	if (tab->name)
		printf ("%22.22s", tab->name);
	else
		printf ("Unknown 0x%08x", offset);
	printf (": %d\n", value);
}

void
PrintRIFShort (int offset, unsigned char b1, unsigned char b2)
{
	TAG_ENTRY	*tab = riftab;
	int		value;

	value = b1 & 0x80 ? -1 : 0;
	value = (value << 8) | b1;
	value = (value << 8) | b2;
	while (tab->name != NULL && tab->tag != offset)
		tab++;
	printf (" RIF ");
	if (tab->name)
		printf ("%22.22s", tab->name);
	else
		printf ("Unknown 0x%08x", offset);
	printf (": %d\n", value);
}

int
main (int argc, char *argv[])
{
	mrw_header	hdr;
	int		num, denom;
	char		*errmsg;
	int		i;

	progname = argv[0];
	if (!LoadMRW (&hdr, stdin, &errmsg)) {
		fprintf (stderr, "%s: Unable to read MRW header:\n", progname);
		fprintf (stderr, "    %s\n", errmsg);
		Usage();
	}
	printf ("MRM Block Length: %d\n", hdr.mrm.length);

	printf ("\n");
	printf ("PRD Block Length: %d\n", hdr.prd.length);
	printf ("CCD Height: %d\n", MRW_Height (&hdr));
	printf ("CCD Width: %d\n", MRW_Width (&hdr));
	printf ("Image height: %d\n", mrw_val (hdr.prd.data, MRW_PRD_IMAGE_HEIGHT, MRW_PRD_IMAGE_HEIGHT_LEN));
	printf ("Image width: %d\n", mrw_val (hdr.prd.data, MRW_PRD_IMAGE_WIDTH, MRW_PRD_IMAGE_WIDTH_LEN));

	printf ("Unknown2: 0x%02x\n", mrw_val (hdr.prd.data, MRW_PRD_UNKNOWN2, MRW_PRD_UNKNOWN2_LEN));
	printf ("Unknown3: 0x%02x\n", mrw_val (hdr.prd.data, MRW_PRD_UNKNOWN3, MRW_PRD_UNKNOWN3_LEN));
	printf ("Unknown4: 0x%02x\n", mrw_val (hdr.prd.data, MRW_PRD_UNKNOWN4, MRW_PRD_UNKNOWN4_LEN));
	printf ("Unknown5: 0x%02x\n", mrw_val (hdr.prd.data, MRW_PRD_UNKNOWN5, MRW_PRD_UNKNOWN5_LEN));


	printf ("\n");
	printf ("TTW Block Length: %d\n", hdr.ttw.length);

	printf ("TiffMagic: 0x%04x\n", mrw_val (hdr.ttw.data, MRW_TIFFMAGIC, MRW_TIFFMAGIC_LENGTH));
	printf ("TiffVersion: %d\n", mrw_val (hdr.ttw.data, MRW_TIFFVERSION, MRW_TIFFVERSION_LENGTH));
	printf ("TiffIFD0 Offset: %d\n", mrw_val (hdr.ttw.data, MRW_TIFFIFD0_OFFSET, MRW_TIFFIFD0_LENGTH));

	printf ("\n");
	printf ("DCF version: %s\n", IFD_GetString (hdr.mrw_ifd, IFDTAG_DCFVER));
	printf ("Manufacturer: %s\n", IFD_GetString (hdr.mrw_ifd, IFDTAG_MANUF));
	printf ("Camera: %s\n", IFD_GetString (hdr.mrw_ifd, IFDTAG_CAMERA));
	printf ("Firmware: %s\n", IFD_GetString (hdr.mrw_ifd, IFDTAG_FIRMWARE));
	printf ("Date/time: %s\n", IFD_GetString (hdr.mrw_ifd, IFDTAG_DATETIME));

	printf ("ISO: %d\n", IFD_GetShort (hdr.exif_ifd, EXIFTAG_ISOSPEED));
	if (IFD_GetSRational (hdr.exif_ifd, EXIFTAG_SHUTTERSPEED, &num, &denom))
		printf ("Shutter: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetRational (hdr.exif_ifd, EXIFTAG_EXPOSURETIME, &num, &denom))
		printf ("Exp Time: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetRational (hdr.exif_ifd, EXIFTAG_APERTURE, &num, &denom))
		printf ("Aperture: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetRational (hdr.exif_ifd, EXIFTAG_FNUMBER, &num, &denom))
		printf ("F Number: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetSRational (hdr.exif_ifd, EXIFTAG_EXPOSUREBIAS, &num, &denom))
		printf ("Exposure Bias: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetRational (hdr.exif_ifd, EXIFTAG_MAXAPERTURE, &num, &denom))
		printf ("Max Aperture: %d/%d = %g\n", num, denom, num/(double)denom);
	if (IFD_GetRational (hdr.exif_ifd, EXIFTAG_FOCALLEN, &num, &denom))
		printf ("Focal Length: %d/%d = %g\n", num, denom, num/(double)denom);
	printf ("Meter mode: %d\n", IFD_GetShort (hdr.exif_ifd, EXIFTAG_METERMODE));
	printf ("Light source: %d\n", IFD_GetShort (hdr.exif_ifd, EXIFTAG_LIGHTSOURCE));
	printf ("Flash: %d\n", IFD_GetShort (hdr.exif_ifd, EXIFTAG_FLASH));
	printf ("Color Space: %d\n", IFD_GetShort (hdr.exif_ifd, EXIFTAG_COLORSPACE));
	printf ("Thumbnail: %d\n", IFD_GetLong (hdr.manu_ifd, MANUTAG_THUMBNAIL));
	printf ("ManutagUnknown: %d\n", IFD_GetLong (hdr.manu_ifd, MANUTAG_UNKNOWN));

	printf ("Focus Distance: %g\n", MRW_FocusLen (&hdr));

	printf ("Macro: %d\n", mrw_val (hdr.settings, 0x2F, 1));
	printf ("Flash: %d\n", mrw_val (hdr.settings, 0x53, 1));
	printf ("Pictnum: %d\n", mrw_val (hdr.settings, 0x6E, 2));
	printf ("WBalance: %d/%d/%d\n", mrw_val (hdr.settings, 0x72, 2), mrw_val (hdr.settings, 0x76, 2),
		                     mrw_val (hdr.settings, 0x7A, 2));
	printf ("Color: %d\n", mrw_val (hdr.settings, 0x7F, 1));
	printf ("Contrast: %d\n", mrw_val (hdr.settings, 0x83, 1));
	{ int fMode = MRW_FocusMode (&hdr);
	  if (fMode == MRI_FOCUS_AUTO)
	      printf ("Focus mode: Auto\n");
	  else if (fMode == MRI_FOCUS_MANUAL)
	      printf ("Focus mode: Manual\n");
	  else
	      printf ("Focus mode: Unknown\n");
	}

	printf ("\n");
	printf ("MRW Table:\n");
	PrintTable (hdr.mrw_ifd, ifdtab);
	printf ("\n");
	printf ("Exif Table:\n");
	PrintTable (hdr.exif_ifd, exiftab);
	printf ("\n");
	printf ("Manufacturer Table:\n");
	PrintTable (hdr.manu_ifd, manutab);

	printf ("\n");
	printf ("WBG Block Length: %d\n", hdr.wbg.length);
	printf (" WBG R scale = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_RSCALE_OFFSET, MRW_WBG_RSCALE_LENGTH));
	printf (" WBG G1 scale = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_G1SCALE_OFFSET, MRW_WBG_G1SCALE_LENGTH));
	printf (" WBG G2 scale = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_G2SCALE_OFFSET, MRW_WBG_G2SCALE_LENGTH));
	printf (" WBG B scale = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_BSCALE_OFFSET, MRW_WBG_BSCALE_LENGTH));
	printf (" WBG R gain = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_RGAIN_OFFSET, MRW_WBG_RGAIN_LENGTH));
	printf (" WBG G1 gain = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_G1GAIN_OFFSET, MRW_WBG_G1GAIN_LENGTH));
	printf (" WBG G2 gain = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_G2GAIN_OFFSET, MRW_WBG_G2GAIN_LENGTH));
	printf (" WBG B gain = %d\n", mrw_val (hdr.wbg.data, MRW_WBG_BGAIN_OFFSET, MRW_WBG_BGAIN_LENGTH));

	printf ("\n");
	printf ("RIF Block Length: %d\n", hdr.rif.length);
	for (i = 0; i < (hdr.rif.length < 8 ? hdr.rif.length : 8); i++)
		PrintRIFByte (i, hdr.rif.data[i]);
	for (i = 8; i < (hdr.rif.length < 0x38 ? hdr.rif.length : 0x38); i+=2)
		PrintRIFShort (i, hdr.rif.data[i], hdr.rif.data[i+1]);
	for (i = 0x38; i < hdr.rif.length; i++)
		PrintRIFByte (i, hdr.rif.data[i]);

	exit (0);
}
