/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include "MRI.h"

static inline float
median3 (float a, float b, float c)
{
	if (a > b) {
		if (a > c)
			return b > c ? b : c;
		else
			return a;
	}
	else {
		if (b > c)
			return a > b ? a : c;
		else
			return b;
	}
}

static float
median4 (float a, float b, float c, float d)
{
	float t;

	/* Sort ab */
	if (a > b) { t = b; b = a; a = t; }
	/* Sort abc */
	if (b > c) {
		t = c;
		c = b;
		if (a > t) {
			b = a;
			a = t;
		}
		else {
			b = t;
		}
	}
	if (d >= c) /* Order is abcd */
		return (b + c) / 2;
	else if (d >= a) /* Order is abdc or adbc */
		return (b + d) / 2;
	else /* Order is dabc */
		return (a + b) / 2;
}

int
median5 (float a, float b, float c, float d, float e)
{
	float t;

	/* Sort ab */
	if (a > b) { t = b; b = a; a = t; }
	/* Sort abc */
	if (b > c) {
		t = c;
		c = b;
		if (a > t) {
			b = a;
			a = t;
		}
		else {
			b = t;
		}
	}
	/* Sort abcd */
	if (c > d) {
		t = d;
		d = c;
		if (b <= t) {
			c = t;
		}
		else {
			c = b;
			if (a <= t) {
				b = t;
			}
			else {
				b = a;
				a = t;
			}
		}
	}
	if (e >= c) /* Order is abcde or abced */
		return c;
	else if (e >= b) /* Order is abecd */
		return e;
	else /* Order is aebcd or eabcd */
		return b;
}

static inline float sub (float a, float b)
{
	return a - b;
}
	
static inline float add (float a, float b)
{
	return a + b;
}
	
struct InterpData {
	struct link *next;
	int	width;
	int	height;
	int	y;
	int	redGreen;
	struct MRI_ScanLine *old, *cur;
};

static void
StartInterpolation (void *private, int width, int height, int freedata)
{
	struct InterpData *wd = private;

	wd->width = width;
	wd->height = height;
	(*wd->next->start) (wd->next->private, width, height, freedata);
}

static void
InterpBilinearBRow (void *private, void *data)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	struct MRI_ScanLine *new = data;
	int x;

	if (cur != (struct MRI_ScanLine *)0) {
		if (wd->redGreen) {
			if (old == (struct MRI_ScanLine *)0) {
				/* First row. */
				float *cB = cur->B;
				float *nB = new->B;
				cB[0] = nB[1];
				for (x = 1; x != wd->width-1; x += 2) {
					cB[x] = nB[x];
					cB[x+1] = (nB[x] + nB[x+2])/2;
				}
				cB[x] = nB[x];
			}
			else {
				float *cB = cur->B;
				float *oB = old->B;
				float *nB = new->B;
				cB[0] = (oB[1] + nB[1]) / 2;
				for (x = 1; x != wd->width-1; x += 2) {
					cB[x] = (oB[x] + nB[x])/2;
					cB[x+1] = (oB[x] + oB[x+2] + nB[x] + nB[x+2])/4;
				}
				cB[x] = (oB[x] + nB[x]) / 2;
			}
		}
		else {
			/* Rows GBGBGB...GB */
			float *cB = cur->B;
			cB[0] = cB[1];
			for (x = 2; x < wd->width; x += 2)
				cB[x] = (cB[x-1] + cB[x+1])/2;
		}
		wd->redGreen = !wd->redGreen;
	}
	wd->y++;
	if (old != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, old);
	wd->old = cur;
	wd->cur = new;
}

static void
InterpBilinearBClose (void *private)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	int x;

	/* Last row. */
	if (wd->redGreen) {
		float *cB = cur->B;
		float *oB = old->B;
		cB[0] = oB[1];
		for (x = 1; x != wd->width-1; x += 2) {
			cB[x] = oB[x];
			cB[x+1] = (oB[x] + oB[x+2])/2;
		}
		cB[x] = oB[x];
	}
	else {
		float *cB = cur->B;
		cB[0] = cB[1];
		for (x = 2; x < wd->width; x += 2)
			cB[x] = (cB[x-1] + cB[x+1])/2;
	}
	(*wd->next->row) (wd->next->private, old);
	(*wd->next->row) (wd->next->private, cur);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static MRI_Link *
MRI_GenInterpBilinearB (MRI_Link *next, int RGFirst)
{
	struct InterpData *wd = malloc (sizeof (struct InterpData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct InterpData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartInterpolation;
	ep->row = InterpBilinearBRow;
	ep->close = InterpBilinearBClose;
	ep->private = wd;
	wd->next = next;

	wd->cur = wd->old = (struct MRI_ScanLine *)0;
	wd->y = 0;
	wd->redGreen = RGFirst;

	return ep;
}

static void
InterpBilinearRRow (void *private, void *data)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	struct MRI_ScanLine *new = data;
	int x;

	if (cur != (struct MRI_ScanLine *)0) {
		if (wd->redGreen) {
			/* Even rows RGRGRG...RG. */
			float *cR = cur->R;
			for (x = 1; x != wd->width-1; x += 2)
				cR[x] = (cR[x-1] + cR[x+1]) / 2;
			cR[x] = cR[x-1];
		}
		else if (old == (struct MRI_ScanLine *)0)  {
			/* First row and GBGBGB...GB */
			float *cR = cur->R;
			float *nR = new->R;
			cR[0] = nR[0];
			for (x = 1; x != wd->width-1; x += 2) {
				cR[x] = (nR[x-1] + nR[x+1])/2;
				cR[x+1] = nR[x+1];
			}
			cR[x] = nR[x-1];
		}
		else {
			/* Other rows GBGBGB...GB */
			float *cR = cur->R;
			float *oR = old->R;
			float *nR = new->R;

			cR[0] = (oR[0] + nR[0])/2;
			for (x = 1; x != wd->width-1; x += 2) {
				cR[x] = (oR[x-1] + oR[x+1] + nR[x-1] + nR[x+1])/4;
				cR[x+1] = (oR[x+1] + nR[x+1])/2;
			}
			cR[x] = (oR[x-1] + nR[x-1]) / 2;
		}
		wd->redGreen = !wd->redGreen;
	}
	wd->y++;
	if (old != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, old);
	wd->old = cur;
	wd->cur = new;
}

static void
InterpBilinearRClose (void *private)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	int x;

	/* Last row. */
	if (wd->redGreen) {
		float *cR = cur->R;
		for (x = 1; x != wd->width-1; x += 2)
			cR[x] = (cR[x-1] + cR[x+1]) / 2;
		cR[x] = cR[x-1];
	}
	else {
		float *cR = cur->R;
		float *oR = old->R;
		cR[0] = oR[0];
		for (x = 1; x != wd->width-1; x += 2) {
			cR[x] = (oR[x-1] + oR[x+1])/2;
			cR[x+1] = oR[x+1];
		}
		cR[x] = oR[x-1];
	}
	(*wd->next->row) (wd->next->private, old);
	(*wd->next->row) (wd->next->private, cur);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static void
InterpBilinearGRow (void *private, void *data)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	struct MRI_ScanLine *new = data;
	int x;

	if (cur != (struct MRI_ScanLine *)0) {
		if (wd->redGreen) {
			if (old == (struct MRI_ScanLine *)0) {
				float *cG = cur->G;
				float *nG = new->G;
				cG[0] = (nG[0] + cG[1]) / 2;
				for (x = 2; x < wd->width; x += 2)
					cG[x] = (cG[x-1] + nG[x] + cG[x+1])/3;
			}
			else {
				/* Rows RGRGRG...RG after the first. */
				float *cG = cur->G;
				float *oG = old->G;
				float *nG = new->G;
				cG[0] = (oG[0] + cG[1] + nG[0]) / 3;
				for (x = 2; x < wd->width; x += 2)
					cG[x] = (oG[x] + cG[x-1] + nG[x] + cG[x+1])/4;
			}
		}
		else {
			/* Rows GBGBGB...GB */
			if (old == (struct MRI_ScanLine *)0) {
				float *cG = cur->G;
				float *nG = new->G;
				for (x = 1; x != wd->width-1; x += 2)
					cG[x] = (cG[x-1] + nG[x] + cG[x+1])/3;
				cG[x] = (cG[x-1] + nG[x])/2;
			}
			else {
				float *cG = cur->G;
				float *oG = old->G;
				float *nG = new->G;
				for (x = 1; x != wd->width-1; x += 2)
					cG[x] = (oG[x] + cG[x-1] + nG[x] + cG[x+1])/4;
				cG[x] = (oG[x] + cG[x-1] + nG[x])/3;
			}
		}
		wd->redGreen = !wd->redGreen;
	}
	wd->y++;
	if (old != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, old);
	wd->old = cur;
	wd->cur = new;
}

static void
InterpBilinearGClose (void *private)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	int x;

	/* Last row. */
	if (wd->redGreen) {
		float *cG = cur->G;
		float *oG = old->G;
		cG[0] = (oG[0] + cG[1]) / 2;
		for (x = 2; x < wd->width; x += 2)
			cG[x] = (oG[x] + cG[x-1] + cG[x+1])/3;
	}
	else {
		float *cG = cur->G;
		float *oG = old->G;
		for (x = 1; x != wd->width-1; x += 2)
			cG[x] = (oG[x] + cG[x-1] + cG[x+1])/3;
		cG[x] = (oG[x] + cG[x-1])/2;
	}
	(*wd->next->row) (wd->next->private, old);
	(*wd->next->row) (wd->next->private, cur);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static MRI_Link *
MRI_GenInterpBilinearR (MRI_Link *next, int RGFirst)
{
	struct InterpData *wd = malloc (sizeof (struct InterpData));
	struct link *ep = malloc (sizeof (*ep));

	if (wd == (struct InterpData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartInterpolation;
	ep->row = InterpBilinearRRow;
	ep->close = InterpBilinearRClose;
	ep->private = wd;
	wd->next = next;

	wd->cur = wd->old = (struct MRI_ScanLine *)0;
	wd->y = 0;
	wd->redGreen = RGFirst;

	return ep;
}

static MRI_Link *
MRI_GenInterpBilinearG (MRI_Link *next, int RGFirst)
{
	struct InterpData *wd = malloc (sizeof (struct InterpData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct InterpData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartInterpolation;
	ep->row = InterpBilinearGRow;
	ep->close = InterpBilinearGClose;
	ep->private = wd;
	wd->next = next;

	wd->cur = wd->old = (struct MRI_ScanLine *)0;
	wd->y = 0;
	wd->redGreen = RGFirst;

	return ep;
}

MRI_Link *
MRI_GenInterpBilinearRGB (MRI_Link *next, int RGFirst)
{
	next = MRI_GenInterpBilinearB (next, RGFirst);
	next = MRI_GenInterpBilinearR (next, RGFirst);
	return MRI_GenInterpBilinearG (next, RGFirst);
}

static void
MRIMedianGRow (void *private, void *data)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	struct MRI_ScanLine *new = data;
	float *G0, *G1, *G2;
	int x;

	if (cur != (struct MRI_ScanLine *)0) {
		G1 = cur->G;
		G2 = new->G;
		if (wd->redGreen) {
			if (old == (struct MRI_ScanLine *)0) {
				G1[0] = (G2[0] + G1[1]) / 2;
				for (x = 2; x < wd->width; x += 2)
					G1[x] = median3 (G1[x-1], G2[x], G1[x+1]);
			}
			else {
				G0 = old->G;
				G1[0] = median3 (G0[0], G1[1], G2[0]);
				for (x = 2; x < wd->width; x += 2)
					G1[x] = median4 (G0[x], G1[x-1], G2[x], G1[x+1]);
			}
		}
		else {
			if (old == (struct MRI_ScanLine *)0) {
				for (x = 1; x != wd->width-1; x += 2)
					G1[x] = median3 (G1[x-1], G2[x], G1[x+1]);
				G1[x] = (G1[x-1] + G2[x]) / 2;
			}
			else {
				G0 = old->G;
				for (x = 1; x != wd->width-1; x += 2)
					G1[x] = median4 (G0[x], G1[x-1], G2[x], G1[x+1]);
				G1[x] = median3 (G0[x], G1[x-1], G2[x]);
			}
		}
		wd->redGreen = !wd->redGreen;
	}
	wd->y++;
	if (old != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, old);
	wd->old = cur;
	wd->cur = new;
}

static void
MRIMedianGClose (void *private)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	int x;
	float *G0, *G1;

	G0 = old->G;
	G1 = cur->G;
	if (wd->redGreen) {
		G1[0] = (G0[0] + G1[1]) / 2;
		for (x = 2; x < wd->width; x += 2)
			G1[x] = median3 (G0[x], G1[x-1], G1[x+1]);
	}
	else {
		for (x = 1; x != wd->width-1; x += 2)
			G1[x] = median3 (G0[x], G1[x-1], G1[x+1]);
		G1[x] = (G0[x] + G1[x-1])/2;
	}
	(*wd->next->row) (wd->next->private, old);
	(*wd->next->row) (wd->next->private, cur);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static MRI_Link *
MRI_GenInterpMedianG (MRI_Link *next, int RGFirst)
{
	struct InterpData *wd = malloc (sizeof (struct InterpData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct InterpData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartInterpolation;
	ep->row = MRIMedianGRow;
	ep->close = MRIMedianGClose;
	ep->private = wd;
	wd->next = next;

	wd->cur = wd->old = (struct MRI_ScanLine *)0;
	wd->y = 0;
	wd->redGreen = RGFirst;

	return ep;
}


MRI_Link *
MRI_GenInterpMedianGBilinearRB (MRI_Link *next, int RGFirst)
{
	next = MRI_GenInterpBilinearB (next, RGFirst);
	next = MRI_GenInterpBilinearR (next, RGFirst);
	return MRI_GenInterpMedianG (next, RGFirst);
}

static void
MRIHueRRow (void *private, void *data)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	struct MRI_ScanLine *new = data;
	int x;

	if (cur != (struct MRI_ScanLine *)0) {
		if (wd->redGreen) {
			/* RG... row */
			float *cR = cur->R;
			float *cG = cur->G;
			for (x = 1; x != wd->width-1; x += 2)
				cR[x] = sub (cG[x] + (cR[x-1] + cR[x+1]) / 2, (cG[x-1] + cG[x+1])/2);
			cR[x] = sub (cG[x] + cR[x-1], cG[x-1]);
		}
		else if (wd->old == (struct MRI_ScanLine *)0) {
			/* GB... row iff first processed */
			float *cR = cur->R;
			float *cG = cur->G;
			float *nR = new->R;
			float *nG = new->G;
			cR[0] = sub (cG[0] + nR[0], nG[0]);
			for (x = 1; x != wd->width-1; x += 2) {
				cR[x] = add (cG[x],
					       ( nR[x-1]-nG[x-1] + nR[x+1]-nG[x+1])/2);
				cR[x+1] = sub(cG[x+1] + nR[x+1], nG[x+1]);
			}
			cR[x] = sub (cG[x] + nR[x-1], nG[x-1]);
		}
		else {
			/* GB... row after first row processed. */
			float *cR = cur->R;
			float *cG = cur->G;
			float *nR = new->R;
			float *nG = new->G;
			float *oR = old->R;
			float *oG = old->G;
			cR[0] = sub (cG[0] + (oR[0] + nR[0])/2, (oG[0] + nG[0])/2);
			for (x = 1; x != wd->width-1; x += 2) {
				cR[x] = add (cG[x],
					       median4 (oR[x-1]-oG[x-1], oR[x+1]-oG[x+1],
							nR[x-1]-nG[x-1], nR[x+1]-nG[x+1]));
				cR[x+1] = sub(cG[x+1] + (oR[x+1] + nR[x+1])/2, (oG[x+1] + nG[x+1])/2);
			}
			cR[x] = sub (cG[x] + (oR[x-1] + nR[x-1]) / 2, (oG[x-1] + nG[x-1])/2);
		}
		wd->redGreen = !wd->redGreen;
	}
	wd->y++;
	if (old != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, old);
	wd->old = cur;
	wd->cur = new;
}


static void
MRIHueRClose (void *private)
{
	struct InterpData *wd = private;
	struct MRI_ScanLine *old = wd->old;
	struct MRI_ScanLine *cur = wd->cur;
	int x;

	/* Last row. */
	if (wd->redGreen) {
		/* RG... row */
		float *cR = cur->R;
		float *cG = cur->G;
		for (x = 1; x != wd->width-1; x += 2)
			cR[x] = sub (cG[x] + (cR[x-1] + cR[x+1]) / 2, (cG[x-1] + cG[x+1])/2);
		cR[x] = sub (cG[x] + cR[x-1], cG[x-1]);
	}
	else {
		/* GB... row */
		float *cR = cur->R;
		float *cG = cur->G;
		float *oR = old->R;
		float *oG = old->G;
		cR[0] = sub (cG[0] + oR[0], oG[0]);
		for (x = 1; x != wd->width-1; x += 2) {
			cR[x] = add (cG[x],
				       ( oR[x-1]-oG[x-1] + oR[x+1]-oG[x+1])/2);
			cR[x+1] = sub(cG[x+1] + oR[x+1], oG[x+1]);
		}
		cR[x] = sub (cG[x] + oR[x-1], oG[x-1]);
	}
	(*wd->next->row) (wd->next->private, old);
	(*wd->next->row) (wd->next->private, cur);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

static MRI_Link *
MRI_GenInterpHueR (MRI_Link *next, int RGFirst)
{
	struct InterpData *wd = malloc (sizeof (struct InterpData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct InterpData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartInterpolation;
	ep->row = MRIHueRRow;
	ep->close = MRIHueRClose;
	ep->private = wd;
	wd->next = next;

	wd->cur = wd->old = (struct MRI_ScanLine *)0;
	wd->y = 0;
	wd->redGreen = RGFirst;

	return ep;
}

MRI_Link *
MRI_GenInterpMedianGHueRBilinearB (MRI_Link *next, int RGFirst)
{
	next = MRI_GenInterpBilinearB (next, RGFirst);
	next = MRI_GenInterpHueR (next, RGFirst);
	return MRI_GenInterpMedianG (next, RGFirst);
}

/******************************************************************************/

struct MedianFilterData {
	struct link *next;
	int	width;
	int	height;
	int	freedata;
	struct MRI_ScanLine *buf[5];
};

static void
StartMedianFilter (void *private, int width, int height, int freedata)
{
	struct MedianFilterData *wd = private;

	wd->width = width;
	wd->height = height;
	wd->freedata = freedata;
	(*wd->next->start) (wd->next->private, width, height, TRUE);
}

static void
MedianFilterRow (void *private, void *data)
{
	struct MedianFilterData *wd = private;
	int i, x;

	wd->buf[4] = (struct MRI_ScanLine *)data;
	if (wd->buf[0] != (struct MRI_ScanLine *)0) {
		struct MRI_ScanLine *out = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
		float *R[6], *G[6], *B[6];
		for (x = 0; x < 5; x++) {
			R[x] = wd->buf[x]->R;
			G[x] = wd->buf[x]->G;
			B[x] = wd->buf[x]->B;
		}
		R[5] = out->R;
		G[5] = out->G;
		B[5] = out->B;
		for (x = 0; x < 2; x++) {
			R[5][x] = R[2][x];
			G[5][x] = G[2][x];
			B[5][x] = B[2][x];
		}
		for (x = 2; x < wd->width-2; x++) {
			R[5][x] = median5 (R[2][x],
median5(R[0][x-2], R[1][x-1], R[2][x], R[3][x+1], R[4][x+2]),
median5(R[0][x], R[1][x], R[2][x], R[3][x], R[4][x]),
median5(R[2][x-2], R[2][x-1], R[2][x], R[2][x+1], R[2][x+2]),
median5(R[0][x+2], R[1][x+1], R[2][x], R[3][x-1], R[4][x-2]));
			G[5][x] = median5 (G[2][x],
median5(G[0][x-2], G[1][x-1], G[2][x], G[3][x+1], G[4][x+2]),
median5(G[0][x], G[1][x], G[2][x], G[3][x], G[4][x]),
median5(G[2][x-2], G[2][x-1], G[2][x], G[2][x+1], G[2][x+2]),
median5(G[0][x+2], G[1][x+1], G[2][x], G[3][x-1], G[4][x-2]));
			B[5][x] = median5 (B[2][x],
median5(B[0][x-2], B[1][x-1], B[2][x], B[3][x+1], B[4][x+2]),
median5(B[0][x], B[1][x], B[2][x], B[3][x], B[4][x]),
median5(B[2][x-2], B[2][x-1], B[2][x], B[2][x+1], B[2][x+2]),
median5(B[0][x+2], B[1][x+1], B[2][x], B[3][x-1], B[4][x-2]));
		}
		for (x = wd->width-2; x < wd->width; x++) {
			R[5][x] = R[2][x];
			G[5][x] = G[2][x];
			B[5][x] = B[2][x];
		}
		(*wd->next->row) (wd->next->private, out);
		if (wd->freedata)
			MRI_FreeScanLine (wd->buf[0]);
	}
	else if (wd->buf[2] != (struct MRI_ScanLine *)0) {
		struct MRI_ScanLine *out = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
		float *R = wd->buf[2]->R;
		float *G = wd->buf[2]->G;
		float *B = wd->buf[2]->B;
		for (x = 0; x < wd->width; x++) {
			((float *)(out->R))[x] = R[x];
			((float *)(out->G))[x] = G[x];
			((float *)(out->B))[x] = B[x];
		}
		(*wd->next->row) (wd->next->private, out);
	}

	for (i = 0; i < 4; i++)
		wd->buf[i] = wd->buf[i+1];
}

static void
CloseMedianFilter (void *private)
{
	struct MedianFilterData *wd = private;
	int i;

	for (i = 2; i < 4; i++) {
		float *R = wd->buf[i]->R;
		float *G = wd->buf[i]->G;
		float *B = wd->buf[i]->B;
		struct MRI_ScanLine *out = MRI_NewScanLine (LINETYPE_FLOAT, wd->width);
		int x;
		for (x = 0; x < wd->width; x++) {
			((float *)(out->R))[x] = R[x];
			((float *)(out->G))[x] = G[x];
			((float *)(out->B))[x] = B[x];
		}
		(*wd->next->row) (wd->next->private, out);
	}
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}


MRI_Link *
MRI_GenMedianFilter (MRI_Link *next)
{
	int i;
	struct MedianFilterData *wd = malloc (sizeof (struct MedianFilterData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MedianFilterData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	ep->start = StartMedianFilter;
	ep->row = MedianFilterRow;
	ep->close = CloseMedianFilter;
	ep->private = wd;
	wd->next = next;

	for (i = 0; i < 5; i++)
		wd->buf[i] = (struct MRI_ScanLine *)0;

	return ep;
}
