/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <math.h>
#include "MRI.h"
#include "vmedian.h"
#include <lcms.h>

struct MRIPPEData {
	struct link *next;
	int	iwidth, owidth;
	int	iheight, oheight;
	int	freedata;
	int	redGreen;
	int	type;
	struct MRI_ScanLine *a, *b;
	struct MRI_ScanLine *bsave;
};

static void
MRIPPEStart (void *private, int width, int height, int freedata)
{
	struct MRIPPEData *wd = private;

	wd->iwidth = width;
	wd->iheight = height;
	switch (wd->type) {
	case MRI_PSEUDO_ORIG:
	    wd->owidth = width-1;
	    wd->oheight = (height-1)*2;
	    break;
	case MRI_PSEUDO_SMOOTH:
	    wd->owidth = width;
	    wd->oheight = (height-1)*2;
	    break;
	default:
	    break;
	}
	(*wd->next->start) (wd->next->private, wd->owidth, wd->oheight, TRUE);
	wd->freedata = freedata;
	wd->a = (struct MRI_ScanLine *)0;
	wd->b = (struct MRI_ScanLine *)0;
	wd->bsave = (struct MRI_ScanLine *)0;
}

static void
MRIPPERow (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			unsigned short *wdbG = wd->b->G;
			unsigned short *wdaR = wd->a->R;
			unsigned short *wdbR = wd->b->R;
			unsigned short *slG = sl->G;
			unsigned short *slR = sl->R;
			wdbR[0] = wdaR[0] = slR[0];
			for (x = 1; x < wd->iwidth-2; x+=2) {
				wdaR[x] = wdaR[x+1] = slR[x+1];
				wdbR[x] = wdbR[x+1] = slR[x+1];
			}
			for (x = 0; x < wd->iwidth-2; x+=2)
				wdbG[x] = wdbG[x+1] = slG[x+1];
			wdbG[wd->iwidth-2] = slG[wd->iwidth-1];
		}
		else {
			unsigned short *wdbG = wd->b->G;
			unsigned short *wdaB = wd->a->B;
			unsigned short *wdbB = wd->b->B;
			unsigned short *slG = sl->G;
			unsigned short *slB = sl->B;
			wdbG[0] = slG[0];
			for (x = 1; x < wd->iwidth-2; x+=2)
				wdbG[x] = wdbG[x+1] = slG[x+1];
			for (x = 0; x < wd->iwidth-2; x+=2) {
				wdaB[x] = wdaB[x+1] = slB[x+1];
				wdbB[x] = wdbB[x+1] = slB[x+1];
			}
			wdbB[wd->iwidth-2] = wdaB[wd->iwidth-2] = slB[wd->iwidth-1];
		}
		(*wd->next->row) (wd->next->private, wd->a);
		(*wd->next->row) (wd->next->private, wd->b);
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	if (wd->redGreen) {
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdaR = wd->a->R;
		unsigned short *wdbR = wd->b->R;
		unsigned short *slR = sl->R;
		unsigned short *slG = sl->G;
		wdbR[0] = wdaR[0] = slR[0];
		for (x = 1; x < wd->iwidth-2; x+=2) {
			wdaR[x] = wdaR[x+1] = slR[x+1];
			wdbR[x] = wdbR[x+1] = slR[x+1];
		}
		for (x = 0; x < wd->iwidth-2; x+=2)
			wdaG[x] = wdaG[x+1] = slG[x+1];
		wdaG[wd->iwidth-2] = slG[wd->iwidth-1];
	}
	else {
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdaB = wd->a->B;
		unsigned short *wdbB = wd->b->B;
		wdaG[0] = ((unsigned short *)(sl->G))[0];
		for (x = 1; x < wd->iwidth-2; x+=2)
			wdaG[x] = wdaG[x+1] = ((unsigned short *)(sl->G))[x+1];
		for (x = 0; x < wd->iwidth-2; x+=2) {
			unsigned short val = ((unsigned short *)(sl->B))[x+1];
			wdaB[x] = wdaB[x+1] = val;
			wdbB[x] = wdbB[x+1] = val;
		}
		wdbB[wd->iwidth-2] = wdaB[wd->iwidth-2] = ((unsigned short *)(sl->B))[wd->iwidth-1];
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPERowSmooth (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			/* RGRGRG...RGRG */
			unsigned int R0, R1, G0, G1;
			unsigned short *wdaR = wd->a->R;
			unsigned short *wdbR = wd->b->R;
			unsigned short *wdbG = wd->b->G;

			R0 = ((unsigned short *)(sl->R))[0];
			x = 0;
			while (x < wd->iwidth-2) {
				R1 = ((unsigned short *)(sl->R))[x+2];
				wdaR[x] = (wdaR[x] + R0) >> 1;
				wdaR[x+1] = ((wdaR[x+1] << 1) + R0 + R1) >> 2;
				wdbR[x] = (wdbR[x] + R0*3) >> 2;
				wdbR[x+1] = (wdbR[x+1] + R0 + R1 + ((R0 + R1) >> 1)) >> 2;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					unsigned short *wdbsR = wd->bsave->R;
					wdbsR[x] = (wdbsR[x]*3 + R0) >> 2;
					wdbsR[x+1] = (wdbsR[x+1]*3 + ((R0 + R1) >> 1)) >> 2;
				}
				R0 = R1;
				x += 2;
			}
			wdbR[x] = wdaR[x] = R0;
			wdbR[x+1] = wdaR[x+1] = R0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				unsigned short *wdbsR = wd->bsave->R;
				wdbsR[x] = (wdbsR[x]*3 + R0) >> 2;
				wdbsR[x+1] = (wdbsR[x+1]*3 + R0) >> 2;
			}

			G0 = ((unsigned short *)(sl->G))[1] >> 1;
			wdbG[0] += G0;
			wdbG[1] += G0;
			x = 2;
			while (x < wd->iwidth) {
				unsigned int G2;
				G1 = ((unsigned short *)(sl->G))[x+1] >> 1;
				G2 = wdbG[x] + ((G0 + G1) >> 1);
				wdbG[x] = G2 > 65535 ? 65535 : G2;
				G2 = wdbG[x+1] + G1;
				wdbG[x+1] = G2 > 65535 ? 65535 : G2;
				G0 = G1;
				x += 2;
			}
		}
		else {
			/* GBGBGB...GBGB */
			int G0, G1, B0, B1;
			unsigned short *wdbG = wd->b->G;
			unsigned short *wdaB = wd->a->B;
			unsigned short *wdbB = wd->b->B;

			x = 0;
			G0 = ((unsigned short *)(sl->G))[0] >> 1;
			while (x < wd->iwidth-2) {
				G1 = ((unsigned short *)(sl->G))[x+2] >> 1;
				wdbG[x] += G0;
				wdbG[x+1] += (G0 + G1) >> 1;
				G0 = G1;
				x += 2;
			}
			wdbG[x] = wdbG[x+1] = G0;

			B0 = ((unsigned short *)(sl->B))[1];
			wdaB[0] = wdaB[1] = B0;
			wdbB[0] = wdbB[1] = B0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				unsigned short *wdbsB = wd->bsave->B;
				wdbsB[0] = (wdbsB[0]*3 + B0) >> 2;
				wdbsB[1] = (wdbsB[1]*3 + B0) >> 2;
			}
			x = 2;
			while (x < wd->iwidth) {
				B1 = ((unsigned short *)(sl->B))[x+1];
				wdaB[x] = (B0 + B1) >> 1;
				wdaB[x+1] = B1;
				wdbB[x] = (B0 + B1) >> 1;
				wdbB[x+1] = B1;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					unsigned short *wdbsB = wd->bsave->B;
					wdbsB[x] = (wdbsB[x]*3 + ((B0 + B1) >> 1)) >> 2;
					wdbsB[x+1] = (wdbsB[x+1]*3 + B1) >> 2;
				}
				B0 = B1;
				x += 2;
			}
		}
		if (wd->bsave != (struct MRI_ScanLine *)0)
			(*wd->next->row) (wd->next->private, wd->bsave);
		(*wd->next->row) (wd->next->private, wd->a);
		wd->bsave = wd->b;
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	if (wd->redGreen) {
		int R0, R1, G0, G1;
		unsigned short *wdaR = wd->a->R;
		unsigned short *wdbR = wd->b->R;
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			unsigned short *wdbsB = wd->bsave->B;
			unsigned short *wdaB = wd->a->B;
			unsigned short *wdbB = wd->b->B;
			x = 0;
			while (x < wd->iwidth) {
				wdaB[x] = wdbB[x] = wdbsB[x];
				x++;
			}
		}

		R0 = ((unsigned short *)(sl->R))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			R1 = ((unsigned short *)(sl->R))[x+2];
			wdaR[x] = R0;
			wdaR[x+1] = (R0 + R1) >> 1;
			wdbR[x] = R0;
			wdbR[x+1] = (R0 + R1) >> 1;
			R0 = R1;
			x += 2;
		}
		wdaR[x] = wdaR[x+1] = R0;
		wdbR[x] = wdbR[x+1] = R0;

		G0 = ((unsigned short *)(sl->G))[1];
		wdaG[0] = wdaG[1] = G0;
		wdbG[0] = wdbG[1] = G0 >> 1;
		x = 2;
		while (x < wd->iwidth) {
			G1 = ((unsigned short *)(sl->G))[x+1];
			wdaG[x] = (G0 + G1) >> 1;
			wdaG[x+1] = G1;
			wdbG[x] = (G0 + G1) >> 2;
			wdbG[x+1] = G1 >> 1;
			G0 = G1;
			G0 = G1;
			x+=2;
		}
	}
	else {
		/* GBGBGB...GBGBGB */
		int G0, G1, B0, B1;
		unsigned short *wdaB = wd->a->B;
		unsigned short *wdbB = wd->b->B;
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			unsigned short *wdbsR = wd->bsave->R;
			unsigned short *wdaR = wd->a->R;
			unsigned short *wdbR = wd->b->R;
			x = 0;
			while (x < wd->iwidth) {
				wdaR[x] = wdbR[x] = wdbsR[x];
				x++;
			}
		}

		G0 = ((unsigned short *)(sl->G))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			G1 = ((unsigned short *)(sl->G))[x+2];
			wdaG[x] = G0;
			wdaG[x+1] = (G0 + G1) >> 1;
			wdbG[x] = G0 >> 1;
			wdbG[x+1] = (G0 + G1) >> 2;
			G0 = G1;
			x+=2;
		}
		wdaG[x] = wdaG[x+1] = G0;
		wdbG[x] = wdbG[x+1] = G0 >> 1;

		B0 = ((unsigned short *)(sl->B))[1];
		wdaB[0] = wdaB[1] = B0;
		wdbB[0] = wdbB[1] = B0;
		x = 2;
		while (x < wd->iwidth) {
			B1 = ((unsigned short *)(sl->B))[x+1];
			wdaB[x] = (B0 + B1) >> 1;
			wdaB[x+1] = B1;
			wdbB[x] = (B0 + B1) >> 1;
			wdbB[x+1] = B1;
			B0 = B1;
			x += 2;
		}
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPERowDouble (void *private, void *data)
{
	struct MRIPPEData *wd = private;
	struct MRI_ScanLine *sl = data;
	int x;

	if (wd->a != (struct MRI_ScanLine *)0) {
		/* Add pixels to previous pseudo-pixel lines and output. */
		if (wd->redGreen) {
			/* RGRGRG...RGRG */
			unsigned int R0, R1, G0, G1;
			unsigned short *wdaR = wd->a->R;
			unsigned short *wdbR = wd->b->R;
			unsigned short *wdbG = wd->b->G;

			R0 = ((unsigned short *)(sl->R))[0];
			x = 0;
			while (x < wd->iwidth-2) {
				R1 = ((unsigned short *)(sl->R))[x+2];
				wdaR[x] = (wdaR[x] + R0) >> 1;
				wdaR[x+1] = ((wdaR[x+1] << 1) + R0 + R1) >> 2;
				wdbR[x] = (wdbR[x] + R0*3) >> 2;
				wdbR[x+1] = (wdbR[x+1] + R0 + R1 + ((R0 + R1) >> 1)) >> 2;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					unsigned short *wdbsR = wd->bsave->R;
					wdbsR[x] = (wdbsR[x]*3 + R0) >> 2;
					wdbsR[x+1] = (wdbsR[x+1]*3 + ((R0 + R1) >> 1)) >> 2;
				}
				R0 = R1;
				x += 2;
			}
			wdbR[x] = wdaR[x] = R0;
			wdbR[x+1] = wdaR[x+1] = R0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				unsigned short *wdbsR = wd->bsave->R;
				wdbsR[x] = (wdbsR[x]*3 + R0) >> 2;
				wdbsR[x+1] = (wdbsR[x+1]*3 + R0) >> 2;
			}

			G0 = ((unsigned short *)(sl->G))[1] >> 1;
			wdbG[0] += G0;
			wdbG[1] += G0;
			x = 2;
			while (x < wd->iwidth) {
				G1 = ((unsigned short *)(sl->G))[x+1] >> 1;
				wdbG[x] += (G0 + G1) >> 1;
				wdbG[x+1] += G1;
				G0 = G1;
				x += 2;
			}
		}
		else {
			/* GBGBGB...GBGB */
			int G0, G1, B0, B1;
			unsigned short *wdbG = wd->b->G;
			unsigned short *wdaB = wd->a->B;
			unsigned short *wdbB = wd->b->B;

			x = 0;
			G0 = ((unsigned short *)(sl->G))[0] >> 1;
			while (x < wd->iwidth-2) {
				G1 = ((unsigned short *)(sl->G))[x+2] >> 1;
				wdbG[x] += G0;
				wdbG[x+1] += (G0 + G1) >> 1;
				G0 = G1;
				x += 2;
			}
			wdbG[x] = wdbG[x+1] = G0;

			B0 = ((unsigned short *)(sl->B))[1];
			wdaB[0] = wdaB[1] = B0;
			wdbB[0] = wdbB[1] = B0;
			if (wd->bsave != (struct MRI_ScanLine *)0) {
				unsigned short *wdbsB = wd->bsave->B;
				wdbsB[0] = (wdbsB[0]*3 + B0) >> 2;
				wdbsB[1] = (wdbsB[1]*3 + B0) >> 2;
			}
			x = 2;
			while (x < wd->iwidth) {
				B1 = ((unsigned short *)(sl->B))[x+1];
				wdaB[x] = (B0 + B1) >> 1;
				wdaB[x+1] = B1;
				wdbB[x] = (B0 + B1) >> 1;
				wdbB[x+1] = B1;
				if (wd->bsave != (struct MRI_ScanLine *)0) {
					unsigned short *wdbsB = wd->bsave->B;
					wdbsB[x] = (wdbsB[x]*3 + ((B0 + B1) >> 1)) >> 2;
					wdbsB[x+1] = (wdbsB[x+1]*3 + B1) >> 2;
				}
				B0 = B1;
				x += 2;
			}
		}
		if (wd->bsave != (struct MRI_ScanLine *)0)
			(*wd->next->row) (wd->next->private, wd->bsave);
		(*wd->next->row) (wd->next->private, wd->a);
		wd->bsave = wd->b;
	}
	/* Add pixels to next two pseudo-pixel rows. */
	wd->a = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	wd->b = MRI_NewScanLine (LINETYPE_SHORT, wd->owidth);
	if (wd->redGreen) {
		int R0, R1, G0, G1;
		unsigned short *wdaR = wd->a->R;
		unsigned short *wdbR = wd->b->R;
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			unsigned short *wdbsB = wd->bsave->B;
			unsigned short *wdaB = wd->a->B;
			unsigned short *wdbB = wd->b->B;
			x = 0;
			while (x < wd->iwidth) {
				wdaB[x] = wdbB[x] = wdbsB[x];
				x++;
			}
		}

		R0 = ((unsigned short *)(sl->R))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			R1 = ((unsigned short *)(sl->R))[x+2];
			wdaR[x] = R0;
			wdaR[x+1] = (R0 + R1) >> 1;
			wdbR[x] = R0;
			wdbR[x+1] = (R0 + R1) >> 1;
			R0 = R1;
			x += 2;
		}
		wdaR[x] = wdaR[x+1] = R0;
		wdbR[x] = wdbR[x+1] = R0;

		G0 = ((unsigned short *)(sl->G))[1];
		wdaG[0] = wdaG[1] = G0;
		wdbG[0] = wdbG[1] = G0 >> 1;
		x = 2;
		while (x < wd->iwidth) {
			G1 = ((unsigned short *)(sl->G))[x+1];
			wdaG[x] = (G0 + G1) >> 1;
			wdaG[x+1] = G1;
			wdbG[x] = (G0 + G1) >> 2;
			wdbG[x+1] = G1 >> 1;
			G0 = G1;
			G0 = G1;
			x+=2;
		}
	}
	else {
		/* GBGBGB...GBGBGB */
		int G0, G1, B0, B1;
		unsigned short *wdaB = wd->a->B;
		unsigned short *wdbB = wd->b->B;
		unsigned short *wdaG = wd->a->G;
		unsigned short *wdbG = wd->b->G;

		if (wd->bsave != (struct MRI_ScanLine *)0) {
			unsigned short *wdbsR = wd->bsave->R;
			unsigned short *wdaR = wd->a->R;
			unsigned short *wdbR = wd->b->R;
			x = 0;
			while (x < wd->iwidth) {
				wdaR[x] = wdbR[x] = wdbsR[x];
				x++;
			}
		}

		G0 = ((unsigned short *)(sl->G))[0];
		x = 0;
		while (x < wd->iwidth-2) {
			G1 = ((unsigned short *)(sl->G))[x+2];
			wdaG[x] = G0;
			wdaG[x+1] = (G0 + G1) >> 1;
			wdbG[x] = G0 >> 1;
			wdbG[x+1] = (G0 + G1) >> 2;
			G0 = G1;
			x+=2;
		}
		wdaG[x] = wdaG[x+1] = G0;
		wdbG[x] = wdbG[x+1] = G0 >> 1;

		B0 = ((unsigned short *)(sl->B))[1];
		wdaB[0] = wdaB[1] = B0;
		wdbB[0] = wdbB[1] = B0;
		x = 2;
		while (x < wd->iwidth) {
			B1 = ((unsigned short *)(sl->B))[x+1];
			wdaB[x] = (B0 + B1) >> 1;
			wdaB[x+1] = B1;
			wdbB[x] = (B0 + B1) >> 1;
			wdbB[x+1] = B1;
			B0 = B1;
			x += 2;
		}
	}
	wd->redGreen = !wd->redGreen;
	if (wd->freedata)
		MRI_FreeScanLine (sl);
}

static void
MRIPPEClose (void *private)
{
	struct MRIPPEData *wd = private;

	if (wd->bsave != (struct MRI_ScanLine *)0)
		(*wd->next->row) (wd->next->private, wd->bsave);
	MRI_FreeScanLine (wd->a);
	MRI_FreeScanLine (wd->b);
	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}


struct link *
MRI_GenPseudoPixelExpander (struct link *next, int RGFirst, int type)
{
	struct MRIPPEData *wd = malloc (sizeof (struct MRIPPEData));
	struct link *ep = malloc (sizeof (*ep));
	if (wd == (struct MRIPPEData *)0 || ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory\n");
		exit (1);
	}
	wd->type = type;
	ep->start = MRIPPEStart;
	switch (type) {
	case MRI_PSEUDO_ORIG:
	    ep->row = MRIPPERow;
	    break;
	case MRI_PSEUDO_SMOOTH:
	    ep->row = MRIPPERowSmooth;
	    break;
	default:
	    break;
	}
	ep->close = MRIPPEClose;
	ep->private = wd;
	wd->next = next;
	wd->redGreen = RGFirst;

	return ep;
}
