/*
 *   Written by Bradley Broom (2002).
 *
 *   Copyright (c) 2002 Bradley Broom
 *
 *   This program is free software; you can redistribute it and/or modify
 *   it under the terms of the GNU General Public License as published by
 *   the Free Software Foundation; either version 2, or (at your option)
 *   any later version.
 *
 *   This program is distributed in the hope that it will be useful,
 *   but WITHOUT ANY WARRANTY; without even the implied warranty of
 *   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *   GNU General Public License for more details.
 *
 *   You should have received a copy of the GNU General Public License
 *   along with this program; if not, write to the Free Software
 *   Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
 */
#include <stdio.h>
#include <math.h>
#include <unistd.h>
#include <string.h>
#include <lcms.h>

#include <MRI.h>

extern char *progname;

struct DarkFieldData {
	struct link *next;
	MRI	*mri;
	int	width;
	int	height;
	int	byRow;
	int	i, di;
};

static void
DarkFieldStart (void *private, int width, int height, int freedata)
{
	struct DarkFieldData *wd = private;

	(*wd->next->start) (wd->next->private, width, height, freedata);
	wd->width = width;
	wd->height = height;
}

static void
DarkFieldRow (void *private, void *data)
{
	struct DarkFieldData *wd = private;
	struct MRI_ScanLine *sl = data;
	struct MRI_ScanLine *df;
	unsigned short *in[3], *sub[3];
	int x;

	if (wd->byRow)
	  df = MRI_GetRow (wd->mri, wd->i, 0, wd->width);
	else
	  df = MRI_GetCol (wd->mri, wd->i, 0, wd->height);
	wd->i += wd->di;
	in[0] = sl->R; sub[0] = df->R;
	in[1] = sl->G; sub[1] = df->G;
	in[2] = sl->B; sub[2] = df->B;
	for (x = 0; x < wd->width; x++) {
		in[0][x] = in[0][x] > sub[0][x] ?  in[0][x] - sub[0][x] : 0;
		in[1][x] = in[1][x] > sub[1][x] ?  in[1][x] - sub[1][x] : 0;
		in[2][x] = in[2][x] > sub[2][x] ?  in[2][x] - sub[2][x] : 0;
	}
	(*wd->next->row) (wd->next->private, sl);
}

static void
DarkFieldClose (void *private)
{
	struct DarkFieldData *wd = private;

	(*wd->next->close) (wd->next->private);
	free (wd->next);
	free (wd);
}

struct link *
MRI_GenDarkFieldFilter (MRI *darkField, int rotate, MRI_balance bal, int dFactor, MRI *mp, struct link *next)
{
	FILE	*f;
	struct DarkFieldData *wd = malloc (sizeof (struct DarkFieldData));
	struct link *ep = malloc (sizeof (*ep));
	char *errmsg;

	if (wd == (struct DarkFieldData *)0 ||
	    ep == (struct link *)0) {
		fprintf (stderr, "Error: unable to allocate memory!\n");
		exit (1);
	}
	wd->mri = darkField;
	MRI_SetBalance (wd->mri, bal);

	fprintf (stderr, "%s: Info: Dark field image from camera %s\n", progname, MRI_GetCameraName (wd->mri));
	fprintf (stderr, "%s: Info: Dark field size = %dx%d\n", progname, MRI_GetHeight(wd->mri), MRI_GetWidth(wd->mri));

	if (MRI_GetWidth(wd->mri) != mp->width || MRI_GetHeight(wd->mri) != mp->height) {
		fprintf (stderr, "%s: Error: Dark field has different dimensions to input image.\n", progname);
		exit (1);
	}

	switch (rotate) {
	case 0: wd->byRow = TRUE; wd->i = 0; wd->di = 1; break;
	case 90: wd->byRow = FALSE; wd->i = 0; wd->di = 1; break;
	case 180: wd->byRow = TRUE; wd->i = wd->mri->height-1; wd->di = -1; break;
	case 270: wd->byRow = FALSE; wd->i = wd->mri->width-1; wd->di = -1; break;
	default:
		fprintf (stderr, "%s: Error: rotate must be 0, 90, 180, or 270\n", progname);
		break;
	}
	ep->start = DarkFieldStart;
	ep->row = DarkFieldRow;
	ep->close = DarkFieldClose;
	ep->private = wd;
	wd->next = next;

	return ep;
}
