# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Dialog with statistical calibration.

# load libs
import wx

# load modules
import count
from nucleus import mwx
from dlg_calibration import dlgCalibration


class dlgStatCalib(dlgCalibration):
    """ Statistical calibration. """

    # ----
    def __init__(self, parent, peaklist, config):

        # init base class
        self.digits = config.cfg['common']['digits']
        self.grayColour = config.cfg['colours']['grayed']
        dlgCalibration.__init__(self, parent, title="Statistical Calibration", calibType='statistical', grayColour=self.grayColour)

        # calculate peaklist
        for peak in peaklist:
            decimal = count.calcDecimal(peak[0])
            diff = count.calcDecimalDiff(peak[0], decimal)
            self.peaklist.append([True, peak[0], decimal, None, diff])

        # make items
        self.makePeakList()
        calibration = self.makeCalibrationBox()
        results = self.makeResultsBox()
        zoom = self.makeZoomBox()
        buttons = self.makeButtonBox()

        # pack main frame
        colSizer = wx.BoxSizer(wx.VERTICAL)
        rowSizer = wx.BoxSizer(wx.HORIZONTAL)
        mainSizer = wx.BoxSizer(wx.VERTICAL)

        if wx.Platform == '__WXMAC__':
            colSizer.Add(calibration, 0, wx.EXPAND|wx.BOTTOM, 5)
            colSizer.Add(results, 0, wx.EXPAND|wx.BOTTOM, 5)
            colSizer.Add(zoom, 0, wx.EXPAND)
            rowSizer.Add(colSizer, 0, wx.EXPAND|wx.RIGHT, 20)
            rowSizer.Add(self.peaklist_list, 1, wx.EXPAND)
            mainSizer.Add(rowSizer, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALIGN_CENTER|wx.ALL, 20)
        else:
            colSizer.Add(calibration, 0, wx.EXPAND|wx.BOTTOM, 5)
            colSizer.Add(results, 0, wx.EXPAND|wx.BOTTOM, 5)
            colSizer.Add(zoom, 0, wx.EXPAND)
            rowSizer.Add(colSizer, 0, wx.EXPAND|wx.RIGHT, 10)
            rowSizer.Add(self.peaklist_list, 1, wx.EXPAND)
            mainSizer.Add(rowSizer, 1, wx.EXPAND|wx.ALL, 5)
            mainSizer.Add(buttons, 0, wx.ALIGN_CENTER|wx.ALL, 5)

        # update peaklist
        self.updatePeaklist()
        self.updatePeaksCount()

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makePeakList(self):
        """ Make box with peaks. """

        # set style
        if wx.Platform == '__WXMAC__':
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL|wx.SIMPLE_BORDER
        else:
            style = wx.LC_REPORT|wx.LC_VRULES|wx.LC_HRULES|wx.LC_SINGLE_SEL

        # make items
        self.peaklist_list = mwx.ListCtrl(self, -1, size=(310, 300), style=style)
        self.peaklist_list.InsertColumn(0, "#", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(1, "Status", wx.LIST_FORMAT_CENTER)
        self.peaklist_list.InsertColumn(2, "m/z", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(3, "Decimal", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(4, "Calib.", wx.LIST_FORMAT_RIGHT)
        self.peaklist_list.InsertColumn(5, "Error [Da]", wx.LIST_FORMAT_RIGHT)

        # set events
        self.peaklist_list.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onPeakSelected)
        self.peaklist_list.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onPeakActivated)

        # set columns width
        for col in range(6):
            self.peaklist_list.SetColumnWidth(col, wx.LIST_AUTOSIZE_USEHEADER)
    # ----


    # ----
    def updatePeaklist(self):
        """ Update list of loaded peaks. """

        # clear list
        self.peaklist_list.DeleteAllItems()

        # paste new data
        digitsFormat = '%0.' + `self.digits` + 'f'
        for x, peak in enumerate(self.peaklist):

            # format data
            use = "skip"
            measured = digitsFormat % peak[1]
            decimal = ''
            calibrated = ''
            error = ''
            if peak[0]:
                use = "use"
            if peak[2] != None:
                decimal = digitsFormat % peak[2]
            if peak[3] != None:
                calibrated = digitsFormat % peak[3]
            if peak[4] != None:
                error = digitsFormat % peak[4]

            # show data
            self.peaklist_list.InsertStringItem(x, str(x+1))
            self.peaklist_list.SetStringItem(x, 1, use)
            self.peaklist_list.SetStringItem(x, 2, measured)
            self.peaklist_list.SetStringItem(x, 3, decimal)
            self.peaklist_list.SetStringItem(x, 4, calibrated)
            self.peaklist_list.SetStringItem(x, 5, error)

            # set colour
            if not peak[0]:
                self.peaklist_list.SetItemTextColour(x, self.grayColour)

        # set columns width
        self.peaklist_list.SetColumnWidth(0, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(1, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklist_list.SetColumnWidth(2, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(3, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklist_list.SetColumnWidth(4, wx.LIST_AUTOSIZE)
        self.peaklist_list.SetColumnWidth(5, wx.LIST_AUTOSIZE_USEHEADER)
        self.peaklist_list.updateLastCol()
    # ----


    # ----
    def onPeakSelected(self, evt):
        """ Highlight selected point in the spectrum. """

        # get and show selected peak
        self.selected = evt.m_itemIndex
        peak = self.peaklist[self.selected][1]
        self.highlightPoint(peak)
    # ----


    # ----
    def onClear(self, evt):
        """ Clear calibration data. """

        self.fit = []
        self.Clear_button.Enable(False)
        self.OK_button.Enable(False)

        # clear peaklist data
        for x, peak in enumerate(self.peaklist):
            peak[3] = None
            peak[4] = count.calcDecimalDiff(peak[1], peak[2])

        # update peaklist and results
        self.updatePeaklist()
        self.updateResults()
    # ----
