# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: View calibration plot.

# load libs
import wx

# load modules
from nucleus import plot
from nucleus import commfce


class calibPlot(wx.Dialog):
    """View calibration plot"""

    # ----
    def __init__(self, parent, data):
        wx.Dialog.__init__(self, parent, -1, "Calibration Plot", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        # make items
        self.makePlotCanvas()
        buttons = self.makeButtons()

        # make plot
        self.data = data
        self.updatePlot()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.plotCanvas, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_RIGHT, 20)
        else:
            mainSizer.Add(self.plotCanvas, 1, wx.EXPAND|wx.LEFT|wx.RIGHT|wx.BOTTOM, 3)
            mainSizer.Add(buttons, 0, wx.ALL|wx.ALIGN_RIGHT, 5)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()
    # ----


    # ----
    def makePlotCanvas(self):
        """ Make error-distribution plot. """

        # initialize plotcanvas
        if wx.Platform == '__WXMAC__':
            self.plotCanvas = plot.plotCanvas(self, size=(440, 260), style=wx.SIMPLE_BORDER)
        else:
            self.plotCanvas = plot.plotCanvas(self, size=(440, 260), style=wx.SUNKEN_BORDER)

        self.plotCanvas.setXLabel("m/z")
        self.plotCanvas.setYLabel("error")
    # ----


    # ----
    def makeButtons(self):
        """ Make box with error type switch. """

        # make items
        errors_label = wx.StaticText(self, -1, "Errors in: ")
        self.errorDa_radio = wx.RadioButton(self, -1, "Da", style=wx.RB_GROUP)
        self.errorPpm_radio = wx.RadioButton(self, -1, "ppm")
        self.errorPerc_radio = wx.RadioButton(self, -1, "%")

        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Close")
        Cancel_button.SetDefault()

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(errors_label, 0, wx.ALL, 5)
        mainBox.Add(self.errorDa_radio, 0, wx.ALL, 5)
        mainBox.Add(self.errorPpm_radio, 0, wx.ALL, 5)
        mainBox.Add(self.errorPerc_radio, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.LEFT|wx.RIGHT, 5)

        # set defaults
        self.errorDa_radio.SetValue(True)

        # set events
        self.errorDa_radio.Bind(wx.EVT_RADIOBUTTON, self.updatePlot)
        self.errorPpm_radio.Bind(wx.EVT_RADIOBUTTON, self.updatePlot)
        self.errorPerc_radio.Bind(wx.EVT_RADIOBUTTON, self.updatePlot)

        return mainBox
    # ----


    # ----
    def updatePlot(self, evt=None):
        """ Recalculate and update plot objects. """

        # get units
        units = 'Da'
        if self.errorPpm_radio.GetValue():
            units = 'ppm'
        elif self.errorPerc_radio.GetValue():
            units = '%'

        # recalc measured data
        measured = []
        for point in self.data[0]:
            error = commfce.calcMassError(point[0], point[0]-point[1], units)
            measured.append((point[0], error))

        # recalc calibration curve
        calibration = []
        for point in self.data[1]:
            error = commfce.calcMassError(point[0], point[0]-point[1], units)
            calibration.append((point[0], error))

        # make plot objects
        objects = []
        objects.append(plot.polyMarker(measured, colour='blue', size=1))
        if calibration:
            objects.append(plot.polyLine(calibration, colour='red', width=1))

        # draw plot
        yUnits = 'error [%s]' % units
        plotData = plot.plotGraphics(objects)
        self.plotCanvas.setYLabel(yUnits)
        self.plotCanvas.draw(plotData)
    # ----
