# -------------------------------------------------------------------------
#     This file is part of mMass - the spectrum analysis tool for MS.
#     Copyright (C) 2005-07 Martin Strohalm <mmass@biographics.cz>

#     This program is free software; you can redistribute it and/or modify
#     it under the terms of the GNU General Public License as published by
#     the Free Software Foundation; either version 2 of the License, or
#     (at your option) any later version.

#     This program is distributed in the hope that it will be useful,
#     but WITHOUT ANY WARRANTY; without even the implied warranty of
#     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
#     GNU General Public License for more details.

#     Complete text of GNU GPL can be found in the file LICENSE in the
#     main directory of the program
# -------------------------------------------------------------------------

# Function: Configure available modifications.

# load libs
import re
import wx
import sys
import os
from copy import copy

# load modules
from nucleus import mwx
from modules.mformula.main import mFormula


class mModPref:
    """ Raise edit-dialog and set data to main config. """

    # ----
    def __init__(self, parent, config):

        # raise dialog
        dlg = mModDlg(parent, config)

        # get dialog data
        if dlg.ShowModal() == wx.ID_OK:
            dataNew = dlg.data

            # set data to be actual in config
            config.mod = {}
            for mod in dataNew:
                name = mod[0]
                config.mod[name] = {}
                config.mod[name]['descr'] = mod[1]
                config.mod[name]['formula'] = mod[2]
                config.mod[name]['mmass'] = mod[3]
                config.mod[name]['amass'] = mod[4]
                config.mod[name]['specifity'] = mod[5]
                config.mod[name]['n-term'] = mod[6]
                config.mod[name]['c-term'] = mod[7]
                config.mod[name]['group'] = mod[8]

        # destroy dialog
        dlg.Destroy()
    # ----


class mModDlg(wx.Dialog):
    """ Add/edit/remove available sequence modifications. """

    # ----
    def __init__(self, parent, config):
        wx.Dialog.__init__(self, parent, -1, "Available Modifications", style=wx.DEFAULT_DIALOG_STYLE|wx.RESIZE_BORDER)

        self.config = config

        # init variables
        self.data = []
        self.currentMod = 0

        # copy modifications
        for name in config.mod:
            mod = [name]
            for item in ('descr', 'formula', 'mmass', 'amass', 'specifity', 'n-term', 'c-term', 'group'):
                mod.append(config.mod[name][item])
            self.data.append(mod)

        # make gui items
        self.makeList()

        # load data to list
        self.updateModList()

        # select first
        self.currentMod = 0
        self.modList.SetItemState(self.currentMod, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)
        if wx.Platform == '__WXMAC__':
            mainSizer.Add(self.modList, 1, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 20)
        else:
            mainSizer.Add(self.modList, 1, wx.EXPAND|wx.ALL, 3)
            mainSizer.Add(self.makeButtonBox(), 0, wx.ALL|wx.ALIGN_CENTER, 3)
        self.SetSizer(mainSizer)

        # fit layout
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.SetMinSize(self.GetSize())
        self.Centre()


    # ----
    def makeList(self):
        """ Make list to show available modifications. """

        # make list
        self.modList = mwx.ListCtrl(self, -1, size=(600, 350))

        # create columns
        self.modList.InsertColumn(0, "Name")
        self.modList.InsertColumn(1, "Formula", wx.LIST_FORMAT_LEFT)
        self.modList.InsertColumn(2, "Mo. mass", wx.LIST_FORMAT_RIGHT)
        self.modList.InsertColumn(3, "Av. mass", wx.LIST_FORMAT_RIGHT)
        self.modList.InsertColumn(4, "Specifity", wx.LIST_FORMAT_LEFT)
        self.modList.InsertColumn(5, "Group", wx.LIST_FORMAT_LEFT)
        self.modList.InsertColumn(6, "Description", wx.LIST_FORMAT_LEFT)

        # set events
        self.modList.Bind(wx.EVT_LIST_ITEM_SELECTED, self.onItemSelected)
        self.modList.Bind(wx.EVT_LIST_ITEM_ACTIVATED, self.onLMDC)
    # ----


    # ----
    def makeButtonBox(self):
        """ Make main dialog buttons. """

        # create buttons
        Add_button = wx.Button(self, -1, "Add")
        Edit_button = wx.Button(self, -1, "Edit")
        Delete_button = wx.Button(self, -1, "Delete")
        self.Save_button = wx.Button(self, -1, "Save")
        self.OK_button = wx.Button(self, wx.ID_OK, "Apply")
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # add buttons to box
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(Add_button, 0, wx.ALL, 5)
        mainBox.Add(Edit_button, 0, wx.ALL, 5)
        mainBox.Add(Delete_button, 0, wx.ALL, 5)
        mainBox.Add(self.Save_button, 0, wx.ALL, 5)
        mainBox.Add(self.OK_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set buttons events
        self.Bind(wx.EVT_BUTTON, self.onAddModification, Add_button)
        self.Bind(wx.EVT_BUTTON, self.onEditModification, Edit_button)
        self.Bind(wx.EVT_BUTTON, self.onDeleteModifications, Delete_button)
        self.Bind(wx.EVT_BUTTON, self.onSave, self.Save_button)

        # set defaults
        self.Save_button.Enable(False)
        self.OK_button.Enable(False)

        return mainBox
    # ----


    # ----
    def updateModList(self):
        """ Show modifications in the list. """

        # clear list
        self.modList.DeleteAllItems()

        # sort data
        self.data.sort()
        modifs = len(self.data)

        # paste data
        for row in range(modifs):

            # count masses
            mmass = self.data[row][3]
            mmass = round(mmass, self.config.cfg['common']['digits'])
            amass = self.data[row][4]
            amass = round(amass, self.config.cfg['common']['digits'])

            # format groups
            groups = self.data[row][8]
            if groups[-1] == ';':
                groups = groups[:-1]
            groups = groups.replace(';', ', ')

            # paste data to list
            self.modList.InsertStringItem(row, self.data[row][0])
            self.modList.SetStringItem(row, 1, self.data[row][2])
            self.modList.SetStringItem(row, 2, str(mmass))
            self.modList.SetStringItem(row, 3, str(amass))
            self.modList.SetStringItem(row, 4, self.data[row][5])
            self.modList.SetStringItem(row, 5, groups)
            self.modList.SetStringItem(row, 6, self.data[row][1])

        # set columns width
        for col in range(7):
            self.modList.SetColumnWidth(col, wx.LIST_AUTOSIZE)
        self.modList.SetColumnWidth(2, wx.LIST_AUTOSIZE_USEHEADER)
        self.modList.SetColumnWidth(3, wx.LIST_AUTOSIZE_USEHEADER)
    # ----


    # ----
    def onSave(self, evt):
        """ Save available modifications to config file. """

        # paste data to buffer
        buff = '# list of modificaions'
        for row in range(len(self.data)):
            buff += '\n\n[%s]' % self.data[row][0]
            buff += '\ndescr=%s' % self.data[row][1]
            buff += '\nformula=%s' % self.data[row][2]
            buff += '\nspecifity=%s' % self.data[row][5]
            buff += '\nn-term=%d' % self.data[row][6]
            buff += '\nc-term=%d' % self.data[row][7]
            buff += '\ngroup=%s' % self.data[row][8]

        # get current executing folder and set path
        path = sys.path[0]
        if os.path.isfile(path):
            path = os.path.dirname(path)
        path += '/configs/modifications.ini'

        # paste buffer to file
        try:
            cfgFile = file(path, 'w')
            cfgFile.write(buff)
            cfgFile.close()
            self.Save_button.Enable(False)
        except IOError:
            errorDlg = wx.MessageDialog(self, "Modifications cannot be saved! Check permissions please.", "Save Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
    # ----


    # ----
    def onItemSelected(self, evt):
        """ Get index of currently selected modification. """
        self.currentMod = evt.m_itemIndex
    # ----


    # ----
    def onLMDC(self, evt):
        """ Raise edit dialog when double-click on modification. """
        self.onEditModification(evt)
    # ----


    # ----
    def onAddModification(self, evt):
        """ Raise dialog for adding new modification. """

        # get used names
        usedNames = []
        for row in self.data:
            usedNames.append(row[0])

        # raise dialog
        dlg = dlgAddEditModification(self, "Add New Modification", self.config.elem, usedNames, '')
        if dlg.ShowModal() == wx.ID_OK:

            # get data
            self.data.append(dlg.data)
            currentMod = dlg.data[0]
            dlg.Destroy()

            # update mod list
            self.updateModList()

            # select item
            self.currentMod = self.modList.FindItem(-1, currentMod)
            self.modList.SetItemState(self.currentMod, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
            self.modList.EnsureVisible(self.currentMod)

            # update buttons
            self.Save_button.Enable(True)
            self.OK_button.Enable(True)

        else:
            dlg.Destroy()
    # ----


    # ----
    def onEditModification(self, evt):
        """ Raise dialog for editing selected modification. """

        # get modification data
        data = self.data[self.currentMod]

        # get used names
        usedNames = []
        for row in self.data:
            usedNames.append(row[0])

        # raise dialog
        dlg = dlgAddEditModification(self, "Edit Modification", self.config.elem, usedNames, data)
        if dlg.ShowModal() == wx.ID_OK:

            # delete old version first
            del(self.data[self.currentMod])

            # get data
            self.data.append(dlg.data)
            currentMod = dlg.data[0]
            dlg.Destroy()

            # update mod list
            self.updateModList()

            # select item
            self.currentMod = self.modList.FindItem(-1, currentMod)
            self.modList.SetItemState(self.currentMod, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)
            self.modList.EnsureVisible(self.currentMod)

            # update buttons
            self.Save_button.Enable(True)
            self.OK_button.Enable(True)

        else:
            dlg.Destroy()
    # ----


    # ----
    def onDeleteModifications(self, evt):
        """ Delete selected modification. """

        dlg = wx.MessageDialog(self, "Delete selected modifications?", "Delete Modifications", wx.YES_NO|wx.ICON_QUESTION|wx.YES_DEFAULT)
        if dlg.ShowModal() == wx.ID_YES:
            dlg.Destroy()

            # get selected items
            indexes = mwx.getSelectedListItems(self.modList,  reverse=True)

            # delete selected modifications
            for index in indexes:
                del self.data[index]

            # update mod list
            self.updateModList()

            # select item
            self.currentMod = 0
            self.modList.SetItemState(self.currentMod, wx.LIST_STATE_SELECTED, wx.LIST_STATE_SELECTED)

            # update buttons
            self.Save_button.Enable(True)
            self.OK_button.Enable(True)

        else:
            dlg.Destroy()
    # ----


class dlgAddEditModification(wx.Dialog):
    """ Dialog for adding and editing modification. """

    # ----
    def __init__(self, parent, title, elements, usedNames, data):
        wx.Dialog.__init__(self, parent, -1, title, style=wx.DEFAULT_DIALOG_STYLE)

        self.data = copy(data)
        self.usedNames = copy(usedNames)
        self.mFormula = mFormula(elements)

        # delete current name
        if self.data !='':
            index = self.usedNames.index(self.data[0])
            del(self.usedNames[index])

        # make data structure
        if data == '':
            self.data = ['','','','','','','','','']

        # make gui items
        basics = self.makeBasicBox()
        specifity = self.makeSpecifityBox()
        groups = self.makeGroupBox()
        buttond = self.makeButtonBox()

        # pack main frame
        mainSizer = wx.BoxSizer(wx.VERTICAL)

        if wx.Platform == '__WXMAC__':
            grid = wx.GridBagSizer(8,20)

            grid.Add(basics, (0,0), flag=wx.EXPAND)
            grid.Add(specifity, (1,0), flag=wx.EXPAND)
            grid.Add(groups, (2,0), flag=wx.EXPAND)

            mainSizer.Add(grid, 0, wx.TOP|wx.LEFT|wx.RIGHT, 20)
            mainSizer.Add(buttond, 0, wx.ALL|wx.ALIGN_CENTER, 10)
        else:
            grid = wx.GridBagSizer(4,2)

            grid.Add(basics, (0,0), flag=wx.EXPAND)
            grid.Add(specifity, (1,0), flag=wx.EXPAND)
            grid.Add(groups, (2,0), flag=wx.EXPAND)

            mainSizer.Add(grid, 0, wx.ALL, 3)
            mainSizer.Add(buttond, 0, wx.ALL|wx.ALIGN_CENTER, 5)

        # fit layout
        self.SetSizer(mainSizer)
        mainSizer.Fit(self)
        self.SetSizer(mainSizer)
        self.Centre()

        # set default or loaded values
        self.setDefaults()
    # ----


    # ----
    def makeBasicBox(self):
        """ Make fields for basic modification info. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Define Modification"), wx.VERTICAL)
        grid = mwx.GridBagSizer()

        name_label = wx.StaticText(self, -1, "Name: ")
        self.name_value = wx.TextCtrl(self, -1, '', size=(250, -1))
        self.name_value.SetToolTip(wx.ToolTip("Use 0-9A-Za-z_ only"))

        description_label = wx.StaticText(self, -1, "Description: ")
        self.description_value = wx.TextCtrl(self, -1, '', size=(250, -1))
        self.description_value.SetToolTip(wx.ToolTip("Simple description"))

        formula_label = wx.StaticText(self, -1, "Formula: ")
        formula_info = wx.StaticText(self, -1, " (gain-loss)")
        self.formula_value = wx.TextCtrl(self, -1, '', size=(160, -1), validator=mwx.txtValidator('formula'))

        mass_label = wx.StaticText(self, -1, "Mass: ")
        mass_info = wx.StaticText(self, -1, " (Mo. / Av.)")
        self.mass_value = wx.TextCtrl(self, -1, '', size=(160, -1))
        self.mass_value.Enable(False)

        # pack items
        grid.Add(name_label, (0,0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.name_value, (0,1), (1,2), flag=wx.EXPAND)

        grid.Add(description_label, (1,0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.description_value, (1,1), (1,2), flag=wx.EXPAND)

        grid.Add(formula_label, (2,0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.formula_value, (2,1), flag=wx.EXPAND)
        grid.Add(formula_info, (2,2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.Add(mass_label, (3,0), flag=wx.ALIGN_CENTER_VERTICAL|wx.ALIGN_RIGHT)
        grid.Add(self.mass_value, (3,1), flag=wx.EXPAND)
        grid.Add(mass_info, (3,2), flag=wx.ALIGN_CENTER_VERTICAL)

        grid.AddGrowableCol(2)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(grid, 0, 0)
        else:
            mainBox.Add(grid, 0, wx.ALL, 5)

        # set events
        self.formula_value.Bind(wx.EVT_TEXT, self.onFormulaEditing)

        return mainBox
    # ----


    # ----
    def makeSpecifityBox(self):
        """ Make fields for specifity. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Specifity"), wx.VERTICAL)
        terminusBox = wx.BoxSizer(wx.HORIZONTAL)

        specifity_label = wx.StaticText(self, -1, "Amino acid side chain:")
        self.specifity_value = wx.TextCtrl(self, -1, 'ACDEFGHIKLMNPQRSTVWY', size=(300, -1), validator=mwx.txtValidator('amino'))
        self.specifity_value.SetToolTip(wx.ToolTip("Use single amino acid letter"))

        self.terminusN_check = wx.CheckBox(self, -1, "N-terminus")
        self.terminusN_check.SetToolTip(wx.ToolTip("N-terminal modification allowed"))
        
        self.terminusC_check = wx.CheckBox(self, -1, "C-terminus")
        self.terminusC_check.SetToolTip(wx.ToolTip("C-terminal modification allowed"))

        # pack items
        terminusBox.Add(self.terminusN_check, 0, wx.RIGHT, 5)
        terminusBox.Add(self.terminusC_check, 0, 0)

        if wx.Platform == '__WXMAC__':
            mainBox.Add(specifity_label, 0, 0)
            mainBox.Add(self.specifity_value, 0, wx.EXPAND|wx.TOP, 5)
            mainBox.Add(terminusBox, 0, wx.TOP, 10)
        else:
            mainBox.Add(specifity_label, 0, wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(self.specifity_value, 0, wx.EXPAND|wx.TOP|wx.LEFT|wx.RIGHT, 5)
            mainBox.Add(terminusBox, 0, wx.TOP|wx.LEFT|wx.RIGHT, 5)

        return mainBox
    # ----


    # ----
    def makeGroupBox(self):
        """ Make group selection. """

        # make items
        mainBox = wx.StaticBoxSizer(wx.StaticBox(self, -1, "Group"), wx.HORIZONTAL)

        self.groupPTM_check = wx.CheckBox(self, -1, "PTM")
        self.groupPTM_check.SetToolTip(wx.ToolTip("Post-translational modifications"))
        
        self.groupChemical_check = wx.CheckBox(self, -1, "Chemical")
        self.groupChemical_check.SetToolTip(wx.ToolTip("Chemical derivative"))

        self.groupGlyco_check = wx.CheckBox(self, -1, "Glyco")
        self.groupGlyco_check.SetToolTip(wx.ToolTip("Glycosylation"))

        self.groupArtefact_check = wx.CheckBox(self, -1, "Artefact")
        self.groupArtefact_check.SetToolTip(wx.ToolTip("Preparation artefacts"))

        self.groupOther_check = wx.CheckBox(self, -1, "Other")
        self.groupOther_check.SetToolTip(wx.ToolTip("All others"))

        # pack items
        if wx.Platform == '__WXMAC__':
            mainBox.Add(self.groupPTM_check, 0, wx.RIGHT, 5)
            mainBox.Add(self.groupChemical_check, 0, wx.RIGHT, 5)
            mainBox.Add(self.groupGlyco_check, 0, wx.RIGHT, 5)
            mainBox.Add(self.groupArtefact_check, 0, wx.RIGHT, 5)
            mainBox.Add(self.groupOther_check, 0, 0)
        else:
            mainBox.Add(self.groupPTM_check, 0, wx.ALL, 5)
            mainBox.Add(self.groupChemical_check, 0, wx.ALL, 5)
            mainBox.Add(self.groupGlyco_check, 0, wx.ALL, 5)
            mainBox.Add(self.groupArtefact_check, 0, wx.ALL, 5)
            mainBox.Add(self.groupOther_check, 0, wx.ALL, 5)

        return mainBox
    # ----


    # ----
    def makeButtonBox(self):
        """ Make buttons. """

        # make items
        OK_button = wx.Button(self, wx.ID_OK, "OK")
        OK_button.SetDefault()
        Cancel_button = wx.Button(self, wx.ID_CANCEL, "Cancel")

        # pack items
        mainBox = wx.BoxSizer(wx.HORIZONTAL)
        mainBox.Add(OK_button, 0, wx.ALL, 5)
        mainBox.Add(Cancel_button, 0, wx.ALL, 5)

        # set events
        wx.EVT_BUTTON(self, wx.ID_OK, self.onDlgOK)

        return mainBox
    # ----


    # ----
    def setDefaults(self):
        """ Set values of given modofication. """

        # set default values
        if self.data != ['','','','','','','','','']:
            self.name_value.SetValue(str(self.data[0]))
            self.description_value.SetValue(str(self.data[1]))
            self.formula_value.SetValue(str(self.data[2]))
            self.specifity_value.SetValue(str(self.data[5]))
            if self.data[6]:
                self.terminusN_check.SetValue(True)
            if self.data[7]:
                self.terminusC_check.SetValue(True)
            if 'ptm' in self.data[8]:
                self.groupPTM_check.SetValue(True)
            if 'chemical' in self.data[8]:
                self.groupChemical_check.SetValue(True)
            if 'glyco' in self.data[8]:
                self.groupGlyco_check.SetValue(True)
            if 'artefact' in self.data[8]:
                self.groupArtefact_check.SetValue(True)
            if 'other' in self.data[8]:
                self.groupOther_check.SetValue(True)

        # count formula
        self.onFormulaEditing(0)
    # ----


    # ----
    def onFormulaEditing(self, evt):
        """ Parse formula when typing and update formula info. """

        # get masses
        self.data[2] = self.formula_value.GetValue()
        self.data[3] = self.mFormula.getMass(self.data[2], 'mmass')
        self.data[4] = self.mFormula.getMass(self.data[2], 'amass')

        # display masses
        if self.data[3] and self.data[4]:
            label = '%01.5f / %01.5f' % (self.data[3], self.data[4])
            self.mass_value.SetValue(label)
        else:
            self.mass_value.SetValue(' --- ')
    # ----


    # ----
    def onDlgOK(self, evt):
        """ Check values and close dialog. """

        errorMessage = ''

        # get values
        self.data[0] = self.name_value.GetValue()
        self.data[1] = self.description_value.GetValue()
        self.data[5] = self.specifity_value.GetValue()

        if self.terminusN_check.IsChecked():
            self.data[6] = 1
        else:
            self.data[6] = 0

        if self.terminusC_check.IsChecked():
            self.data[7] = 1
        else:
            self.data[7] = 0

        self.data[8] = ''
        if self.groupPTM_check.IsChecked():
            self.data[8] += 'ptm;'
        if self.groupChemical_check.IsChecked():
            self.data[8] += 'chemical;'
        if self.groupGlyco_check.IsChecked():
            self.data[8] += 'glyco;'
        if self.groupArtefact_check.IsChecked():
            self.data[8] += 'artefact;'
        if self.groupOther_check.IsChecked():
            self.data[8] += 'other;'

        # parse values
        alphanum = re.compile('^[0-9A-Za-z_-]*$')
        amino = re.compile('^[ACDEFGHIKLMNPQRSTVWY]*$')

        if not self.data[0]:
            errorMessage = "Name must be set!"
        elif not alphanum.search(self.data[0]):
            errorMessage = "Name contains illegal characters!\nUse 0-9A-Za-z_ only!"
        elif self.data[0] in self.usedNames:
            errorMessage = "Modification name is not unique!"
        elif not self.data[2]:
            errorMessage = "Formula must be set!"
        elif not self.data[3] or not self.data[4]:
            errorMessage = "Formula is not valid!"
        elif not amino.search(self.data[5]):
            errorMessage = "Use single letters only to define side-chain specifity!"
        elif self.data[8] == '':
            errorMessage = "Select at least one group for modification!"

        # if no error
        if errorMessage == '':
            self.EndModal(wx.ID_OK)
        else:
            errorDlg = wx.MessageDialog(self, errorMessage, "Value Error", wx.OK|wx.ICON_ERROR)
            errorDlg.ShowModal()
            errorDlg.Destroy()
    # ----
