#!/usr/bin/tclsh
# Part of MCU 8051 IDE ( http://mcu8051ide.sf.net )

############################################################################
#    Copyright (C) 2007-2009 by Martin Ošmera                              #
#    martin.osmera@gmail.com                                               #
#                                                                          #
#    This program is free software; you can redistribute it and#or modify  #
#    it under the terms of the GNU General Public License as published by  #
#    the Free Software Foundation; either version 2 of the License, or     #
#    (at your option) any later version.                                   #
#                                                                          #
#    This program is distributed in the hope that it will be useful,       #
#    but WITHOUT ANY WARRANTY; without even the implied warranty of        #
#    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the         #
#    GNU General Public License for more details.                          #
#                                                                          #
#    You should have received a copy of the GNU General Public License     #
#    along with this program; if not, write to the                         #
#    Free Software Foundation, Inc.,                                       #
#    59 Temple Place - Suite 330, Boston, MA  02111-1307, USA.             #
############################################################################

# --------------------------------------------------------------------------
# DESCRIPTION
# Implements global configuration dialog
# --------------------------------------------------------------------------


## Global configuration dialog
 # see Array OPTION in root NS
namespace eval global {

	variable dialog_opened	0	;# Bool: True if this dialog is already opened
	variable win			;# ID of dialog toplevel window

	variable avaliable_languages	;# List of avaliable languages

	## Configuration variables
	variable show_splash		;# Bool: Show splash creen on startup
	variable show_tips		;# Bool: Show tips on startup
	variable language		;# String: Language

	## Create the dialog
	 # @return void
	proc mkDialog {} {
		variable win			;# ID of toplevel dialog window
		variable dialog_opened		;# Bool: True if this dialog is already opened
		variable avaliable_languages	;# List of avaliable languages

		# Destroy the dialog if it's alredy opened
		if {$dialog_opened} {
			destroy .global_config_dialog
		}
		set dialog_opened 1

		# Get settings from main NS
		getSettings

		# Determinate avaliable languages
		get_languages

		# Create toplevel window
		set win [toplevel .global_config_dialog -class {Configuration dialog} -bg {#EEEEEE}]

		# Create window header
		label $win.header_label				\
			-compound left				\
			-image ::ICONS::32::kcmmemory		\
			-text [mc "MCU 8051 IDE configuration"]	\
			-font [font create -size -20]

		# Create horizontal separator
		Separator $win.sep -orient horizontal

		## Create middle frame
		set middle_frame [frame $win.middle_frame]
			# Checkbutton "Display splash screen"
		grid [Label $middle_frame.lbl_splash			\
			-text [mc "Display splash screen"]		\
			-helptext [mc "Show splash screen on startup"]	\
		] -row 0 -column 0 -sticky w
		grid [checkbutton $middle_frame.chb_splash		\
			-variable ::configDialogs::global::show_splash	\
		] -row 0 -column 1 -sticky w
		DynamicHelp::add $middle_frame.chb_splash	\
			-text [mc "Show splash screen on startup"]
			# Checkbutton "Show tips on startup"
		grid [Label $middle_frame.lbl_tips		\
			-text [mc "Show tips on startup"]	\
			-helptext [mc "Invoke dialog with tip of the day on startup"]	\
		] -row 1 -column 0 -sticky w
		grid [checkbutton $middle_frame.chb_tips		\
			-variable ::configDialogs::global::show_tips	\
		] -row 1 -column 1 -sticky w
		DynamicHelp::add $middle_frame.chb_tips	\
			-text [mc "Invoke dialog with tip of the day on startup"]
			# Combo "Language"
		grid [Label $middle_frame.lbl_lang			\
			-text [mc "Language"]				\
			-helptext [mc "Your preferred language"]	\
		] -row 2 -column 0 -sticky w
		grid [ttk::combobox $middle_frame.cb_lang		\
			-values $avaliable_languages			\
			-state readonly					\
			-textvariable ::configDialogs::global::language	\
		] -row 2 -column 1 -sticky w
		bind $middle_frame.cb_lang <<ComboboxSelected>>	\
			{::configDialogs::global::language_changed}
		DynamicHelp::add $middle_frame.cb_lang	\
			-text [mc "Your preferred language"]
		 # Separator
		grid [ttk::separator $middle_frame.sep -orient horizontal] -columnspan 2 -sticky we -row 3 -column 0 -pady 5
		 # Checkbutton "Do not ask whether ..."
		grid [text $middle_frame.lbl_ask				\
			-wrap word -height 4 -width 0 -bg {#EEEEEE} -bd 0	\
		] -row 4 -column 0 -sticky we
		$middle_frame.lbl_ask insert end [mc "Do not always ask whether to add file to the project after the file is opened"]
		$middle_frame.lbl_ask configure -state disabled
		grid [checkbutton $middle_frame.cb_ask		\
			-onvalue 0 -offvalue 1 			\
			-variable ::FileList::ask__append_file_to_project	\
		] -row 4 -column 1 -sticky w

			# Finalize
		grid columnconfigure $middle_frame 0 -minsize 200

		## Button frame at the bottom
		set but_frame [frame $win.button_frame]
		 # Button "Reset"
		pack [ttk::button $but_frame.but_default		\
			-text [mc "Reset to defaults"]			\
			-command {::configDialogs::global::DEFAULTS}	\
		] -side left
		DynamicHelp::add $but_frame.but_default	\
			-text [mc "Reset all settings to defaults"]
		 # Button "Ok"
		pack [ttk::button $but_frame.but_ok			\
			-text [mc "Ok"]					\
			-compound left					\
			-image ::ICONS::16::ok				\
			-command {::configDialogs::global::OK}		\
		] -side right
		DynamicHelp::add $but_frame.but_ok	\
			-text [mc "Commit new settings"]
		 # Button "Cancel"
		pack [ttk::button $but_frame.but_cancel			\
			-text [mc "Cancel"]				\
			-compound left					\
			-image ::ICONS::16::button_cancel		\
			-command {::configDialogs::global::CANCEL}	\
		] -side right
		DynamicHelp::add $but_frame.but_cancel	\
			-text [mc "Take changes back and close dialog"]

		# Pack frames and notebook
		pack $win.header_label -side top -pady 6
		pack $win.sep -side top -fill x -after $win.header_label
		pack $middle_frame -side top -padx 10 -anchor nw -pady 10
		pack $but_frame -side bottom -fill x -expand 1 -anchor s -padx 10 -pady 5

		# Set window attributes
		wm iconphoto $win ::ICONS::16::configure
		wm transient $win .
		wm title $win [mc "Configure MCU 8051 IDE"]
		wm minsize $win 380 250
		raise $win
		catch {grab $win}
		wm protocol $win WM_DELETE_WINDOW {
			::configDialogs::global::CANCEL
		}
		tkwait window $win
	}

	## Application language changed
	 # Takes any set of arguments and discards them
	 # @parm List - meaningless
	 # @return void
	proc language_changed args {
		tk_messageBox				\
			-parent .global_config_dialog	\
			-type ok -icon info		\
			-title [mc "Application language changed"]	\
			-message [mc "Language for this application has been changed. The change will take effect upon next start of application"]
	}

	## Retrieve list of avaliable translations
	 # @return void
	proc get_languages {} {
		variable avaliable_languages	;# List of avaliable languages

		set avaliable_languages {en}
		set tmp [list]
		catch {	;# For Microsoft Windows it has to be enclosed by catch
			set tmp [glob -nocomplain -types f -tails			\
				-directory "${::LIB_DIRNAME}/../translations" *.msg	\
			]
		}

		foreach translation $tmp {
			lappend avaliable_languages [file rootname $translation]
		}
	}

	## Set configuration variable
	 # @parm String variable	- variable to set
	 # @parm Mixed value		- new value
	proc set_variable {variable value} {
		variable show_splash	;# Bool: Show splash creen on startup
		variable show_tips	;# Bool: Show tips on startup
		variable language	;# String: Language

		getSettings

		switch -- $variable {
			{splash} {
				set show_splash $value
			}
			{tips} {
				set show_tips $value
			}
			{language} {
				set language $value
			}
		}

		use_settings
		save_config
	}

	## Retrieve settings from main NS
	 # @return void
	proc getSettings {} {
		variable show_splash	;# Bool: Show splash creen on startup
		variable show_tips	;# Bool: Show tips on startup
		variable language	;# String: Language

		set show_splash	${::GLOBAL_CONFIG(splash)}
		set show_tips	${::GLOBAL_CONFIG(tips)}
		set language	${::GLOBAL_CONFIG(language)}
	}

	## Set application acording to local settings
	 # @return void
	proc use_settings {} {
		variable show_splash	;# Bool: Show splash creen on startup
		variable show_tips	;# Bool: Show tips on startup
		variable language	;# String: Language

		if {${::GLOBAL_CONFIG(language)} != $language} {
			set lang_changed 1
		} {
			set lang_changed 0
		}

		set ::GLOBAL_CONFIG(splash)	$show_splash
		set ::GLOBAL_CONFIG(tips)	$show_tips
		set ::GLOBAL_CONFIG(language)	$language

		if {$lang_changed} {
			::X::switch_language
		}
	}

	## Save settings to the config file
	 # @return void
	proc save_config {} {
		variable show_splash	;# Bool: Show splash creen on startup
		variable show_tips	;# Bool: Show tips on startup
		variable language	;# String: Language

		if {[catch {
			set conf_file [open "${::CONFIG_DIR}/base.conf" w]
			puts -nonewline $conf_file	\
				[list $show_splash $show_tips $language]
			close $conf_file
		}]} {
			puts stderr [mc "Unable to write to base configuration file"]
		}
	}

	## Destroy the dialog
	 # @return void
	proc CANCEL {} {
		variable win		;# ID of toplevel dialog window
		variable dialog_opened	;# Bool: True if this dialog is already opened

		# Destroy dialog window
		set dialog_opened 0
		grab release $win
		destroy $win
	}

	## Use settings and destroy the dialog
	 # @return void
	proc OK {} {
		variable win	;# ID of toplevel dialog window

		# Use and save settings
		use_settings
		save_config

		# Destroy dialog window
		CANCEL
	}

	## Restrore defaults
	 # @return void
	proc DEFAULTS {} {
		variable show_splash	;# Bool: Show splash creen on startup
		variable show_tips	;# Bool: Show tips on startup
		variable language	;# String: Language

		set show_splash	1
		set show_tips	1
		set language	{en}
	}
}
