/******************************** LICENSE ********************************

 Copyright 2007 European Centre for Medium-Range Weather Forecasts (ECMWF)

 Licensed under the Apache License, Version 2.0 (the "License");
 you may not use this file except in compliance with the License.
 You may obtain a copy of the License at 

    http://www.apache.org/licenses/LICENSE-2.0

 Unless required by applicable law or agreed to in writing, software
 distributed under the License is distributed on an "AS IS" BASIS,
 WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 See the License for the specific language governing permissions and
 limitations under the License.

 ******************************** LICENSE ********************************/

/*! \file Data.h
    \brief Definition of the Abstract template class Data.
    
    Magics Team - ECMWF 2004
    
    Started: Fri 16-Jan-2004
    
    Changes:
    
*/

#ifndef Data_H
#define Data_H

#include "magics.h"
#include "Exception.h"

#include "MatrixHandler.h"
#include "PointsHandler.h"
#include "RasterData.h"
#include "CustomisedPoint.h"
#include "RasterData.h"
#include "Transformation.h"
#include "MagnifierVisitor.h"
#include "DateTime.h"
#include "MagicsEvent.h"
namespace magics {

class LegendVisitor;
class TextVisitor;
class MetaDataVisitor; 
class MagnifierVisitor; 
class LayerNode;
class AnimationRules;
class AnimationStep;


template <class P>
struct MatchCriteria 
{
    MatchCriteria() {}
    virtual ~MatchCriteria() {}
    virtual bool verify(const P&, const string&, const string&) { return false; }
};

class DataInMemory : public MagicsException
{
public:
	 DataInMemory():
		MagicsException("Data in Memory..." ) {}
};     


template <class P>
class Data: public MetviewIcon{

public:
	Data(): name_("no_name") {}
	virtual ~Data() {}
    //! Method to access the data as a matrix Used by pcont action routine
    virtual MatrixHandler<P>& matrix() { throw MethodNotYetImplemented("Data<P>::matrix"); }
    
    //! Methods to access the data as a 2Dmatrix Used by pwind action routine
    virtual MatrixHandler<P>& xComponent() { throw MethodNotYetImplemented("Data<P>::xComponent"); }
    virtual MatrixHandler<P>& yComponent() { throw MethodNotYetImplemented("Data<P>::yComponent"); }
    
    //! Method to access the data as a list of points : Used by psymb.
    virtual PointsHandler<P>& points() { throw MethodNotYetImplemented("Data<P>::points"); }
    virtual PointsHandler<P>& points(const Transformation&) { throw MethodNotYetImplemented("Data<P>::points"); }
  
    
    virtual void getReady(const Transformation&) {}
    virtual void visit(Transformation&)          {}
    
      //! Methods to access the data as a file used by pplot action routine
    virtual string path() { throw DataInMemory(); }
     //! Methods to access the data as a RasterData used by pimage action routine.
    virtual RasterData<P>& raster() { throw MethodNotYetImplemented("Data<P>::raster"); }
    
    virtual void getInfo(const std::set<string>&, multimap<string, string>&) 
    { throw MethodNotYetImplemented("Data<P>::getInfo"); }
    
    virtual void customisedPoints(const std::set<string>&, CustomisedPointsList&) 
    { throw MethodNotYetImplemented("Data<P>::customisedPoints"); }
    virtual void customisedPoints(const Transformation&, const std::set<string>&, CustomisedPointsList& ) 
        { throw MethodNotYetImplemented("Data<P>::customisedPoints"); }
    virtual void visit(TextVisitor&) {}
    virtual void visit(LegendVisitor&) {}
    virtual void visit(AnimationRules&) {}
    virtual void visit(AnimationStep&) {}
    virtual void visit(MetaDataVisitor&) {} 
    virtual void visit(Layer& layer) { 
    	MetviewIcon::visit(layer); 	
    	layer.name(name());
    	layer.valid(from(), to());
    }
    virtual void visit(MagnifierVisitor& magnify) {
    	PointsHandler<P>& list = this->points(); 
    	const Transformation& transformation = magnify.transformation();
    	list.setToFirst();
    	while (list.more()) {   		
    		magnify.add(transformation(list.current()));
    		list.advance();
    	}  
    }
    
    string legend() { return legend_; }
    
    // Information needed fron layer management!
    virtual string layerId()  {  return  ( layerId_.empty() ) ?  iconName_ + "/ " + iconClass_  : layerId_  ; }
    virtual string name()  { return  ( iconName_.empty() ) ? name_ : iconName_; }
    virtual const DateTime& from()  { return from_; }
    virtual const DateTime& to()  { return to_; }
 

protected:
     //! Method to print string about this class on to a stream of type ostream (virtual).
    virtual void print(ostream& out) const { out << "Data<P>"; }
   
    
     VectorOfPointers<vector<PointsHandler<P>* > > pointsHandlers_; 
     // The objects are put in a list, they will be automaticcaly destructed 
     VectorOfPointers<vector<MatrixHandler<P>* > > matrixHandlers_; 

	string legend_;
	
	string name_;
	string layerId_;
	DateTime from_;
	DateTime to_;
	
	
private:
    //! Copy constructor - No copy allowed
	Data(const Data&);
    //! Overloaded << operator to copy - No copy allowed
	Data& operator=(const Data&);
    
// -- Friends
    //! Overloaded << operator to call print().
	friend ostream& operator<<(ostream& s,const Data<P>& p)
		{ p.print(s); return s; }

};
template <class P>
class DataLoop
{
public:
	DataLoop() {}
	virtual ~DataLoop() {}
	virtual void set(const map<string, string>&) {}
	virtual void set(const XmlNode&) {}
	virtual void set(LayerNode&) {}
	
	virtual void setToFirst() {}
	virtual Data<P>* current() = 0;
	virtual bool         hasMore() = 0;
	virtual void         next() = 0;
	virtual void add(Data<P>*) {}
};
} // namespace magics


#endif
