/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.wsee;

import java.io.File;
import java.text.MessageFormat;
import java.util.Collection;
import java.util.Iterator;
import java.util.List;
import java.util.Properties;
import java.util.StringTokenizer;

import xjavadoc.XClass;
import xjavadoc.XPackage;
import xjavadoc.XTag;

import xdoclet.ConfigParamIntrospector;
import xdoclet.DocletContext;
import xdoclet.DocletSupport;
import xdoclet.XDocletException;

import xdoclet.XDocletTagSupport;
import xdoclet.tagshandler.PackageTagsHandler;

/**
 * Tags handler for dealing with wsee extensions.
 *
 * @author               Christoph G. Jung (christoph.jung@infor.de)
 * @author               Jason Essington (jasone@greenrivercomputing.com)
 * @created              23.12.03
 * @version              $Revision: 1.5 $
 * @xdoclet.taghandler   namespace="Wsee"
 */

public class WseeTagsHandler extends XDocletTagSupport
{
    /**
     * name of javadoc tag indicating a port component
     */
    public final static String PORT_COMPONENT = "wsee.port-component";
    /**
     * name of javadoc tag indicating a handler
     */
    public final static String HANDLER = "wsee.handler";

    /**
     * state
     */
    public XTag     currentHandler;

    /**
     * check whether the given class is a port component
     *
     * @param clazz                 class
     * @return
     * @exception XDocletException
     */
    public static boolean isPortComponent(XClass clazz)
         throws XDocletException
    {
        return clazz.getDoc().hasTag(PORT_COMPONENT);
    }

    /**
     * Gets the namespace for the specified package.
     *
     * @param pak  package
     * @return     namespace
     * @see        #getNamespaceForPackage(java.lang.String)
     */
    public static String getNamespaceForPackage(XPackage pak)
    {
        return getNamespaceForPackage(pak.getName());
    }

    /**
     * Gets the namespace for the specified package.
     *
     * @param pak  package
     * @return     namespace
     */
    public static String getNamespaceForPackage(String pak)
    {
        List packageSubstitutions = getPackageNamespaceMappings();

        for (int i = 0; i < packageSubstitutions.size(); i++) {
            WseeDocletTask.PackageNamespaceMapping ps = (WseeDocletTask.PackageNamespaceMapping) packageSubstitutions.get(i);
            StringTokenizer st = new StringTokenizer(ps.getPackages(), ",", false);

            while (st.hasMoreTokens()) {
                String packages = st.nextToken();

                if (pak.startsWith(packages)) {
                    return ps.getNamespace() + pak.substring(packages.length()).replace('.', '-');
                }
            }
        }

        return "urn-" + pak.replace('.', '-');
    }

    /**
     * Gets the package-namespace mappings for the subtask.
     *
     * @return   List of packageNamespaceMapping config params
     */
    public static List getPackageNamespaceMappings()
    {
        // SubTask's packageNamespaceMappings has precedence over
        // the global packageNamespaceMappings defined in DocletTask
        List packageNamespaceMappings = (List) DocletContext.getInstance().getConfigParam("packageNamespaceMappings");

        return packageNamespaceMappings;
    }

    /**
     * return the namespace of the specified class
     *
     * @param clazz                 class
     * @return                      namespace URI
     * @exception XDocletException
     */
    public String getNamespaceURI(XClass clazz)
         throws XDocletException
    {
        XTag wsTag = clazz.getDoc().getTag("wsee.port-component");
        String nameSpace = null;

        if (wsTag == null) {
            wsTag = clazz.getDoc().getTag("wsee.jaxrpc-mapping");
        }
        if (wsTag != null) {
            nameSpace = wsTag.getAttributeValue("namespace-uri");
        }
        if (nameSpace == null) {
            nameSpace = getNamespaceForPackage(clazz.getContainingPackage());
        }
        return nameSpace;
    }

    /**
     * Iterates over all classes loaded by javadoc and being a port component.
     *
     * @param template              The body of the block tag
     * @param attributes            The attributes of the template tag
     * @exception XDocletException  Description of Exception
     * @doc.tag                     type="block"
     * @doc.param                   name="abstract" optional="true" values="true,false" description="If true then accept
     *      abstract classes also; otherwise don't."
     * @doc.param                   name="type" optional="true" description="For all classes by the type."
     */
    public void forAllPortComponents(String template, Properties attributes)
         throws XDocletException
    {
        Collection classes = getXJavaDoc().getSourceClasses();

        for (Iterator i = classes.iterator(); i.hasNext(); ) {
            XClass clazz = (XClass) i.next();

            setCurrentClass(clazz);

            if (DocletSupport.isDocletGenerated(getCurrentClass())) {
                continue;
            }

            if (isPortComponent(getCurrentClass())) {
                generate(template);
            }
        }
    }

    /**
     * returns the service endpoint interface name belonging to the current class
     *
     * @param props                 The attributes of the template tag
     * @return
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String serviceEndpoint(Properties props) throws XDocletException
    {
        XClass clazz = getCurrentClass();
        String pkg = PackageTagsHandler.getPackageNameFor(clazz.getContainingPackage(), true);
        XTag ejbTag = null;
        String spec = null;

        if (clazz.getDoc().hasTag("ejb.bean")) {
            ejbTag = clazz.getDoc().getTag("ejb.interface");
            if (ejbTag != null) {
                spec = ejbTag.getAttributeValue("service-endpoint-class");
            }
            // if we haven't explicitly defined a service interface name, try to build the default name.
            if (spec == null || "".equals(spec)) {
                spec = pkg + "." + clazz.getName();
                if (spec.endsWith("Bean"))
                    spec = spec.substring(0, spec.length() - 4);
            }
        }
        else {

            ejbTag = clazz.getDoc().getTag("web.servlet");
            if (ejbTag != null) {
                spec = ejbTag.getAttributeValue("service-endpoint-class");
            }
            if (spec == null || "".equals(spec)) {
                spec = pkg + "." + clazz.getName();
                spec += "Service";
            }
        }

        return spec;
    }

    /**
     * returns the service endpoint link pointing to the current class
     *
     * @param props                 The attributes of the template tag
     * @return
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String serviceEndpointLink(Properties props)
         throws XDocletException
    {
        XClass clazz = getCurrentClass();
        XTag ejbTag = clazz.getDoc().getTag("ejb.bean");

        if (ejbTag != null) {
            return "<ejb-link>" + ejbTag.getAttributeValue("name") + "</ejb-link>";
        }
        ejbTag = clazz.getDoc().getTag("web.servlet");
        if (ejbTag != null) {
            return "<servlet-link>"
                + ejbTag.getAttributeValue("name")
                + "</servlet-link>";
        }
        return null;
    }

    /**
     * return the namespace of the current clazz/package
     *
     * @return
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String namespaceURI()
         throws XDocletException
    {
        String ns = "";

        if (getCurrentClass() != null) {
            ns = getNamespaceURI(getCurrentClass());
        }
        else if (getCurrentPackage() != null) {
            ns = getNamespaceForPackage(getCurrentPackage());
        }
        else {
            // we don't have a current package or class, so just get the first namespace.
            List nsmappings = getPackageNamespaceMappings();

            if (!nsmappings.isEmpty()) {
                WseeDocletTask.PackageNamespaceMapping ps = (WseeDocletTask.PackageNamespaceMapping) nsmappings.get(0);

                ns = ps.getNamespace();
            }
        }
        return ns;
    }

    /**
     * Iterates over all handler tags annotating the current class.
     *
     * @param template           The body of the block tag
     * @param attributes         The attributes of the template tag
     * @throws XDocletException
     * @doc.tag                  type="block"
     */
    public void forAllHandlers(String template, Properties attributes)
         throws XDocletException
    {
        XClass clazz = getCurrentClass();
        Iterator allTags = clazz.getDoc().getTags(HANDLER).iterator();

        while (allTags.hasNext()) {
            currentHandler = (XTag) allTags.next();
            generate(template);
        }

    }

    /**
     * conditional checking presence of a handler tag
     *
     * @param template              The body of the block tag
     * @param props                 The attributes of the template tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     * @doc.param                   name="paramName" optional="false" description="The name of the parameter"
     */
    public void ifHasHandlerTag(String template, Properties props)
         throws XDocletException
    {
        if (handlerTagValue(props) != null) {
            generate(template);
        }
    }

    /**
     * extract the value of the current handler tag
     *
     * @param props                 The attributes of the template tag
     * @return
     * @exception XDocletException
     * @doc.tag                     type="content"
     * @doc.param                   name="paramName" optional="false" description="The name of the parameter"
     */
    public String handlerTagValue(Properties props) throws XDocletException
    {
        return currentHandler.getAttributeValue(props.getProperty("paramName"));
    }

    /**
     * conditional to handle per class wsdl
     *
     * @param template           The body of the block tag
     * @param props              The attributes of the template tag
     * @throws XDocletException
     * @doc.tag                  type="block"
     */
    public void ifWsdlPerClass(String template, Properties props) throws XDocletException
    {
        if (isWsdlPerClass())
            generate(template);
    }

    /**
     * conditional to handle single wsdl generation
     *
     * @param template           The body of the block tag
     * @param props              The attributes of the template tag
     * @throws XDocletException
     * @doc.tag                  type="block"
     */
    public void ifNotWsdlPerClass(String template, Properties props) throws XDocletException
    {
        if (!isWsdlPerClass())
            generate(template);
    }

    /**
     * Constructs a guestimated filename for the wsdl file. It also attempts to decide if the file should be in META-INF
     * or WEB-INF. This should yield a filename that will be correct for use within the webservices.xml file.
     *
     * @param props  If prefixWithPackageStructure is specified for the wsdl sub task, the property
     *      prefixWithPackage="true" will need to be specified.
     * @return       filename
     * @doc.tag      type="content"
     * @doc.param    name="prefixWithPackage" optional="true" values="true,false" description="Whether to prefix the
     *      filename with the package hierarchy."
     */
    public String wsdlFilename(Properties props)
    {
        XClass clazz = getCurrentClass();
        String wsdlPattern = getWsdlFilePattern();

        String packageName = null;
        String file = null;

        if (isWsdlPerClass()) {

            boolean prefixWithPackage = false;
            String hasPrefix = props.getProperty("prefixWithPackage");

            if (hasPrefix != null && !"".equals(hasPrefix)) {
                prefixWithPackage = Boolean.getBoolean(hasPrefix);
            }

            if (prefixWithPackage) {
                packageName = PackageTagsHandler.packageNameAsPathWithoutSubstitutionFor(clazz.getContainingPackage());
            }

            String serviceName = getCurrentClass().getDoc().getTagAttributeValue(WseeTagsHandler.PORT_COMPONENT, "name");

            file = new File(packageName, serviceName).toString();
        }

        // assume our wsdl files will start in WEB-INF/ unless the current class has an ejb.bean tag
        String prefix = "WEB-INF/";

        if (clazz != null && clazz.getDoc().hasTag("ejb.bean")) {
            prefix = "META-INF/";
        }
        return prefix + MessageFormat.format(wsdlPattern, new Object[]{file});
    }

    /**
     * Constructs a guestimated filename for the jaxrpc file.
     *
     * @param props  If prefixWithPackageStructure is specified for the wsdl sub task, the property
     *      prefixWithPackage="true" will need to be specified.
     * @return       filename
     * @doc.tag      type="content"
     * @doc.param    name="prefixWithPackage" optional="true" values="true,false" description="Whether to prefix the
     *      filename with the package hierarchy."
     */
    public String jaxrpcMappingFilename(Properties props)
    {
        XClass clazz = getCurrentClass();
        String jaxrpcPattern = getJaxrpcFilePattern();

        String packageName = null;
        String file = null;

        if (isJaxrpcPerClass()) {

            boolean prefixWithPackage = true;
            String hasPrefix = props.getProperty("prefixWithPackage");

            if (hasPrefix != null && !"".equals(hasPrefix)) {
                prefixWithPackage = Boolean.getBoolean(hasPrefix);
            }

            if (prefixWithPackage) {
                packageName = PackageTagsHandler.packageNameAsPathWithoutSubstitutionFor(clazz.getContainingPackage());
            }

            file = new File(packageName, getCurrentClass().getName()).toString();
        }


        // assume our wsdl files will start in WEB-INF/ unless the current class has an ejb.bean tag
        String prefix = "WEB-INF/";

        if (clazz != null && clazz.getDoc().hasTag("ejb.bean")) {
            prefix = "META-INF/";
        }
        return prefix + MessageFormat.format(jaxrpcPattern, new Object[]{file});
    }

    /**
     * Is wsdl generation by class or as a single file?
     *
     * @return   true if by class
     */
    protected boolean isWsdlPerClass()
    {
        return getWsdlFilePattern().indexOf("{0}") != -1;
    }

    /**
     * Is jaxrpc generation by class or as a single file?
     *
     * @return   true if by class
     */
    protected boolean isJaxrpcPerClass()
    {
        return getJaxrpcFilePattern().indexOf("{0}") != -1;
    }

    /**
     * Get the value of the wsdl file pattern
     *
     * @return   pattern
     */
    protected String getWsdlFilePattern()
    {
        String pattern = null;
        Object wsdlFile = DocletContext.getInstance().getConfigParam("wsdlFile");

        if (wsdlFile == ConfigParamIntrospector.NULL || "".equals(wsdlFile)) {
            pattern = WsdlSubTask.DEFAULT_WSDL_FILE_PATTERN;
        }
        else {
            pattern = (String) wsdlFile;
        }
        return pattern;
    }

    /**
     * Get the value of the jaxrpc file pattern
     *
     * @return   pattern
     */
    protected String getJaxrpcFilePattern()
    {
        String pattern = "";
        Object jaxrpcFile = DocletContext.getInstance().getConfigParam("jaxrpcMappingFile");

        if (jaxrpcFile != ConfigParamIntrospector.NULL) {
            pattern = (String) jaxrpcFile;
        }
        return pattern;
    }
}
