/*
 * Copyright (c) 2001, 2002 The XDoclet team
 * All rights reserved.
 */
package xdoclet.modules.ejb.mdb;

import java.text.MessageFormat;

import java.util.*;

import xjavadoc.*;

import xdoclet.DocletContext;
import xdoclet.DocletSupport;
import xdoclet.DocletTask;
import xdoclet.XDocletException;
import xdoclet.modules.ejb.EjbTagsHandler;

/**
 * @author               Ara Abrahamian (ara_e@email.com)
 * @created              Oct 16, 2001
 * @xdoclet.taghandler   namespace="EjbMdb"
 * @version              $Revision: 1.16 $
 */
public class MdbTagsHandler extends EjbTagsHandler
{

    private XTag    currentDestination;

    /**
     * Gets the MdbClassFor attribute of the MdbTagsHandler class
     *
     * @param clazz  Describe what the parameter does
     * @return       The MessageDrivenClassFor value
     */
    public static String getMessageDrivenClassFor(XClass clazz)
    {
        String fileName = clazz.getContainingPackage().getName();
        String mdbName = MessageFormat.format(getMessageDrivenClassPattern(), new Object[]{getShortEjbNameFor(clazz)});

        // Fix package name
        fileName = choosePackage(fileName, null, DocletTask.getSubTaskName(MdbSubTask.class));
        if (fileName.length() > 0) {
            fileName += ".";
        }

        fileName += mdbName;

        return fileName;
    }

    /**
     * Returns true if clazz is a message-driven bean, false otherwise.
     *
     * @param clazz  Description of Parameter
     * @return       The MessageDriven value
     */
    public static boolean isMessageDriven(XClass clazz)
    {
        return clazz.isA("javax.ejb.MessageDrivenBean");
    }

    /**
     * Gets the MdbClassPattern attribute of the MdbTagsHandler class
     *
     * @return   The MdbClassPattern value
     */
    protected static String getMessageDrivenClassPattern()
    {
        MdbSubTask mdbSubtask = ((MdbSubTask) DocletContext.getInstance().getSubTaskBy(DocletTask.getSubTaskName(MdbSubTask.class)));

        if (mdbSubtask != null) {
            return mdbSubtask.getMessageDrivenClassPattern();
        }
        else {
            return MdbSubTask.DEFAULT_MESSAGE_DRIVEN_CLASS_PATTERN;
        }
    }

    /**
     * Evaluates the body block for each EJBean derived from MessageDrivenBean.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @see                         #isMessageDriven(xjavadoc.XClass)
     * @doc.tag                     type="block"
     */
    public void forAllMDBeans(String template) throws XDocletException
    {
        Collection classes = getXJavaDoc().getSourceClasses();

        for (Iterator i = classes.iterator(); i.hasNext(); ) {
            XClass clazz = (XClass) i.next();

            setCurrentClass(clazz);

            if (DocletSupport.isDocletGenerated(getCurrentClass())) {
                continue;
            }

            if (!hasHavingClassTag(getCurrentClass())) {
                continue;
            }

            if (isMessageDriven(getCurrentClass())) {
                generate(template);
            }
        }
    }

    /**
     * Evaluate the body block if current class is of a message driven bean type.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifMessageDriven(String template) throws XDocletException
    {
        if (isMessageDriven(getCurrentClass())) {
            generate(template);
        }
    }

    /**
     * Evaluate the body block if current class is not of a message driven bean type.
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifNotMessageDriven(String template) throws XDocletException
    {
        if (!isMessageDriven(getCurrentClass())) {
            generate(template);
        }
    }

    /**
     * Returns the name of message-driven bean class.
     *
     * @return                      The name of generated message-driven bean class.
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String messageDrivenClass() throws XDocletException
    {
        return getMessageDrivenClassFor(getCurrentClass());
    }

    /**
     * Returns the name of message-driven bean class.
     *
     * @return                      The name of generated message-driven bean class.
     * @exception XDocletException
     * @doc.tag                     type="content"
     */
    public String mdbClass() throws XDocletException
    {
        return getMessageDrivenClassFor(getCurrentClass());
    }


    /**
     * Evaluate the body block if the current class has a activation config
     *
     * @param template              The body of the block tag
     * @param attributes
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifHasActivationConfig(String template, Properties attributes) throws XDocletException
    {
        if (hasActivationConfig()) {
            generate(template);
        }
    }

    /**
     * Evaluates the body block for each ejb:message-destination defined in class level
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @see                         #destinationName()
     * @see                         #destinationDescription()
     * @see                         #destinationDisplayName()
     * @doc.tag                     type="block"
     */
    public void forAllDestinations(String template) throws XDocletException
    {
        Collection classes = getXJavaDoc().getSourceClasses();

        Map dests = new HashMap();

        for (Iterator i = classes.iterator(); i.hasNext(); ) {
            XClass clazz = (XClass) i.next();

            setCurrentClass(clazz);

            if (getCurrentClass().getDoc().hasTag("ejb.message-destination")) {
                XTag tag = getCurrentClass().getDoc().getTag("ejb.message-destination");
                String destName = tag.getAttributeValue("name");

                if (destName != null) {
                    dests.put(destName, tag);
                }
            }
        }

        // Output set of destinations
        for (Iterator it = dests.values().iterator(); it.hasNext(); ) {
            currentDestination = (XTag) it.next();

            generate(template);
        }
    }

    /**
     * Returns the name of the current message destination
     *
     * @return                      Current message destination's name
     * @exception XDocletException
     * @see                         #forAllDestinations(java.lang.String)
     * @doc.tag                     type="content"
     */
    public String destinationName() throws XDocletException
    {
        return currentDestination.getAttributeValue("name");
    }

    /**
     * Returns the display-name of the current message destination
     *
     * @return                      Current message destination's display-name
     * @exception XDocletException
     * @see                         #forAllDestinations(java.lang.String)
     * @doc.tag                     type="content"
     */
    public String destinationDisplayName() throws XDocletException
    {
        return currentDestination.getAttributeValue("display-name");
    }

    /**
     * Evaluate the body block if the current message destination has a display-name
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifDestinationHasDisplayName(String template) throws XDocletException
    {
        if (destinationDisplayName() != null) {
            generate(template);
        }
    }

    /**
     * Returns the jndi-name of the current message destination
     *
     * @return                      Current message destination's jndi-name
     * @exception XDocletException
     * @see                         #forAllDestinations(java.lang.String)
     * @doc.tag                     type="content"
     */
    public String destinationJndiName() throws XDocletException
    {
        return currentDestination.getAttributeValue("jndi-name");
    }

    /**
     * Evaluate the body block if the current message destination has a jndi-name
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifDestinationHasJndiName(String template) throws XDocletException
    {
        if (destinationJndiName() != null) {
            generate(template);
        }
    }

    /**
     * Returns the description of the current message destination
     *
     * @return                      Current message destination's description
     * @exception XDocletException
     * @see                         #forAllDestinations(java.lang.String)
     * @doc.tag                     type="content"
     */
    public String destinationDescription() throws XDocletException
    {
        return currentDestination.getAttributeValue("description");
    }

    /**
     * Evaluate the body block if the current message destination has a descrition
     *
     * @param template              The body of the block tag
     * @exception XDocletException
     * @doc.tag                     type="block"
     */
    public void ifDestinationHasDescription(String template) throws XDocletException
    {
        if (destinationDescription() != null) {
            generate(template);
        }
    }

    /**
     * Returns true if the current class has a activation config
     *
     * @return                      <code>true</code> if the current class has a activation config
     * @exception XDocletException
     */
    private boolean hasActivationConfig() throws XDocletException
    {
        Properties props = new Properties();

        props.setProperty("tagName", "ejb.bean");
        props.setProperty("paramName", "destination-type");
        if (hasTag(props, FOR_CLASS)) {
            return true;
        }

        props.setProperty("paramName", "acknowledge-mode");
        if (hasTag(props, FOR_CLASS)) {
            return true;
        }

        props.setProperty("paramName", "subscription-durability");
        if (hasTag(props, FOR_CLASS)) {
            return true;
        }

        props.setProperty("paramName", "message-selector");
        if (hasTag(props, FOR_CLASS)) {
            return true;
        }

        props.setProperty("tagName", "ejb.activation-config-property");
        props.setProperty("paramName", "");
        return hasTag(props, FOR_CLASS);
    }

}

