#include "TePDIPrincipalComponentsFusion.hpp"

#include <TeAgnostic.h>
#include "TePDIUtils.hpp"
#include "TePDITypes.hpp"
#include "TePDIStatistic.hpp"
#include "TePDIPrincipalComponents.hpp"
#include "TeRasterRemap.h"
#include <TeMatrix.h>
#include <TeUtils.h>
#include <math.h>

TePDIPrincipalComponentsFusion::TePDIPrincipalComponentsFusion()
{
}

TePDIPrincipalComponentsFusion::~TePDIPrincipalComponentsFusion()
{
}

void TePDIPrincipalComponentsFusion::ResetState(const TePDIParameters& /*params*/)
{
}

bool TePDIPrincipalComponentsFusion::CheckParameters(const TePDIParameters& parameters) const
{
/* Input rasters and bands checking */

	TePDITypes::TePDIRasterVectorType input_rasters;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("input_rasters", input_rasters), "Missing parameter: input_rasters");
	TEAGN_TRUE_OR_RETURN(input_rasters.size() > 1, "Invalid input rasters number");
	
	std::vector<int> bands;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("bands", bands), "Missing parameter: bands");
	TEAGN_TRUE_OR_RETURN(bands.size() == input_rasters.size(), "Invalid parameter: bands number");

	for( unsigned int input_rasters_index = 0 ; input_rasters_index < input_rasters.size(); ++input_rasters_index )
	{
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index].isActive(), "Invalid parameter: input_raster " + Te2String(input_rasters_index) + " inactive");
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index]->params().status_ != TeRasterParams::TeNotReady, "Invalid parameter: input_raster " + Te2String(input_rasters_index) + " not ready");
		TEAGN_TRUE_OR_RETURN(bands[input_rasters_index] < input_rasters[input_rasters_index]->nBands(), "Invalid parameter: bands[" + Te2String(input_rasters_index) + "]");
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index]->params().nlines_ == input_rasters[0]->params().nlines_, "Invalid parameter: input_raster " + Te2String(input_rasters_index) + " with imcompatible number of lines");
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index]->params().ncols_ == input_rasters[0]->params().ncols_, "Invalid parameter: input_raster " + Te2String( input_rasters_index ) + " with imcompatible number of columns" );
		TEAGN_TRUE_OR_RETURN(((input_rasters[input_rasters_index]->params().photometric_[bands[input_rasters_index]] == TeRasterParams::TeRGB) || (input_rasters[input_rasters_index]->params().photometric_[bands[input_rasters_index]] == TeRasterParams::TeMultiBand)), "Invalid parameter - input_rasters (invalid photometric interpretation)");
	}

/* Reference raster checking */

	int reference_raster_band;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("reference_raster_band", reference_raster_band), "Missing parameter: reference_raster_band");

	TePDITypes::TePDIRasterPtrType reference_raster;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("reference_raster", reference_raster), "Missing parameter: reference_raster");
	TEAGN_TRUE_OR_RETURN(((reference_raster->params().photometric_[reference_raster_band] == TeRasterParams::TeRGB ) || (reference_raster->params().photometric_[reference_raster_band] == TeRasterParams::TeMultiBand)), "Invalid parameter - reference_raster (invalid photometric interpretation)");

	TEAGN_TRUE_OR_RETURN(reference_raster_band < reference_raster->nBands(), "Invalid parameter - reference_raster_band");

/* Output rasters checking */

	TePDITypes::TePDIRasterVectorType output_rasters;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("output_rasters", output_rasters), "Missing parameter: output_rasters");
	TEAGN_TRUE_OR_RETURN(output_rasters.size() > 1, "Invalid output rasters number");
	TEAGN_TRUE_OR_RETURN(output_rasters.size() == input_rasters.size(), "Invalid parameter: output rasters size");

	for(unsigned int input_rasters_index2 = 0; input_rasters_index2 < input_rasters.size(); ++input_rasters_index2)
	{
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index2].isActive(), "Invalid parameter: input_raster " + Te2String(input_rasters_index2) + " inactive");
		TEAGN_TRUE_OR_RETURN(input_rasters[input_rasters_index2]->params().status_ != TeRasterParams::TeNotReady, "Invalid parameter: input_raster " + Te2String(input_rasters_index2) + " not ready");
	}

	InterpMethod resampling_type;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("resampling_type", resampling_type), "Missing parameter: resampling_type");

	bool fit_histogram;
	TEAGN_TRUE_OR_RETURN(parameters.GetParameter("fit_histogram", fit_histogram), "Missing parameter: fit_histogram");

	return true;
}

bool TePDIPrincipalComponentsFusion::RunImplementation()
{
/* Getting parameters */

	TePDITypes::TePDIRasterVectorType input_rasters;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("input_rasters", input_rasters), "Missing parameter: input_rasters");

	std::vector<int> bands_direct;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("bands", bands_direct), "Missing parameter: bands");

	TePDITypes::TePDIRasterVectorType output_rasters;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("output_rasters", output_rasters), "Missing parameter: output_rasters");

	TePDITypes::TePDIRasterPtrType reference_raster;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("reference_raster", reference_raster), "Missing parameter: reference_raster");

	int reference_raster_band;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("reference_raster_band", reference_raster_band), "Missing parameter: reference_raster_band");

	InterpMethod resampling_type;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("resampling_type", resampling_type), "Missing parameter: resampling_type");

	bool fit_histogram;
	TEAGN_TRUE_OR_RETURN(params_.GetParameter("fit_histogram", fit_histogram), "Missing parameter: fit_histogram");

/* Resampling the input raster */

	TeRasterParams resampled_input_rasters_params = input_rasters[0]->params();
	if (reference_raster->projection() != 0)
		resampled_input_rasters_params.projection(reference_raster->projection());
	resampled_input_rasters_params.boxResolution(reference_raster->params().box().x1(), reference_raster->params().box().y1(), reference_raster->params().box().x2(), reference_raster->params().box().y2(), reference_raster->params().resx_, reference_raster->params().resy_);
  
	TePDITypes::TePDIRasterVectorType resampled_input_rasters;
	for(unsigned int b = 0; b < input_rasters.size(); b++)
	{
		if (resampling_type != NothingMethod)
		{
		  TePDIInterpolator::InterpMethod int_method = 
		    TePDIInterpolator::NNMethod;
		  switch( resampling_type )
		  {
		    case NNMethod :
		      int_method = TePDIInterpolator::NNMethod;
		      break;
		    case BilinearMethod :
		      int_method = TePDIInterpolator::BilinearMethod;
		      break;
		    case BicubicMethod :
		      int_method = TePDIInterpolator::BicubicMethod;
		      break;
		    default :
		      TEAGN_LOG_AND_THROW( "Invalid resapling method" );
		      break;
		  }
		  		
			TePDITypes::TePDIRasterPtrType resampled_input_rasters_temp;
			TEAGN_TRUE_OR_RETURN(TePDIUtils::TeAllocRAMRaster(resampled_input_rasters_temp,  resampled_input_rasters_params, TePDIUtils::TePDIUtilsAutoMemPol), "Unable create the resampled input rasters");
			TEAGN_TRUE_OR_RETURN( TePDIUtils::resampleRasterByLinsCols(input_rasters[b], 
			  resampled_input_rasters_temp, 
			  (unsigned int)reference_raster->params().nlines_, 
			  (unsigned int)reference_raster->params().ncols_, 
			  progress_enabled_,
			  int_method), "Raster resample error" );
			resampled_input_rasters.push_back(resampled_input_rasters_temp);
		}
		else
			resampled_input_rasters.push_back(input_rasters[b]);
	}

/* PCA Direct analysis */

	TePDITypes::TePDIRasterVectorType output_rasters_direct;
	for (unsigned b = 0; b < output_rasters.size(); b++)
	{
		TePDITypes::TePDIRasterPtrType outRaster_direct;
		TEAGN_TRUE_OR_THROW(TePDIUtils::TeAllocRAMRaster(outRaster_direct, 1, 1, 1, false, TeDOUBLE, 0), "RAM Raster " + Te2String(b) + " Alloc error");
		output_rasters_direct.push_back(outRaster_direct);
	}

	TeSharedPtr<TeMatrix> covariance_matrix(new TeMatrix);
	
	TePDIPrincipalComponents::TePDIPCAType analysis_type = 
	  TePDIPrincipalComponents::TePDIPCADirect;

	TePDIParameters params_direct;

	params_direct.SetParameter("analysis_type", analysis_type);
	params_direct.SetParameter("input_rasters", resampled_input_rasters);
	params_direct.SetParameter("bands", bands_direct);
	params_direct.SetParameter("output_rasters", output_rasters_direct);
	params_direct.SetParameter("covariance_matrix", covariance_matrix);
	
	TePDIPrincipalComponents pc_direct;
	TEAGN_TRUE_OR_THROW(pc_direct.Reset(params_direct), "Invalid Parameters");
	pc_direct.ToggleProgInt(progress_enabled_);
	TEAGN_TRUE_OR_THROW(pc_direct.Apply(), "Apply error");

/* Computing statistics to fit the histograms */
	
	double	pixel,
		gain = 1.0,
		offset = 0.0;

	if (fit_histogram)
	{
		TePDIStatistic stat;
		
		TePDIParameters stat_pars;
		TePDITypes::TePDIRasterVectorType stat_rasters;
		stat_rasters.push_back(output_rasters_direct[0]);
		stat_rasters.push_back(reference_raster);
		stat_pars.SetParameter("rasters", stat_rasters);
		std::vector<int> stat_bands;
		stat_bands.push_back(0);
		stat_bands.push_back(reference_raster_band);
		stat_pars.SetParameter("bands", stat_bands);
		
		TEAGN_TRUE_OR_RETURN(stat.Reset(stat_pars), "Unable to inialize the statistc module");
		stat.ToggleProgInt(false);
	
		TeMatrix	std_matrix = stat.getStdDevMatrix(),
				mean_matrix = stat.getMeanMatrix();
	
/* Swapping reference raster by first principal component */

		gain = std_matrix(0, 0) / std_matrix(1, 0);
		offset = mean_matrix(0, 0) - (gain * mean_matrix(0, 1));
	}

	TePDIPIManager progress("Swapping PCA 1 by reference raster", reference_raster->params().ncols_, progress_enabled_);
	for (int j = 0; j < reference_raster->params().nlines_; j++)
	{
		for (int i = 0; i < reference_raster->params().ncols_; i++)
		{
			if (reference_raster->getElement(i, j, pixel, reference_raster_band))
				output_rasters_direct[0]->setElement(i, j, gain*pixel+offset, 0);
		}
		progress.Increment();
	}
	progress.Toggle(false);

/* Inverse analysis */

	analysis_type = TePDIPrincipalComponents::TePDIPCAInverse;
	
	TePDIParameters params_inverse;

	std::vector< int > bands_inverse(output_rasters.size(), 0);

	params_inverse.SetParameter("analysis_type", analysis_type);
	params_inverse.SetParameter("input_rasters", output_rasters_direct);
	params_inverse.SetParameter("bands", bands_inverse);
	params_inverse.SetParameter("output_rasters", output_rasters);
	params_inverse.SetParameter("covariance_matrix", covariance_matrix);
	
	TePDIPrincipalComponents pc_inverse;
	TEAGN_TRUE_OR_THROW(pc_inverse.Reset(params_inverse), "Invalid Parameters");
	pc_inverse.ToggleProgInt(progress_enabled_);
	TEAGN_TRUE_OR_THROW(pc_inverse.Apply(), "Apply error");

	return true;
}
