/*
TerraLib - a library for developing GIS applications.
Copyright  2001, 2002, 2003 INPE and Tecgraf/PUC-Rio.

This code is part of the TerraLib library.
This library is free software; you can redistribute it and/or
modify it under the terms of the GNU Lesser General Public
License as published by the Free Software Foundation; either
version 2.1 of the License, or (at your option) any later version.

You should have received a copy of the GNU Lesser General Public
License along with this library.

The authors reassure the license terms regarding the warranties.
They specifically disclaim any warranties, including, but not limited to,
the implied warranties of merchantability and fitness for a particular
purpose. The library provided hereunder is on an "as is" basis, and the
authors have no obligation to provide maintenance, support, updates,
enhancements, or modifications.
In no event shall INPE be held liable to any party
for direct, indirect, special, incidental, or consequential damages arising
out of the use of this library and its documentation.
*/

#ifndef TEPDIARITHMETIC_HPP
  #define TEPDIARITHMETIC_HPP

  #include "TePDILevelRemap.hpp"
  #include "TePDIParameters.hpp"
  #include "TePDITypes.hpp"
  #include <TeSharedPtr.h>

  /**
   * @brief  This is the base class for level remapping algorithms.
   * @author Emiliano F. Castejon <castejon@dpi.inpe.br>
   *
   * @note The general required parameters are:
   *
   * @param arithmetic_type ( TePDIArithmetic::TePDIArithmeticType) - 
   * Arithmetic type.
   * @param input_image1 (TePDITypes::TePDIRasterPtrType).
   * @param input_image2 (TePDITypes::TePDIRasterPtrType) - Not used by 
   * TePDIAType1.
   * @param img1_chan (int) - Band to process for image 1.
   * @param img2_chan (int) - Band to process for image 2 - Not used by 
   * TePDIAType1.
   * @param gain ( double ).
   * @param offset ( double ).
   * @param output_image (TePDITypes::TePDIRasterPtrType).
   *
   * @note The following paramters are optional and will be used if present.
   *
   * @param normalize_output ( int = 1 ) - If present ( and with value 1 )
   * output level normalization will be activated.
   *
   * @ingroup TePDIGeneralAlgoGroup
   */
  class PDI_DLL TePDIArithmetic : public TePDILevelRemap {
    public :

      /**
       * @enum TePDIArithmeticType Arithmetic Types.
       */       
      enum TePDIArithmeticType{
        /** @brief Gain * A + Offset */
        TePDIAType1 = 1,
        /** @brief Gain * ( A - B ) + Offset */
        TePDIAType2 = 2,
        /** @brief Gain * ( A + B ) + Offset */
        TePDIAType3 = 3,
        /** @brief Gain * ( A / B ) + Offset */
        TePDIAType4 = 4,
        /** @brief ( Gain * ( A - B ) / ( A + B ) ) + Offset */
        TePDIAType5 = 5
      };

      /**
       * @brief Default Constructor.
       *
       */
      TePDIArithmetic();

      /**
       * @brief Default Destructor
       */
      ~TePDIArithmetic();

      /**
       * @brief Checks if the supplied parameters fits the requirements of each
       * PDI algorithm implementation.
       *
       * @note Error log messages must be generated. No exceptions generated.
       *
       * @param parameters The parameters to be checked.
       * @return true if the parameters are OK. false if not.
       */
      bool CheckParameters( const TePDIParameters& parameters ) const;

    protected :
      /**
       * @brief Runs the current algorithm implementation.
       *
       * @return true if OK. false on error.
       */
      bool RunImplementation();

      /**
       * @brief Reset the internal state to the initial state.
       *
       * @param params The new parameters referente at initial state.
       */
      void ResetState( const TePDIParameters& params );

      /**
       * @brief Optmized binding function.
       *
       * @param level Input pixel level.
       * @param gain Input gain.
       * @param offset Input offset.
       * @return Resultant value.
       */      
      inline static double type1_arith( double level, double gain, 
        double offset ) { return ( gain * level ) + offset; };

      /**
       * @brief Optmized binding function.
       *
       * @param level1 Input pixel level 1.
       * @param level2 Input pixel level 2.
       * @param gain Input gain.
       * @param offset Input offset.
       * @return Resultant value.
       */      
      inline static double type2_arith( double level1, double level2, 
        double gain,  double offset )
        { return ( gain * ( level1 - level2 ) ) + offset; };

      /**
       * @brief Optmized binding function.
       *
       * @param level1 Input pixel level 1.
       * @param level2 Input pixel level 2.
       * @param gain Input gain.
       * @param offset Input offset.
       * @return Resultant value.
       */      
      inline static double type3_arith( double level1, double level2, 
        double gain, double offset )
        { return ( gain * ( level1 + level2 ) ) + offset; };

      /**
       * @brief Optmized binding function.
       *
       * @param level1 Input pixel level 1.
       * @param level2 Input pixel level 2.
       * @param gain Input gain.
       * @param offset Input offset.
       * @return Resultant value.
       */      
      inline static double type4_arith( double level1, double level2, 
        double gain, double offset )
        { return ( gain * ( level1 / level2 ) ) + offset; };

      /**
       * @brief Optmized binding function.
       *
       * @param level1 Input pixel level 1.
       * @param level2 Input pixel level 2.
       * @param gain Input gain.
       * @param offset Input offset.
       * @return Resultant value.
       */      
      inline static double type5_arith( double level1, double level2, 
        double gain, double offset )
        { return ( gain * ( level1 - level2 ) / ( level1 + level2 ) ) + 
        offset; };

  };

/** @example TePDIArithmetic_test.cpp
 *    Shows how to use this class.
 */  
  
#endif //TEPDIARITHMETIC_HPP
