/* DropHandler.java
 * =========================================================================
 * This file is part of the SWIRL Library - http://swirl-lib.sourceforge.net
 * 
 * Copyright (C) 2005-2008 Universiteit Gent
 * 
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or (at
 * your option) any later version.
 * 
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * General Public License for more details.
 * 
 * A copy of the GNU General Public License can be found in the file
 * LICENSE.txt provided with the source distribution of this program (see
 * the META-INF directory in the source jar). This license can also be
 * found on the GNU website at http://www.gnu.org/licenses/gpl.html.
 * 
 * If you did not receive a copy of the GNU General Public License along
 * with this program, contact the lead developer, or write to the Free
 * Software Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA
 * 02110-1301, USA.
 * 
 */

package be.ugent.caagt.swirl.dnd;

import javax.swing.JComponent;

/**
 * Handles the 'drop' part of a drag-and-drop operation of a local object of given type.
 * Used in conjunction with {@link LocalTransferHandler}. 
 * Handles objects that belong to a single class.
 */
public interface DropHandler {
    
    
    /**
     * Return the class of elements that can be dropped by this handler
     * onto the given component.
     */
    public Class<?> getDropClass (JComponent target);
    
    /**
     * Does this handler allow multiple objects of this class to be dropped
     * at the same time? 
     * @param target Target onto which the object would be dropped.
     */
    public boolean allowsMultipleDrops (JComponent target);
    
    /**
     * Accept an object which was dropped onto the component handled by this handler.
     * When multiple drops are allowed, this method will be called once for every
     * object being dropped, unless a previous drop in the same bunch was not accepted.<p>
     * It is advisable not to return false except possibly for the first element in the drop,
     * because the drag source is notified only of the fact that the entire drop did not succeed, 
     * and not of the exact number of objects that were accepted.
     * 
     * @param object Object being dropped. The object is known to belong to the correct class. 
     * @param target Target onto which the object would be dropped.
     * @param seqNr Sequence number of this object within a set of objects being dropped in one
     * go. The first object has sequence number 0.
     * @return true if and only if the object was accepted.
     */
    public boolean acceptDrop (JComponent target, Object object, int seqNr);
    
}
