/* spfmilter - SPF mail filter module
**
** Copyright ? 2004 by Jef Poskanzer <jef@acme.com>.
** All rights reserved.
**
** Redistribution and use in source and binary forms, with or without
** modification, are permitted provided that the following conditions
** are met:
** 1. Redistributions of source code must retain the above copyright
**  notice, this list of conditions and the following disclaimer.
** 2. Redistributions in binary form must reproduce the above copyright
**  notice, this list of conditions and the following disclaimer in the
**  documentation and/or other materials provided with the distribution.
**
** THIS SOFTWARE IS PROVIDED BY THE AUTHOR AND CONTRIBUTORS ``AS IS'' AND
** ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE
** IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE
** ARE DISCLAIMED.  IN NO EVENT SHALL THE AUTHOR OR CONTRIBUTORS BE LIABLE
** FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL
** DAMAGES (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS
** OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR BUSINESS INTERRUPTION)
** HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT
** LIABILITY, OR TORT (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY
** OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF
** SUCH DAMAGE.
**
** For commentary on this license please see http://www.acme.com/license.html
*/

#include "spfmilter.h"


/* main
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int main(int argc, char **argv)
{
  char *guess_str;
  char *fallback_filename;
  char *whitelist_filename;
  char *user;
  char *pidfile;
  char *sockpath;
  int nodaemon;
#ifdef HAVE_GETOPT_LONG
  int idx;
#endif
  char c;

  localpolicy_str     = (char *)0;
  trustedforwarders   = 0;
  guess_str           = (char *)0;
  fallback_filename   = (char *)0;
  whitelist_filename  = (char *)0;
  recipientmx         = 0;
  explanation_str     = (char *)0;
  markonly            = 0;
  user                = (char *)0;
  pidfile             = (char *)0;
  nodaemon            = 0;
  debug               = 0;

  /* Figure out the program's name. */
  argv0 = strrchr(argv[0], '/');

  if (argv0 != (char *)0)
  {
    ++argv0;
  }
  else
  {
    argv0 = argv[0];
  }

  header_name_len = strlen(HEADER_NAME);

  /* Parse args. */
  while ((c =
#ifdef HAVE_GETOPT_LONG
    getopt_long(argc, argv, shortopts, longopts, &idx)
#else
    getopt(argc, argv, shortopts)
#endif
   ) != -1)
  {
    switch (c)
    {
      case 'l':
        localpolicy_str = optarg;
        break;
      case 't':
        trustedforwarders = 1;
        break;
      case 'g':
        guess_str = optarg;
        break;
      case 'f':
        fallback_filename = optarg;
        break;
      case 'w':
        whitelist_filename = optarg;
        break;
      case 'r':
        recipientmx = 1;
        break;
      case 'e':
        explanation_str = optarg;
        break;
      case 'm':
        markonly = 1;
        break;
      case 'u':
        user = optarg;
        break;
      case 'p':
        pidfile = optarg;
        break;
      case 'X':
        nodaemon = 1;
        break;
      case 'h':
      case '?':
        usage();
        exit(1);
      case 'd':
        if (optarg)
        {
          debug = atoi(optarg);
        }
        else
        {
          debug = 1;
        }
        break;
      default:
        (void) fprintf(stderr, "Unrecognised option '%c'\n", c);
        exit(1);
    } /* switch */
  } /* while */

  argv += optind;
  argc -= optind;

  if (argc != 1)
  {
    usage();
    exit(1);
  }
  sockpath = argv[0];

  local_hostname = (char*) 0;
  if (! lib_init())
  {
    (void) fprintf(stderr, "%s: lib_init() failed\n", argv0);
    exit(1);
  }

  init_fallback(fallback_filename, guess_str);
  init_whitelist(whitelist_filename);
  init_uid(user);
  init_socket(sockpath);
  openlog(argv0, 0, LOG_MAIL);

  if (!nodaemon)
  {
    init_daemon();
  }
  init_pidfile(pidfile);

  syslog(LOG_NOTICE, "%s %s starting", SPFMILTER_PROGRAM, SPFMILTER_VERSION);

  if (smfi_main() == MI_FAILURE)
  {
    syslog(LOG_ERR, "smfi_main() failed");
    exit(1);
  }

  syslog(LOG_NOTICE, "%s %s terminating", SPFMILTER_PROGRAM, SPFMILTER_VERSION);

  fini_pidfile(pidfile);
  fini_socket(sockpath);
  fini_whitelist();
  fini_fallback();
  lib_fini();
  if (local_hostname != (char*)0)
  {
    free((void*)local_hostname);
  }

  closelog();

  return 0;
}


/* usage
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void usage(void)
{
  (void) fprintf(stderr,
    "usage: %s [options] /path/to/socket\n", argv0);

  DOC_OPT('l', "localpolicy", "<spf-mechanism>",
    "Add local SPF policy mechanisms.", 5, 4);

  DOC_OPT('t', "trustedforwarders", (char*)0,
    "Use the trusted-forwarder.org whitelist.", 20, 14);

  DOC_OPT('g', "guess", "<spf-mechanisms>",
    "Add guess SPF policy mechanisms.", 4, 4);

  DOC_OPT('f', "fallback", "<filename>",
   "A file of fallback SPF mechanisms.", 10, 14);

  DOC_OPT('w', "whitelist", "<filename>",
   "A file of IP addresses to always accept mail from.", 10, 14);

  DOC_OPT('r', "recipientmx", (char*) 0,
   "Check the MX records of the message's reqipients.", 20, 14);

  DOC_OPT('e', "explanation", "<exp>",
   "Change the message returned in mail bounces.", 15, 14);

  DOC_OPT('m', "markonly", (char*) 0,
   "Mark email instead of rejecting it.", 20, 22);

  DOC_OPT('u', "user", "<user|uid>",
   "Run as specified user or UID.", 10, 16);

  DOC_OPT('p', "pidfile", "<filename>",
   "Write the process i.d. to the specified file.", 10, 16);

  DOC_OPT('X', "nodaemon", (char*) 0,
   "Do not fork into the background.", 20, 26);

  DOC_OPT('h', "help", (char*) 0,
    "Show this help.", 20, 26);

  DOC_OPT('d', "debug", "[<int>]",
    "Enable debugging to syslog.", 13, 18);

  return;
}


/* init_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void init_fallback(const char* fallback_filename, const char *guess_str)
{
  FILE *fp;
  char *cp;
  char line[10000];

  max_fallbacks = num_fallbacks = 0;
  fallbacks = (fallback_t*) 0;

  if (fallback_filename != (char*)0)
  {

    fp = fopen(fallback_filename, "r");

    if (fp == (FILE*) 0)
    {
      perror(fallback_filename);
      exit(1);
    }

    while (fgets(line, sizeof(line), fp) != (char*)0)
    {
      trim(line);
      if (line[0] == '\0')
      {
        continue;
      }
      cp = line;
      cp += strcspn(cp, " \t");
      *cp = '\0';
      ++cp;
      cp += strspn(cp, " \t");
      add_fallback(line, cp);
    }

    (void)fclose(fp);
  } /* if */

  if (guess_str != (char*)0)
  add_fallback("*", guess_str);

  return;
}


/* add_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void add_fallback(const char *pattern, const char *str)
{
  if (num_fallbacks >= max_fallbacks)
  {
    if (max_fallbacks == 0)
    {
      max_fallbacks = 50;   /* arbitrary */
      fallbacks = (fallback_t *)malloc((max_fallbacks * sizeof(fallback_t)));
    }
    else
    {
      max_fallbacks *= 2;
      fallbacks = (fallback_t *) realloc((void*) fallbacks,
        (max_fallbacks * sizeof(fallback_t)));
    }

    if (fallbacks == (fallback_t*) 0)
    {
      (void)fprintf(stderr,
        "%s: out of memory enlarging fallbacks array\n", argv0);
      exit(1);
    }
  }

  fallbacks[num_fallbacks].pattern = strdup(pattern);
  fallbacks[num_fallbacks].str = strdup(str);
  if (fallbacks[num_fallbacks].pattern == (char*) 0 ||
       fallbacks[num_fallbacks].str == (char*) 0)
  {
    (void) fprintf(stderr,
      "%s: out of memory saving a fallback entry\n", argv0);
    exit(1);
  }

  fallbacks[num_fallbacks].lib_fallback = lib_init_fallback(
  fallbacks[num_fallbacks].str);

  if (fallbacks[num_fallbacks].lib_fallback == (lib_fallback_t*)0)
  {
    (void) fprintf(stderr,
      "%s: error compiling fallback mechanism '%s'\n",
        argv0, fallbacks[num_fallbacks].str);

    exit(1);
  }

    ++num_fallbacks;
    return;
 }


/* fini_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void fini_fallback(void)
{
  if (fallbacks != (fallback_t*)0)
  {
    int i;

    for (i = 0; i < num_fallbacks; ++i)
    {
      free((void *)fallbacks[i].pattern);
      free((void *)fallbacks[i].str);
      lib_fini_fallback(fallbacks[i].lib_fallback);
    }

    max_fallbacks = num_fallbacks = 0;
    free((void *)fallbacks);
    fallbacks = (fallback_t*)0;
  }
}


/* find_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int find_fallback(const char *from)
{
  int i;
  const char *from_host;

  from_host = strchr(from, '@');
  if (from_host == (char*)0)
  {
    return -1;
  }

  ++from_host;
  if (fallbacks == (fallback_t*)0)
  {
    return -1;
  }

  for (i = 0; i < num_fallbacks; ++i)
  {
    if (match(fallbacks[i].pattern, from_host))
    {
      return(i);
    }
  }

  return(-1);
}


/* init_whitelist
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void init_whitelist(const char* whitelist_filename)
{
  FILE *fp;
  octets ip;
  char line[10000];

  if (whitelist_filename != (char*)0)
  {
    whitelist = iparray_new();

    if (whitelist == (iparray) 0)
    {
      (void)fprintf(stderr,
        "%s: whitelist create failed\n", argv0);
      exit(1);
    }

    fp = fopen(whitelist_filename, "r");

    if (fp == (FILE*)0)
    {
      perror(whitelist_filename);
      exit(1);
    }

    while (fgets(line, sizeof(line), fp) != (char*)0)
    {
      trim(line);

      if (line[0] == '\0')
      {
        continue;
      }

      if (iparray_parse_octets(line, &ip))
      {
        (void)iparray_incr(whitelist, ip);
      }
      else
      {
        (void)fprintf(stderr,
          "%s: unparsable IP address - \"%s\"\n", argv0, line);
      }
    } /* while */

    (void)fclose(fp);
  }
  else
  {
    whitelist = (iparray)0;
  }
}


/* fini_whitelist
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void fini_whitelist(void)
{
  if (whitelist != (iparray)0)
  {
    iparray_delete(whitelist);
  }
}


/* trim
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void trim(char *str)
{
  char* cp;
  int len;

  cp = strchr(str, '#');
  if (cp != (char*) 0)
  {
    *cp = '\0';
  }

  len = strlen(str);
  while (str[len-1] == '\n' || str[len-1] == '\r' ||
          str[len-1] == ' ' || str[len-1] == '\t')
  {
    --len;
    str[len] = '\0';
  }
  return;
}

/* OLD CODE BEFORE CLEAN UP
int
main(int argc, char** argv)
    {
    char* guess_str;
    char* fallback_filename;
    char* whitelist_filename;
    char* user;
    char* pidfile;
    int nodaemon;
    char* sockpath;
#ifdef HAVE_GETOPT_LONG
    int idx;
#endif
    char c;

    localpolicy_str = (char*) 0;
    trustedforwarders = 0;
    guess_str = (char*) 0;
    fallback_filename = (char*) 0;
    whitelist_filename = (char*) 0;
    recipientmx = 0;
    explanation_str = (char*) 0;
    markonly = 0;
    user = (char*) 0;
    pidfile = (char*) 0;
    nodaemon = 0;
    debug = 0;

    *//* Figure out the program's name. *//*
    argv0 = strrchr(argv[0], '/');
    if (argv0 != (char*) 0)
  ++argv0;
    else
  argv0 = argv[0];

    header_name_len = strlen(HEADER_NAME);

    *//* Parse args. *//*
    while ((c =
#ifdef HAVE_GETOPT_LONG
      getopt_long(argc, argv, shortopts, longopts, &idx)
#else
      getopt(argc, argv, shortopts)
#endif
     ) != -1)
  {
  switch (c)
      {
      case 'l':
    localpolicy_str = optarg;
    break;
      case 't':
    trustedforwarders = 1;
    break;
      case 'g':
    guess_str = optarg;
    break;
      case 'f':
    fallback_filename = optarg;
    break;
      case 'w':
    whitelist_filename = optarg;
    break;
      case 'r':
    recipientmx = 1;
    break;
      case 'e':
    explanation_str = optarg;
    break;
      case 'm':
    markonly = 1;
    break;
      case 'u':
    user = optarg;
    break;
      case 'p':
    pidfile = optarg;
    break;
      case 'X':
    nodaemon = 1;
    break;
      case 'h':
      case '?':
    usage();
    exit(1);
      case 'd':
    if (optarg)
        debug = atoi(optarg);
    else
        debug = 1;
    break;
      default:
    (void) fprintf(stderr, "Unrecognised option '%c'\n", c);
    exit(1);
      }
  }
    argv += optind;
    argc -= optind;

    if (argc != 1)
  {
  usage();
  exit(1);
  }
    sockpath = argv[0];

    local_hostname = (char*) 0;
    if (! lib_init())
  {
  (void) fprintf(stderr, "%s: lib_init() failed\n", argv0);
  exit(1);
  }
    init_fallback(fallback_filename, guess_str);
    init_whitelist(whitelist_filename);
    init_uid(user);
    init_socket(sockpath);
    openlog(argv0, 0, LOG_MAIL);
    if (! nodaemon)
  init_daemon();
    init_pidfile(pidfile);

    syslog(LOG_NOTICE, "%s %s starting", SPFMILTER_PROGRAM,
      SPFMILTER_VERSION);
    if (smfi_main() == MI_FAILURE)
  {
  syslog(LOG_ERR, "smfi_main() failed");
  exit(1);
  }
    syslog(LOG_NOTICE, "%s %s terminating", SPFMILTER_PROGRAM,
      SPFMILTER_VERSION);

    fini_pidfile(pidfile);
    fini_socket(sockpath);
    fini_whitelist();
    fini_fallback();
    lib_fini();
    if (local_hostname != (char*) 0)
  free((void*) local_hostname);

    closelog();

    return 0;
    }


void
usage(void)
    {
    (void) fprintf(stderr, "usage: %s [options] /path/to/socket\n", argv0);
    DOC_OPT('l', "localpolicy", "<spf-mechanism>", "Add local SPF policy
mechanisms.", 5, 4);
    DOC_OPT('t', "trustedforwarders", (char*) 0, "Use the trusted-forwarder.org
whitelist.", 20, 14);
    DOC_OPT('g', "guess", "<spf-mechanisms>", "Add guess SPF policy
mechanisms.", 4, 4);
    DOC_OPT('f', "fallback", "<filename>", "A file of fallback SPF
mechanisms.", 10, 14);
    DOC_OPT('w', "whitelist", "<filename>", "A file of IP addresses to always
accept mail from.", 10, 14);
    DOC_OPT('r', "recipientmx", (char*) 0, "Check the MX records of the
message's reqipients.", 20, 14);
    DOC_OPT('e', "explanation", "<exp>", "Change the message returned in mail
bounces.", 15, 14);
    DOC_OPT('m', "markonly", (char*) 0, "Mark email instead of rejecting it.",
20, 22);
    DOC_OPT('u', "user", "<user|uid>", "Run as specified user or UID.", 10, 16
);
    DOC_OPT('p', "pidfile", "<filename>", "Write the process i.d. to the
specified file.", 10, 16);
    DOC_OPT('X', "nodaemon", (char*) 0, "Do not fork into the background.", 20,
26);
    DOC_OPT('h', "help", (char*) 0, "Show this help.", 20, 26);
    DOC_OPT('d', "debug", "[<int>]", "Enable debugging to syslog.", 13, 18);
    }


void
init_fallback(const char* fallback_filename, const char* guess_str)
    {
    FILE* fp;
    char line[10000];
    char* cp;

    max_fallbacks = num_fallbacks = 0;
    fallbacks = (fallback_t*) 0;

    if (fallback_filename != (char*) 0)
  {
  fp = fopen(fallback_filename, "r");
  if (fp == (FILE*) 0)
      {
      perror(fallback_filename);
      exit(1);
      }
  while (fgets(line, sizeof(line), fp) != (char*) 0)
      {
      trim(line);
      if (line[0] == '\0')
    continue;
      cp = line;
      cp += strcspn(cp, " \t");
      *cp = '\0';
      ++cp;
      cp += strspn(cp, " \t");
      add_fallback(line, cp);
      }
  (void) fclose(fp);
  }

    if (guess_str != (char*) 0)
  add_fallback("*", guess_str);
    }


void
add_fallback(const char* pattern, const char* str)
    {
    if (num_fallbacks >= max_fallbacks)
  {
  if (max_fallbacks == 0)
      {
      max_fallbacks = 50;   *//* arbitrary *//*
      fallbacks = (fallback_t*) malloc(max_fallbacks * sizeof(fallback_t));
      }
  else
      {
      max_fallbacks *= 2;
      fallbacks = (fallback_t*) realloc((void*) fallbacks, max_fallbacks *
sizeof(fallback_t));
      }
  if (fallbacks == (fallback_t*) 0)
      {
      (void) fprintf(stderr, "%s: out of memory enlarging fallbacks array\n",
argv0);
      exit(1);
      }
  }
    fallbacks[num_fallbacks].pattern = strdup(pattern);
    fallbacks[num_fallbacks].str = strdup(str);
    if (fallbacks[num_fallbacks].pattern == (char*) 0 ||
         fallbacks[num_fallbacks].str == (char*) 0)
  {
  (void) fprintf(stderr, "%s: out of memory saving a fallback entry\n", argv0
);
  exit(1);
  }

    fallbacks[num_fallbacks].lib_fallback = lib_init_fallback(
fallbacks[num_fallbacks].str);
    if (fallbacks[num_fallbacks].lib_fallback == (lib_fallback_t*) 0)
  {
  (void) fprintf(stderr, "%s: error compiling fallback mechanism '%s'\n",
argv0, fallbacks[num_fallbacks].str);
  exit(1);
  }

    ++num_fallbacks;
    }

*/
      /*
void
fini_fallback(void)
    {
    if (fallbacks != (fallback_t*) 0)
  {
  int i;

  for (i = 0; i < num_fallbacks; ++i)
      {
      free((void*) fallbacks[i].pattern);
      free((void*) fallbacks[i].str);
      lib_fini_fallback(fallbacks[i].lib_fallback);*//*
      }
  max_fallbacks = num_fallbacks = 0;
  free((void*) fallbacks);
  fallbacks = (fallback_t*) 0;
  }
    }

*//*
int
find_fallback(const char* from)
    {
    const char* from_host;
    int i;

    from_host = strchr(from, '@');
    if (from_host == (char*) 0)
  return -1;
    ++from_host;
    if (fallbacks == (fallback_t*) 0)
  return -1;
    for (i = 0; i < num_fallbacks; ++i)
  if (match(fallbacks[i].pattern, from_host))
      return i;
    return -1;
    }


void
init_whitelist(const char* whitelist_filename)
    {
    FILE* fp;
    char line[10000];
    octets ip;

    if (whitelist_filename != (char*) 0)
  {
  whitelist = iparray_new();
  if (whitelist == (iparray) 0)
      {
      (void) fprintf(stderr, "%s: whitelist create failed\n", argv0);
      exit(1);
      }
  fp = fopen(whitelist_filename, "r");
  if (fp == (FILE*) 0)
      {
      perror(whitelist_filename);
      exit(1);
      }
  while (fgets(line, sizeof(line), fp) != (char*) 0)
      {
      trim(line);
      if (line[0] == '\0')
    continue;
      if (iparray_parse_octets(line, &ip))
    (void) iparray_incr(whitelist, ip);
      else
    (void) fprintf(stderr, "%s: unparsable IP address - \"%s\"\n", argv0, line
);
      }
  (void) fclose(fp);
  }
    else
  whitelist = (iparray) 0;
    }


void
fini_whitelist(void)
    {
    if (whitelist != (iparray) 0)
  iparray_delete(whitelist);
    }
*/

void init_uid(const char* user)
{
  struct passwd* pwd;
  char* ep;
  int uid;

  if (getuid() == 0)
  {
    /* If we're root, the very first thing we do is become another user. */
    if (user == (char*)0)
    {
      (void)fprintf(stderr,
        "%s: warning: started as root but no --user flag specified\n", argv0);
    }
    else
    {
      /* Is it a number? */
      uid = strtol(user, &ep, 0);
      if (*ep == '\0')
        pwd = getpwuid(uid);
      else
        pwd = getpwnam(user);
      if (pwd == (struct passwd*) 0)
      {
        (void) fprintf(stderr, "%s: unknown user: '%s'\n", argv0, user);
        exit(1);
      }
      /* Set aux groups to null. */
      if (setgroups(0, (gid_t*) 0) < 0)
    {
    perror("setgroups");
    exit(1);
    }
      /* Set primary group. */
      if (setgid(pwd->pw_gid) < 0)
    {
    perror("setgid");
    exit(1);
    }

      /* Try setting aux groups correctly - not critical if this fails. */
      if (initgroups(user, pwd->pw_gid) < 0)
        perror("initgroups");
      /* Set uid. */
      if (setuid(pwd->pw_uid) < 0)
    {
    perror("setuid");
    exit(1);
    }
      }
  }
    else
  {
  /* If we're not root but got a -user flag anyway, that's an error. */
  if (user != (char*) 0)
      {
      (void) fprintf(stderr, "%s: can't switch users if not started as root\n",
argv0);
      exit(1);
      }
  }
    }


void
init_socket(const char* sockpath)
    {
    unlink_socket(sockpath);

    /* Harden our umask so that the new socket gets created securely. */
    umask(0077);

    /* Initialize milter stuff. */
    smfi_setconn((char*) sockpath);
    if (smfi_register(smfilter) == MI_FAILURE)
  {
  (void) fprintf(stderr, "%s: smfi_register() failed\n", argv0);
  exit(1);
  }
    }


void
fini_socket(const char* sockpath)
    {
    unlink_socket(sockpath);
    }


void
unlink_socket(const char* sockpath)
    {
    /* Remove old local socket.  Actually we should probably stat
    ** the file and check that it is in fact a socket before removing
    ** it.
    */
    if (strncasecmp(sockpath, "unix:", 5) == 0)
  {
  if (unlink(&sockpath[5]) < 0)
      if (errno != ENOENT)
    perror(&sockpath[5]);
  }
    else if (strncasecmp(sockpath, "local:", 6) == 0)
  {
  if (unlink(&sockpath[6]) < 0)
      if (errno != ENOENT)
    perror(&sockpath[6]);
  }
    else if (strchr(sockpath, ':') == (char*) 0)
  {
  if (unlink(sockpath) < 0)
      if (errno != ENOENT)
    perror(sockpath);
  }
    }


void
init_daemon(void)
    {
    /* Daemonize. */
#ifdef HAVE_DAEMON
    if (daemon(0, 0) < 0)
  {
  perror("daemon");
  exit(1);
  }
#else /* HAVE_DAEMON */
    switch (fork())
  {
  case 0:
      break;
  case -1:
      syslog(LOG_CRIT, "fork: %m");
      perror("fork");
      exit(1);
  default:
      exit(0);
  }
#ifdef HAVE_SETSID
    setsid();
#endif /* HAVE_SETSID */
#endif /* HAVE_DAEMON */
    }


void
init_pidfile(const char* pidfile)
    {
    if (pidfile != (char*) 0)
  {
  FILE* fp;

  fp = fopen(pidfile, "w");
  if (fp == (FILE*) 0)
      syslog(LOG_ERR, "unable to write PID file - %m");
  else
      {
      (void) fprintf(fp, "%ld\n", (long) getpid());
      (void) fclose(fp);
      }
  }
    }


void
fini_pidfile(const char* pidfile)
    {
    if (pidfile != (char*) 0)
  (void) unlink(pidfile);
    }


/* Milter routines. */


/* spf_connect - handle the initial TCP connection
**
** Called at the start of a connection.  Any per-connection data should
** be initialized here.
**
** connhost: The hostname of the client, based on a reverse lookup.
** connaddr: The client's IP address, based on getpeername().
*/
sfsistat
spf_connect(SMFICTX* ctx, char* connhost, _SOCK_ADDR* connaddr)
    {
    connection_data_t* cd;
    char* jmacro;
    struct sockaddr_in* sin;
    struct sockaddr_in6* sin6;
    char str[1000];
    const char* r;

    /* Create & initialize the connection_data object. */
    cd = init_connection_data();
    if (cd == (connection_data_t*) 0)
  {
  syslog(LOG_ERR, "couldn't allocate connection_data");
  return SMFIS_TEMPFAIL;
  }
    if (smfi_setpriv(ctx, (void*) cd) != MI_SUCCESS)
  {
  syslog(LOG_ERR, "smfi_setpriv() failed");
  fini_connection_data(cd);
  return SMFIS_TEMPFAIL;
  }

    /* The first time through, we figure out the local hostname. */
    if (local_hostname == (char*) 0)
  {
  jmacro = smfi_getsymval(ctx, "{j}");
  if (jmacro == (char*) 0)
      {
      syslog(LOG_ERR, "couldn't get local hostname");
      return SMFIS_TEMPFAIL;
      }
  /* It's probably safe to just save the pointer returned by
  ** smfi_getsymval, but strduping it is guaranteed safe so
  ** why not.
  */
  local_hostname = strdup(jmacro);
  if (local_hostname == (char*) 0)
      {
      syslog(LOG_ERR, "couldn't strdup local hostname");
      return SMFIS_TEMPFAIL;
      }
  local_hostname_len = strlen(local_hostname);
  if (! lib_set_local_hostname(cd->lib_data))
      {
      syslog(LOG_ERR, "lib_set_local_hostname() failed");
      return SMFIS_TEMPFAIL;
      }
  }

    /* Set the IP address. */
    switch (connaddr->sa_family)
  {
  case AF_INET:
      sin = (struct sockaddr_in*) connaddr;
      if (whitelist != (iparray) 0)
    {
    char* char_addr = (char*) &sin->sin_addr.s_addr;
    octets ip;

    ip.a = char_addr[0];
    ip.b = char_addr[1];
    ip.c = char_addr[2];
    ip.d = char_addr[3];
    ip.w = 32;
    if (iparray_get(whitelist, ip) > 0)
        {
        syslog(LOG_INFO, "whitelist \"%s\" [%d.%d.%d.%d]", connhost, (int)
ip.a, (int) ip.b, (int) ip.c, (int) ip.d);
        cd->whitelisted = 1;
        }
    }
      r = inet_ntop(AF_INET, &sin->sin_addr, str, sizeof(str));
      if (r == (char*) 0)
    {
    syslog(LOG_ERR, "couldn't convert IPv4 address to a string - %m");
    return SMFIS_TEMPFAIL;
    }
      if (! lib_set_ipv4(cd->lib_data, sin->sin_addr, str))
    {
    syslog(LOG_ERR, "lib_set_ipv4() failed");
    return SMFIS_TEMPFAIL;
    }
      break;
  case AF_INET6:
      sin6 = (struct sockaddr_in6*) connaddr;
      r = inet_ntop(AF_INET6, &sin6->sin6_addr, str, sizeof(str));
      if (r == (char*) 0)
    {
    syslog(LOG_ERR, "couldn't convert IPv6 address to a string - %m");
    return SMFIS_TEMPFAIL;
    }
      if (! lib_set_ipv6(cd->lib_data, sin6->sin6_addr, str))
    {
    syslog(LOG_ERR, "lib_set_ipv6() failed");
    return SMFIS_TEMPFAIL;
    }
      break;
  default:
      syslog(LOG_ERR, "unknown address family");
      return SMFIS_TEMPFAIL;
  }
    cd->ip_str = strdup(str);
    if (cd->ip_str == (char*) 0)
  {
  syslog(LOG_ERR, "couldn't strdup ip_str");
  return SMFIS_TEMPFAIL;
  }

    return SMFIS_CONTINUE;
    }


/* spf_helo - handle the MAIL HELO
**
** Called at the start of a connection.
**
** helohost: The string passed to the HELO/EHLO command.
*/
sfsistat spf_helo(SMFICTX* ctx, char* helohost)
{
  connection_data_t* cd;

  cd = (connection_data_t*) smfi_getpriv(ctx);

  cd->helo = strdup(helohost);
  if (cd->helo == (char*) 0)
  {
    syslog(LOG_ERR, "couldn't strdup helo");
    return(SMFIS_TEMPFAIL);
  }

  if (cd->whitelisted)
  {
    return(SMFIS_CONTINUE);
  }

  if (! lib_set_helo_hostname(cd->lib_data, helohost))
  {
    syslog(LOG_ERR, "lib_set_helo_hostname() failed");
    smfi_setreply(ctx,
      "452", "4.3.1", "Please try again later - lib_set_helo_hostname() failed.");
    return(SMFIS_TEMPFAIL);
  }

  return(SMFIS_CONTINUE);
}


/* spf_envfrom - handle the MAIL FROM:<> command
**
** Called at the start of each message.  There may be multiple messages
** in a connection.  Any per-message data should be initialized here.
**
** The sender address is also known as the "envelope from" address.
**
** fromargs: Null-terminated SMTP command arguments; fromargs[0] is the
**       sender address.
*/
sfsistat spf_envfrom(SMFICTX* ctx, char** fromargs)
{
  connection_data_t* cd;
  char* from;
  int len;

  cd = (connection_data_t*) smfi_getpriv(ctx);

  if (! init_message_data(cd))
  {
    syslog(LOG_ERR, "init_message_data() failed");
    return SMFIS_TEMPFAIL;
  }

  from = fromargs[0];
  /* The milter API sometimes gives us strings with angle brackets. */
  while (*from == '<')
  {
    ++from;
  }

  len = strlen(from);

  while (len > 0 && from[len - 1] == '>')
  {
    from[--len] = '\0';
  }

  cd->from = strdup(from);
  if (cd->from == (char*) 0)
  {
    syslog(LOG_ERR, "couldn't strdup from");
    return SMFIS_TEMPFAIL;
  }

  if (cd->whitelisted)
  {
    return(SMFIS_CONTINUE);
  }

    /* Authenticated messages get accepted without question. */
  if (smfi_getsymval(ctx, "{auth_type}") != (char*) 0)
  {
    cd->authenticated = 1;
    return SMFIS_CONTINUE;
  }

  if (! lib_set_from(cd->lib_data, cd->from))
  {
    syslog(LOG_ERR, "lib_set_from() failed");
    smfi_setreply(ctx,
      "452", "4.3.1", "Please try again later - lib_set_from() failed.");
    fini_message_data(cd);
    return(SMFIS_TEMPFAIL);
  }

    /* In MX mode we do a check for each recipient, and only later do the
    ** check for the whole message.
    */
  if (recipientmx)
  {
    return(SMFIS_CONTINUE);
  }

  if (! lib_do_check(cd->lib_data, cd->from))
  {
    syslog(LOG_ERR, "lib_do_check() failed");
    smfi_setreply(ctx,
      "452", "4.3.1", "Please try again later - lib_do_check() failed.");
    fini_message_data(cd);
    return(SMFIS_TEMPFAIL);
  }
  cd->result = lib_get_result(cd->lib_data);

  return handle_result(ctx);
}


/* spf_envrcpt - handle a RCPT TO:<> command
**
** Called separately for each recipient of a message.
**
** rcptargs: Null-terminated SMTP command arguments; rcptargs[0] is the
**           recipient address.
*/
sfsistat spf_envrcpt(SMFICTX* ctx, char** rcptargs)
{
  connection_data_t* cd;
  int len;
  char* rcpt;

  cd = (connection_data_t*) smfi_getpriv(ctx);

  if (cd->whitelisted || cd->authenticated)
  {
    return(SMFIS_CONTINUE);
  }

  /* If we're not in MX mode then we've already checked the message. */
  if (! recipientmx)
  {
    return(SMFIS_CONTINUE);
  }

  rcpt = rcptargs[0];
  /* The milter API sometimes gives us strings with angle brackets. */
  while (*rcpt == '<')
  {
    ++rcpt;
  }

  len = strlen(rcpt);
  while (len > 0 && rcpt[len - 1] == '>')
  {
    rcpt[--len] = '\0';
  }

  if (! lib_do_check_recipient(cd->lib_data, rcpt))
  {
    syslog(LOG_ERR, "lib_do_check_recipient() failed");
    smfi_setreply(ctx,
      "452", "4.3.1", "Please try again later - lib_do_check_recipient() failed.");
    fini_message_data(cd);
    return SMFIS_TEMPFAIL;
  }
  cd->result = lib_get_result(cd->lib_data);

  return(handle_result(ctx));
}


/* spf_header - handle a header line
**
** Called separately for each header line in a message.
**
** name:  Header field name.
** value: Header vield value, including folded whitespace.  The final CRLF
**    is removed.
*/
sfsistat
spf_header(SMFICTX* ctx, char* name, char* value)
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv(ctx);

    /* We look for any instances of our own header in the incoming
    ** message, so we can remove them later.
    */
    if (strcasecmp(name, HEADER_NAME) == 0)
  {
  ++cd->n_spf_headers;

  /* Does this SPF header claim to be from us? */
  if (receiver_is_me(value))
      {
      syslog(LOG_NOTICE, "found possible spoofed SPF header - '%s'", value);
      if (cd->n_del_headers < DEL_HEADER_MAX)
    {
    cd->del_header_list[cd->n_del_headers] = cd->n_spf_headers;
    ++cd->n_del_headers;
    }
      }
  }

    return SMFIS_CONTINUE;
    }


/* spf_eoh - handle the end of the headers
**
** Called once per message after all headers have been processed.
*/
sfsistat spf_eoh(SMFICTX* ctx)
{
  connection_data_t* cd;

  cd = (connection_data_t*) smfi_getpriv(ctx);

  if (cd->whitelisted || cd->authenticated)
  {
    return(SMFIS_CONTINUE);
  }

  if (! recipientmx)
  {
    return(SMFIS_CONTINUE);
  }

  if (! lib_do_check_final(cd->lib_data))
  {
    syslog(LOG_ERR, "lib_do_check_final() failed");
    smfi_setreply(ctx, "452", "4.3.1",
      "Please try again later - lib_do_check_final() failed.");
    fini_message_data(cd);
    return SMFIS_TEMPFAIL;
  }
  cd->result = lib_get_result(cd->lib_data);

  cd->action = SPFMILTER_ACTION_MARK;
  return(SMFIS_CONTINUE);
}


/* spf_eom - handle the end of the message
**
** Called once per message after all body blocks have been processed.
** Any per-message data should be freed both here and in spf_abort(),
** and wherever you return any of SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
sfsistat
spf_eom(SMFICTX* ctx)
    {
    connection_data_t* cd;
    int i;
    char reason[5000];
    const char* rptr;

    cd = (connection_data_t*) smfi_getpriv(ctx);

    /* Header deleting and adding can only happen in the eom handler. */

    /* Remove any previous SPF headers, to prevent spoofing. */
    if (cd->n_del_headers >= DEL_HEADER_MAX)
  {
  /* There were too many bogus SPF headers to keep track of, so to
  ** be safe we just delete all SPF headers.
  */
  for (i = cd->n_spf_headers; i >= 1; --i)
      smfi_chgheader(ctx, HEADER_NAME, i, (char*) 0);
  }
    else
  {
  /* We have a list of only the SPF headers purporting to be from us;
  ** those get deleted.  We do the deletions in reverse order in case
  ** deleting one header causes subsequent headers to get re-numbered.
  ** (I actually checked whether this happens in the current milter
  ** implementation, and it does NOT, but perhaps some future version
  ** will get it wrong.)
  */
  for (i = cd->n_del_headers - 1 ; i >= 0; --i)
      smfi_chgheader(ctx, HEADER_NAME, cd->del_header_list[i], (char*) 0);
  }

    if (cd->whitelisted || cd->authenticated)
  {
  cd->result = SPFMILTER_RESULT_PASS;
  cd->action = SPFMILTER_ACTION_MARK;
  if (cd->whitelisted)
      (void) snprintf(reason, sizeof(reason), "%s: %s is whitelisted",
local_hostname, cd->ip_str);
  else if (cd->authenticated)
      (void) snprintf(reason, sizeof(reason), "%s: authenticated connection",
local_hostname);
  else
      (void) strncpy(reason, "???", sizeof(reason));
  rptr = reason;
  }
    else
  rptr = lib_get_reason(cd->lib_data);

    if (cd->action == SPFMILTER_ACTION_MARK)
  {
  /* Build and add the SPF header. */
  char header[5000];

  build_header(cd, header, sizeof(header), rptr);
  smfi_addheader(ctx, HEADER_NAME, header);
  }

    fini_message_data(cd);

    return SMFIS_CONTINUE;
    }


/* spf_abort - handle the message being aborted
**
** May be called at any time during processing of a message, indicating
** that something went wrong.  Any per-message data should be freed both
** here and in sample_eom(), and wherever you return any of
** SMFIS_REJECT/SMFIS_TEMPFAIL/SMFIS_DISCARD.
*/
sfsistat
spf_abort(SMFICTX* ctx)
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv(ctx);

    if (cd != (connection_data_t*) 0)
  fini_message_data(cd);
    else
  syslog(LOG_NOTICE, "spf_abort called but cd is null");

    return SMFIS_CONTINUE;
    }


/* spf_close - handle the connection being closed
**
** Called once at the end of a connection.  Any per-connection data
** should be freed here.
*/
sfsistat
spf_close(SMFICTX* ctx)
    {
    connection_data_t* cd;

    cd = (connection_data_t*) smfi_getpriv(ctx);

    if (cd != (connection_data_t*) 0)
  {
  smfi_setpriv(ctx, (void*) 0);
  fini_connection_data(cd);
  }
    else
  syslog(LOG_NOTICE, "spf_close called but cd is null");

    return SMFIS_CONTINUE;
    }


sfsistat
handle_result(SMFICTX* ctx)
    {
    connection_data_t* cd;
    const char* exp;
    char exp_escaped[1000];

    cd = (connection_data_t*) smfi_getpriv(ctx);

    switch (cd->result)
  {
  case SPFMILTER_RESULT_PASS:
  case SPFMILTER_RESULT_SOFTFAIL:
  case SPFMILTER_RESULT_NEUTRAL:
  case SPFMILTER_RESULT_UNKNOWN:
  case SPFMILTER_RESULT_UNMECH:
  case SPFMILTER_RESULT_ERROR:  /* typically "DNS lookup failure" */
  case SPFMILTER_RESULT_NONE:
      cd->action = SPFMILTER_ACTION_MARK;
      break;
  case SPFMILTER_RESULT_FAIL:
      if (markonly)
    cd->action = SPFMILTER_ACTION_MARK;
      else
    cd->action = SPFMILTER_ACTION_REJECT;
      break;
  }

    switch (cd->action)
  {
  case SPFMILTER_ACTION_REJECT:
      exp = lib_get_explanation(cd->lib_data);
      syslog(LOG_INFO, "rejecting mail from [%s] - %s", cd->ip_str,
lib_get_reason(cd->lib_data));
      if (exp != (char*) 0)
    escape_percents(exp, exp_escaped, sizeof(exp_escaped));
      else
    (void) strncpy(exp_escaped, "rejected by spfmilter", sizeof(exp_escaped) -
1);
      smfi_setreply(ctx, "550", "5.7.1", exp_escaped);
      fini_message_data(cd);
      return SMFIS_REJECT;

  case SPFMILTER_ACTION_TEMPFAIL:
      syslog(LOG_INFO, "temporarily failing mail from [%s] - %s", cd->ip_str,
lib_get_error(cd->lib_data));
      smfi_setreply(ctx, "451", "4.3.2", "Please try again later.");
      fini_message_data(cd);
      return SMFIS_TEMPFAIL;

  default:
      return SMFIS_CONTINUE;
  }
    }


/* escape_percents - escape '%' characters in a string
**
** Sendmail doesn't like sending percent characters in bounce messages.
** If a milter returns an error message with percents in it, sendmail
** will ignore that and instead return an uninformative default
** message saying "Command rejected".  To avoid this we escape the
** percents witrh a second percent.
*/
void
escape_percents(const char* in, char* out, int out_size)
    {
    const char* i;
    char* o;
    int out_len;

    i = in;
    o = out;
    out_len = 0;
    while (*i != '\0' && out_len + 2 < out_size)
  {
  if (*i == '%')
      {
      /* The character that most often comes back escaped is
      ** the @ sign (%40). Since it doesn't actually need
      ** escaping, unescape it.
      */
      if (i[1] == '4' && i[2] == '0')
    {
    *o++ = '@';
    ++out_len;
    i += 3;
    }
      else
    {
    /* Otherwise, escape the %-sign. */
    *o++ = '%';
    *o++ = '%';
    out_len += 2;
    ++i;
    }
      }
  else
      {
      *o++ = *i++;
      ++out_len;
      }
  }
    *o = '\0';
    }


int
receiver_is_me(const char* value)
    {
    const char* cp;

    /* First look for a receiver= setting after the comment part.
    ** The hostname immediately follows the equals and is immediately
    ** followed by a semicolon.
    */
    cp = strrchr(value, ')');
    if (cp != (char*) 0)
  ++cp;
    else
  cp = value;
    cp = strstr(cp, " receiver=");
    if (cp != (char*) 0)
  {
  cp += 10;
  if (strncasecmp(cp, local_hostname, local_hostname_len) == 0)
      {
      cp += local_hostname_len;
      if (*cp == ';')
    return 1; /* Yes! */
      }
  }

    /* If there was no receiver= (it's optional), try checking the
    ** comment part for a hostname.  That's even more optional
    ** (it's not in the spec at all), but it's actually more likely
    ** to be present since most implementations do add it.
    **
    ** To find the comment hostname, skip one word, whitespace, and
    ** an open-paren, then check for the hostname followed by a colon.
    */
    cp = value;
    cp += strcspn(cp, " \t");
    cp += strspn(cp, " \t");
    if (*cp == '(')
  {
  ++cp;
  if (strncasecmp(cp, local_hostname, local_hostname_len) == 0)
      {
      cp += local_hostname_len;
      if (*cp == ':')
    return 1; /* Yes! */
      }
  }

    /* Nope. */
    return 0;
    }


void
build_header(connection_data_t* cd, char* header, int header_size, const char*
reason)
    {
    int len;

    (void) snprintf(header, header_size, "%s", result_str(cd->result));
    len = strlen(header);
    if (reason != (char*) 0)
  {
  (void) snprintf(&header[len], header_size - len, " (%s)", reason);
  len = strlen(header);
  }
    if (local_hostname != (char*) 0)
  {
  (void) snprintf(&header[len], header_size - len, " receiver=%s;",
local_hostname);
  len = strlen(header);
  }
    if (cd->ip_str!= (char*) 0)
  {
  (void) snprintf(&header[len], header_size - len, " client-ip=%s;",
cd->ip_str);
  len = strlen(header);
  }
    if (cd->helo != (char*) 0)
  {
  (void) snprintf(&header[len], header_size - len, " helo=%s;", cd->helo);
  len = strlen(header);
  }
    if (cd->from != (char*) 0)
  {
  (void) snprintf(&header[len], header_size - len, " envelope-from=%s;",
cd->from);
  len = strlen(header);
  }
    /*!!! Do something about the problem= field. */
    (void) snprintf(&header[len], header_size - len, " x-software=%s %s %s;",
SPFMILTER_PROGRAM, SPFMILTER_VERSION, SPFMILTER_URL);
    }


connection_data_t*
init_connection_data(void)
    {
    connection_data_t* cd;

    cd = (connection_data_t*) malloc(sizeof(connection_data_t));
    if (cd == (connection_data_t*) 0)
  return (connection_data_t*) 0;
    cd->whitelisted = 0;
    cd->ip_str = (char*) 0;
    cd->helo = (char*) 0;
    cd->lib_data = lib_init_connection_data();
    if (cd->lib_data == (lib_data_t*) 0)
  {
  free((void*) cd);
  return (connection_data_t*) 0;
  }
    if (! init_message_data(cd))
  {
  fini_connection_data(cd);
  return (connection_data_t*) 0;
  }
    return cd;
    }


int
init_message_data(connection_data_t* cd)
    {
    cd->authenticated = 0;
    cd->from = (char*) 0;
    cd->result = -1;
    cd->action = SPFMILTER_ACTION_UNKNOWN;
    cd->n_spf_headers = 0;
    cd->n_del_headers = 0;
    if (! lib_init_message_data(cd->lib_data))
  return 0;
    return 1;
    }


/* Frees or resets any items in the connection_data that are for
** an individual message.
*/
void
fini_message_data(connection_data_t* cd)
    {
    if (cd->from != (char*) 0)
  {
  free((void*) cd->from);
  cd->from = (char*) 0;
  }
    lib_fini_message_data(cd->lib_data);
    }


/* Frees all items in the connection_data. */
void
fini_connection_data(connection_data_t* cd)
    {
    fini_message_data(cd);
    if (cd->ip_str != (char*) 0)
  free((void*) cd->ip_str);
    if (cd->helo != (char*) 0)
  free((void*) cd->helo);
    lib_fini_connection_data(cd->lib_data);
    free((void*) cd);
    }


char*
result_str(int result)
    {
    switch (result)
  {
  case SPFMILTER_RESULT_PASS: return "pass";
  case SPFMILTER_RESULT_FAIL: return "fail";
  case SPFMILTER_RESULT_SOFTFAIL: return "softfail";
  case SPFMILTER_RESULT_NEUTRAL: return "neutral";
  case SPFMILTER_RESULT_UNKNOWN: return "unknown";
  case SPFMILTER_RESULT_UNMECH: return "unknown-mechanisms";
  case SPFMILTER_RESULT_ERROR: return "error";
  case SPFMILTER_RESULT_NONE: return "none";
  default: return "???";
  }
    }


/* lib_fallback_s
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
struct lib_fallback_s
{
  /* not used, but mallocing zero bytes can be a problem */
  int junk;
};


/* lib_data_s
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
struct lib_data_s
{
  peer_info_t *peer_info;  /* libSPF peer info data structure */
  SPF_RESULT  res;         /* eventual result of SPF query */
};


/* lib_init
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_init(void)
{
  if (recipientmx)
  {
    (void) fprintf(stderr,
      "Sorry, --recipientmx mode is not implemented with libspf");
    return(0);
  }

  /* SPF_dbg_level(debug); */
  return(1);
}


/* lib_init_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
lib_fallback_t *lib_init_fallback(const char* str)
{
  lib_fallback_t* lf;

  lf = (lib_fallback_t*) malloc(sizeof(lib_fallback_t));

  if (lf == (lib_fallback_t*)0)
  {
    return((lib_fallback_t*)0);
  }

  return(lf);
}


/* lib_set_local_hostname
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_set_local_hostname(lib_data_t *ld)
{
  return(1);
}


/* lib_init_connection_data
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
lib_data_t *lib_init_connection_data(void)
{
  lib_data_t* ld;

  ld = (lib_data_t*) malloc(sizeof(lib_data_t));
  if (ld == (lib_data_t*)0)
  {
    return((lib_data_t*)0);
  }
  return(ld);
}


/* lib_init_message_data
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_init_message_data(lib_data_t *ld)
{
  return(1);
}


/* lib_set_ipv4
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_set_ipv4(lib_data_t *ld, struct in_addr ipv4_addr,
 char *ipv4_str)
{
  ld->peer_info = SPF_init(local_hostname, ipv4_str, explanation_str,
    (char *)0, (char *)0, trustedforwarders, 0);

  if (ld->peer_info == (peer_info_t*)0)
  {
    return(0);
  }

  return(1);
}


/* lib_set_ipv6
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_set_ipv6(lib_data_t *ld, struct in6_addr ipv6_addr,
  char *ipv6_str)
{
  ld->peer_info = SPF_init(local_hostname, ipv6_str, explanation_str,
    (char*)0, (char*)0, trustedforwarders, 0);

  if (ld->peer_info == (peer_info_t*)0)
  {
    return(0);
  }

  return(1);
}


/* lib_set_helo_hostname
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_set_helo_hostname(lib_data_t *ld, char *helo_hostname)
{
  if (!SPF_smtp_helo(ld->peer_info, helo_hostname))
  {
    return(0);
  }

  return(1);
}


/* lib_set_from
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_set_from(lib_data_t *ld, const char *from)
{
  if (!SPF_smtp_from(ld->peer_info, from))
  {
    return(0);
  }

  return(1);
}


/* lib_do_check
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_do_check(lib_data_t *ld, const char *from)
{
  ld->res = SPF_policy_main(ld->peer_info);

  /* If there was no SPF record, then try finding a fallback mechanism. */
  if (ld->res == SPF_NONE)
  {
    int i = find_fallback(from);

    if (i != -1)
    {
      if (!SPF_parse_policy(ld->peer_info, fallbacks[i].str))
      {
        return(0);
      }
    }
  }

  return(1);
}


/* lib_do_check_recipien
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_do_check_recipient(lib_data_t *ld, const char *to)
{
  /* Not implemented */
  return(0);
}


/* lib_do_check_final
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_do_check_final(lib_data_t *ld)
{
  /* Not implemented */
  return(0);
}


/* lib_get_result
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
int lib_get_result(lib_data_t *ld)
{
  /* Convert libspf result to spfmilter result. */
  switch (ld->res)
  {
    case SPF_PASS:
      return(SPFMILTER_RESULT_PASS);
    case SPF_NONE:
      return(SPFMILTER_RESULT_NONE);
    case SPF_S_FAIL:
      return(SPFMILTER_RESULT_SOFTFAIL);
    case SPF_H_FAIL:
      return(SPFMILTER_RESULT_FAIL);
    case SPF_ERROR:
      return(SPFMILTER_RESULT_ERROR);
    case SPF_NEUTRAL:
      return(SPFMILTER_RESULT_NEUTRAL);
    case SPF_UNKNOWN:
      return(SPFMILTER_RESULT_UNKNOWN);
    case SPF_UNMECH:
      return(SPFMILTER_RESULT_UNMECH);
    default:
      return(-1);
  }

  return(-1);
}


/* lib_get_reason
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
const char *lib_get_reason(lib_data_t* ld)
{
  return(ld->peer_info->error);
}


/* lib_get_explanation
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
const char *lib_get_explanation(lib_data_t* ld)
{
  return(ld->peer_info->explain);
}


/* lib_get_error
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
const char *lib_get_error(lib_data_t *ld)
{
  return(ld->peer_info->error);
}


/* lib_fini_message_data
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void lib_fini_message_data(lib_data_t *ld)
{
  return;
}


/* lib_fini_connection_data
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void lib_fini_connection_data(lib_data_t *ld)
{
  if (ld != (lib_data_t*)0)
  {
    if (ld->peer_info != (peer_info_t*)0)
    {
      ld->peer_info = SPF_close(ld->peer_info);
    }
    free((void*) ld);
  }

  return;
}


/* lib_fini_fallback
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void lib_fini_fallback(lib_fallback_t *lf)
{
  if (lf != (lib_fallback_t*)0)
  {
    free((void*)lf);
  }

  return;
}


/* lib_fini
*
*  Author: Jef Poskanzer <jef@acme.com>
*  Author: James Couzens <jcouzens@codeshare.ca>
*
*  Date:   07/30/04
*  Date:   08/03/04 - function cleanup
*
*  Desc:
*
*/
void lib_fini(void)
{
  return;
}

/* end of spfmilter.c */
