/* pawnDropCheckmate.t.cc
 */
#include "osl/move_classifier/pawnDropCheckmate.h"
#include "osl/move_classifier/moveAdaptor.h"
#include "osl/move_classifier/check_.h"
#include "osl/checkmate/dfpn.h"
#include "osl/record/csaRecord.h"
#include "osl/record/csaString.h"
#include "osl/state/numEffectState.h"
#include "osl/container/moveVector.h"
#include "osl/effect_util/effectUtil.h"
#include "osl/apply_move/applyMove.h"
#include "osl/oslConfig.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <string>
#include <iostream>
#include <fstream>

class PawnDropCheckmateTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PawnDropCheckmateTest);
  CPPUNIT_TEST(testOne);
  CPPUNIT_TEST(testBlock);
  CPPUNIT_TEST(testFile);
  CPPUNIT_TEST(testKingExposure);
  CPPUNIT_TEST_SUITE_END();
public:
  void testOne();
  void testBlock();
  void testKingExposure();
  void testFile(const std::string& filename);
  void testFile();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PawnDropCheckmateTest);

using namespace osl;
using namespace osl::move_classifier;

static
bool isPawnCheckmate(const NumEffectState& sstate, Move m)
{
  using namespace osl::checkmate;
  // m は王手であること
  NumEffectState state(sstate);
  ApplyMoveOfTurn::doMove(state, m);

  DfpnTable table(m.player());
  const PathEncoding path(alt(m.player()));
  typedef Dfpn searcher_t;
  searcher_t searcher;
  searcher.setTable(&table);
  ProofDisproof pdp=searcher.hasEscapeMove(state,HashKey(state),path,1500,m);
  return (pdp == ProofDisproof::PawnCheckmate());
}

static
bool isPawnCheckmateTest(const NumEffectState& sstate, Move m)
{
  const Ptype ptype = m.ptype();
  const Position from = m.from();
  const Position to = m.to();
  if (m.player() == BLACK)
    return PawnDropCheckmate<BLACK>::isMember(sstate, ptype, from, to);
  else
    return PawnDropCheckmate<WHITE>::isMember(sstate, ptype, from, to);
}

void PawnDropCheckmateTest::testBlock()
{
  const char *problem = 
    "P1-KY *  * -OU * +KA *  * -KY\n"
    "P2 * +GI *  * +HI * -KI *  * \n"
    "P3 *  * -KE * -FU *  *  *  * \n"
    "P4-FU *  * -KE-GI * -FU * -FU\n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6+FU+FU+KI-FU *  * +FU * +FU\n"
    "P7 *  * +KE *  *  *  *  *  * \n"
    "P8 * +OU+GI *  *  *  *  *  * \n"
    "P9+KY *  *  *  *  *  * +KE+KY\n"
    "P+00FU\n"
    "P-00AL\n"
    "+\n";
  NumEffectState sstate((CsaString(problem).getInitialState()));
  // 62 に歩を打っても飛車の利きが消えるため打歩詰ではない
  const Position target = Position(6,2);
  CPPUNIT_ASSERT((! PawnDropCheckmate<BLACK>::
		  isMember(sstate, PAWN, Position::STAND(), target)));
  const Move m = Move(target, PAWN, BLACK);
  CPPUNIT_ASSERT(! isPawnCheckmate(sstate, m));
}

void PawnDropCheckmateTest::testKingExposure()
{
  const char *problem = 
    "P1-OU-KI * +HI *  *  *  *  * \n"
    "P2 *  *  *  *  *  *  *  *  * \n"
    "P3 * +TO *  *  *  *  *  *  * \n"
    "P4 *  *  *  *  *  *  *  *  * \n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  *  *  *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  *  *  *  *  *  *  *  * \n"
    "P9 *  *  *  *  *  *  *  * +OU\n"
    "P+00FU\n"
    "P-00AL\n"
    "+\n";
  NumEffectState sstate((CsaString(problem).getInitialState()));
  // 92歩は金でとれるが打歩詰
  const Position target = Position(9,2);
  const Move m = Move(target, PAWN, BLACK);
  CPPUNIT_ASSERT(isPawnCheckmate(sstate, m));
  CPPUNIT_ASSERT((PawnDropCheckmate<BLACK>::
		  isMember(sstate, PAWN, Position::STAND(), target)));
}

void PawnDropCheckmateTest::testOne()
{
  const char *problem = 
    "P1 *  *  *  *  *  *  * -KI-KY\n"
    "P2 *  *  *  *  *  *  * -KE-OU\n"
    "P3 *  *  *  *  *  *  *  *  * \n"
    "P4 *  *  *  *  *  *  *  * +GI\n"
    "P5 *  *  *  *  *  *  *  *  * \n"
    "P6 *  *  * -KI *  *  *  *  * \n"
    "P7 *  *  *  *  *  *  *  *  * \n"
    "P8 *  * +KY+OU+KE  *  *  *  * \n"
    "P9-RY *  *  *  *  *  *  *  * \n"
    "P+00FU\n"
    "P-00AL\n"
    "+\n";
  NumEffectState sstate((CsaString(problem).getInitialState()));
  CPPUNIT_ASSERT((PawnDropCheckmate<BLACK>::
		  isMember(sstate, PAWN, Position::STAND(), Position(1,3))));
  const Move m13fu = Move(Position(1,3), PAWN, BLACK);
  CPPUNIT_ASSERT(isPawnCheckmate(sstate, m13fu));
  CPPUNIT_ASSERT((! PawnDropCheckmate<BLACK>::
		  isMember(sstate, LANCE, Position::STAND(), Position(1,3))));
  CPPUNIT_ASSERT((! PawnDropCheckmate<WHITE>::
		  isMember(sstate, PAWN, Position::STAND(), Position(1,3))));

  CPPUNIT_ASSERT((PawnDropCheckmate<WHITE>::
		  isMember(sstate, PAWN, Position::STAND(), Position(6,7))));
  CPPUNIT_ASSERT((! PawnDropCheckmate<WHITE>::
		  isMember(sstate, LANCE, Position::STAND(), Position(6,7))));
  CPPUNIT_ASSERT((! PawnDropCheckmate<BLACK>::
		  isMember(sstate, PAWN, Position::STAND(), Position(6,7))));
}

void PawnDropCheckmateTest::testFile(const std::string& filename)
{
  Record rec=CsaFile(filename).getRecord();
  NumEffectState sstate((SimpleState(rec.getInitialState())));
  vector<osl::Move> moves = rec.getMoves();
  for (size_t i=0; i<moves.size(); i++)
  {
    ApplyMoveOfTurn::doMove(sstate, moves[i]);
    const Player turn = sstate.getTurn();
    if (sstate.inCheck())
      break;

    const Position target = sstate.getKingPosition(alt(turn));
    const Position drop_to 
      = Board_Table.nextPosition(turn, target, D);
    if (! drop_to.isOnBoard())
      continue;
    const Move m = Move(drop_to, PAWN, turn);
    assert(m.isValid());
    CPPUNIT_ASSERT(PlayerMoveAdaptor<Check>::isMember(sstate, m));
    if (sstate.isAlmostValidMove<false>(m))
    {
      CPPUNIT_ASSERT_EQUAL(isPawnCheckmate(sstate, m),
			   isPawnCheckmateTest(sstate, m));
    }
  }
}

void PawnDropCheckmateTest::testFile()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  int i=0;
  int count=1000;
  if (isShortTest) 
    count=50;
  std::string fileName;
  while((ifs >> fileName) && (++i<count))
  {
    if(fileName == "") 
      break;
    if (! isShortTest)
      std::cerr << fileName << " ";
    testFile(OslConfig::testCsaFile(fileName));
    std::cerr << '.';
  }
}

/* ------------------------------------------------------------------------- */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
