#include "osl/ptype.h"
#include "osl/ptypeTraits.h"
#include "osl/ptypeTable.h"
#include <iostream>
#include <iomanip>
#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>

class PtypeTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE( PtypeTest );
  CPPUNIT_TEST( testCanDropTo );
  CPPUNIT_TEST( testMayPromote );
  CPPUNIT_TEST( testIsPromoted );
  CPPUNIT_TEST( testCanPromote );
  CPPUNIT_TEST( testPromote );
  CPPUNIT_TEST( testUnpromote );
  CPPUNIT_TEST( testAlt );
  CPPUNIT_TEST( testAltIfPiece );
  CPPUNIT_TEST( testPromoteUnpromote );
  CPPUNIT_TEST( testShow );
  CPPUNIT_TEST( testTable );
  CPPUNIT_TEST( testInput );
  CPPUNIT_TEST( testIsMajor );
  CPPUNIT_TEST_SUITE_END();
 public:
  void testCanDropTo();
  void testMayPromote();
  void testIsPromoted();
  void testCanPromote();
  void testPromote();
  void testUnpromote();
  void testAlt();
  void testAltIfPiece();
  void testPromoteUnpromote();
  void testShow();
  void testTable();
  void testInput();
  void testIsMajor();
};

using namespace osl;
extern bool isShortTest;

CPPUNIT_TEST_SUITE_REGISTRATION(PtypeTest);

namespace osl
{
  inline void for_each_piece_ptype(void (*func)(Ptype))
  {
    for (int i=static_cast<int>(PTYPE_PIECE_MIN);i<=static_cast<int>(PTYPE_MAX);i++)
    {
      func(static_cast<Ptype>(i));
    }
  }
  
  inline void for_each_ptype(void (*func)(Ptype))
  {
    for (int i=static_cast<int>(PTYPE_MIN);i<=static_cast<int>(PTYPE_MAX);i++)
    {
      func(static_cast<Ptype>(i));
    }
  }
}

void PtypeTest::testInput(){
  std::istringstream is("GOLD ROOK PPAWN PAWN PGOLD PBISHOP");
  Ptype g;
  is >> g;
  CPPUNIT_ASSERT_EQUAL( GOLD, g );
  is >> g;
  CPPUNIT_ASSERT_EQUAL( ROOK, g );
  is >> g;
  CPPUNIT_ASSERT_EQUAL( PPAWN, g );
  is >> g;
  CPPUNIT_ASSERT_EQUAL( PAWN, g );
  is >> g;
  CPPUNIT_ASSERT_EQUAL( PTYPE_EMPTY, g );
  is >> g;
  CPPUNIT_ASSERT_EQUAL( PBISHOP, g );

}

void PtypeTest::testCanDropTo(){
  CPPUNIT_ASSERT_EQUAL( false, (PtypePlayerTraits<PAWN,BLACK>::canDropTo(Position(4,1))));
  CPPUNIT_ASSERT_EQUAL( false, Ptype_Table.canDropTo(BLACK,PAWN,Position(4,1)));
}
void PtypeTest::testMayPromote(){
  CPPUNIT_ASSERT((PtypePlayerTraits<PAWN,BLACK>::mayPromote(Position(9,2))));
  CPPUNIT_ASSERT((PtypePlayerTraits<PAWN,BLACK>::mayPromote(Position(8,4))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<PAWN,BLACK>::mayPromote(Position(1,5))));

  CPPUNIT_ASSERT((PtypePlayerTraits<PAWN,WHITE>::mayPromote(Position(9,8))));
  CPPUNIT_ASSERT((PtypePlayerTraits<PAWN,WHITE>::mayPromote(Position(8,6))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<PAWN,WHITE>::mayPromote(Position(1,5))));

  CPPUNIT_ASSERT((PtypePlayerTraits<SILVER,BLACK>::mayPromote(Position(9,2))));
  CPPUNIT_ASSERT((PtypePlayerTraits<SILVER,BLACK>::mayPromote(Position(8,4))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<SILVER,BLACK>::mayPromote(Position(1,5))));

  CPPUNIT_ASSERT((PtypePlayerTraits<SILVER,WHITE>::mayPromote(Position(9,8))));
  CPPUNIT_ASSERT((PtypePlayerTraits<SILVER,WHITE>::mayPromote(Position(8,6))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<SILVER,WHITE>::mayPromote(Position(1,5))));

  CPPUNIT_ASSERT((PtypePlayerTraits<KNIGHT,BLACK>::mayPromote(Position(9,3))));
  CPPUNIT_ASSERT((PtypePlayerTraits<KNIGHT,BLACK>::mayPromote(Position(8,5))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<KNIGHT,BLACK>::mayPromote(Position(1,6))));

  CPPUNIT_ASSERT((PtypePlayerTraits<KNIGHT,WHITE>::mayPromote(Position(9,7))));
  CPPUNIT_ASSERT((PtypePlayerTraits<KNIGHT,WHITE>::mayPromote(Position(8,6))));
  CPPUNIT_ASSERT((!PtypePlayerTraits<KNIGHT,WHITE>::mayPromote(Position(1,2))));

  CPPUNIT_ASSERT((PtypePlayerTraits<LANCE,BLACK>::mayPromote(Position(1,9))));
  CPPUNIT_ASSERT((PtypePlayerTraits<LANCE,WHITE>::mayPromote(Position(8,2))));

  CPPUNIT_ASSERT((PtypePlayerTraits<ROOK,BLACK>::mayPromote(Position(1,9))));
  CPPUNIT_ASSERT((PtypePlayerTraits<ROOK,WHITE>::mayPromote(Position(8,2))));

  CPPUNIT_ASSERT((PtypePlayerTraits<BISHOP,BLACK>::mayPromote(Position(1,9))));
  CPPUNIT_ASSERT((PtypePlayerTraits<BISHOP,WHITE>::mayPromote(Position(8,2))));
  // 先手BISHOPが49,58,59,69などにいる場合はこれではチェックはできない．
  CPPUNIT_ASSERT((PtypePlayerTraits<BISHOP,BLACK>::mayPromote(Position(5,9))));

}
void PtypeTest::testIsPromoted(){
  CPPUNIT_ASSERT( isPromoted(PPAWN) == true);
  CPPUNIT_ASSERT( isPromoted(PROOK) == true);
  CPPUNIT_ASSERT( isPromoted(KING) == false);
  CPPUNIT_ASSERT( isPromoted(ROOK) == false);
}
static void checkCanPromote(Ptype ptype){
  if(isPromoted(ptype) || ptype==GOLD || ptype==KING){
    CPPUNIT_ASSERT( canPromote(ptype) == false);
  }
  else {
    CPPUNIT_ASSERT( canPromote(ptype) == true);
  }
}
void PtypeTest::testCanPromote(){
  for_each_piece_ptype(checkCanPromote);
}
void PtypeTest::testPromote(){
  CPPUNIT_ASSERT( promote(PAWN) == PPAWN);
  CPPUNIT_ASSERT( promote(ROOK) == PROOK);
}
void PtypeTest::testUnpromote(){
  CPPUNIT_ASSERT( unpromote(PPAWN) == PAWN);
  CPPUNIT_ASSERT( unpromote(PROOK) == ROOK);
  CPPUNIT_ASSERT( unpromote(KING) == KING);
  CPPUNIT_ASSERT( unpromote(ROOK) == ROOK);
}
void PtypeTest::testAlt(){
  for(int i=0;i<2;i++){
    Player pl=static_cast<Player>(-i);
    for(int k=PPAWN;k<=PTYPE_MAX;k++){
      Ptype ptype=static_cast<Ptype>(k);
      PtypeO ptypeO=newPtypeO(pl,ptype);
      PtypeO altPtypeO=newPtypeO(alt(pl),ptype);
      CPPUNIT_ASSERT_EQUAL(altPtypeO,alt(ptypeO));
    }
  }
}
void PtypeTest::testAltIfPiece(){
  for(int i=0;i<2;i++){
    Player pl=static_cast<Player>(-i);
    for(int k=PPAWN;k<=PTYPE_MAX;k++){
      Ptype ptype=static_cast<Ptype>(k);
      PtypeO ptypeO=newPtypeO(pl,ptype);
      PtypeO altPtypeO=newPtypeO(alt(pl),ptype);
      CPPUNIT_ASSERT_EQUAL(altPtypeO,altIfPiece(ptypeO));
    }
  }
  CPPUNIT_ASSERT_EQUAL(PTYPEO_EMPTY,altIfPiece(PTYPEO_EMPTY));
  CPPUNIT_ASSERT_EQUAL(PTYPEO_EDGE,altIfPiece(PTYPEO_EDGE));
}
// promote可能なら
static void checkPromoteUnpromote(Ptype ptype){
  if(canPromote(ptype)){
    CPPUNIT_ASSERT( unpromote(promote(ptype)) == ptype);
  }
}
void PtypeTest::testPromoteUnpromote(){
  for_each_piece_ptype(checkPromoteUnpromote);
}
static void checkTable(Ptype ptype){
  std::cout << ptype << ",Index=" << Ptype_Table.getIndex(ptype) 
	    << ",maskLow=" << Ptype_Table.getMaskLow(ptype).value() 
	    << ",moveMask=" << std::setbase(16) 
	    << Ptype_Table.getMoveMask(ptype) 
	    << std::setbase(10) << std::endl;
}

void PtypeTest::testShow(){
  if (! isShortTest)
  {
    for_each_piece_ptype(checkTable);
    for(int i=0;i<2;i++){
      Player p=static_cast<Player>(-i);
      for(int j=0;j<SHORT_DIRECTION_SIZE;j++){
	Direction dir=static_cast<Direction>(j);
	for(int k=PPAWN;k<=PTYPE_MAX;k++){
	  Ptype ptype=static_cast<Ptype>(k);
	  std::cout << p << "," << dir << "," << ptype << "," << std::setbase(16) 
		    << Ptype_Table.getShortMoveMask(p,static_cast<PtypeO>(k),dir)
		    << ","
		    << Ptype_Table.getShortMoveMask(p,static_cast<PtypeO>(k-16),dir)
		    << std::setbase(10) << std::endl;
	}
      }
    }
  }
}

void PtypeTest::testTable(){
  CPPUNIT_ASSERT( Ptype_Table.hasLongMove(LANCE) );
  CPPUNIT_ASSERT( Ptype_Table.hasLongMove(ROOK) );
#if OSL_WORDSIZE == 64
  CPPUNIT_ASSERT_EQUAL((unsigned long long)PtypeFuns<PAWN>::indexMask,0x3ffffuLL);
  CPPUNIT_ASSERT_EQUAL((unsigned long long)PtypeFuns<ROOK>::indexMask,0xc000000000uLL);
  CPPUNIT_ASSERT_EQUAL(Ptype_Table.getMaskLow(PAWN),mask_t::makeDirect(0x3ffffuLL));
  CPPUNIT_ASSERT_EQUAL(Ptype_Table.getMaskLow(ROOK),mask_t::makeDirect(0xc000000000uLL));
#elif OSL_WORDSIZE == 32
  CPPUNIT_ASSERT_EQUAL((mask_int_t)PtypeFuns<PAWN>::indexMask,(mask_int_t)0x3ffffu);
  CPPUNIT_ASSERT_EQUAL((mask_int_t)PtypeFuns<ROOK>::indexMask,(mask_int_t)0xc0u);
  CPPUNIT_ASSERT_EQUAL(Ptype_Table.getMaskLow(PAWN),mask_t::makeDirect(0x3ffffu));
  CPPUNIT_ASSERT_EQUAL(Ptype_Table.getMaskLow(ROOK),mask_t::makeDirect(0xc0u));
#endif
 {
   int dx= -1, dy=8;
   Ptype ptype=ROOK;
   const EffectContent effect=Ptype_Table.getEffect(newPtypeO(BLACK,ptype),
						    Offset32(dx,dy));
   CPPUNIT_ASSERT(! effect.hasBlockableEffect());
   CPPUNIT_ASSERT(! effect.hasUnblockableEffect());
 }
}

void PtypeTest::testIsMajor()
{
  for (int i=PTYPE_MIN; i<PTYPE_MAX; ++i)
  {
    const Ptype ptype = static_cast<Ptype>(i);
    if (isPiece(ptype))
    {
      CPPUNIT_ASSERT_EQUAL((unpromote(ptype) == ROOK 
			    || unpromote(ptype) == BISHOP),
			   isMajor(ptype));
    }
    CPPUNIT_ASSERT_EQUAL((ptype == ROOK || ptype == BISHOP),
			 isMajorBasic(ptype));
  }  
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
