// piecePairPieceEval.t.cc
#include "osl/eval/ppair/piecePairPieceEval.h"
#include "osl/eval/ppair/piecePairRawEval.h"
#include "osl/eval/pieceEval.h"
#include "osl/container/pieceValues.h"
#include "../consistencyTest.h"

#include <cppunit/TestCase.h>
#include <cppunit/extensions/HelperMacros.h>
#include <iostream>

using namespace osl;

class PiecePairPieceEvalTest : public CppUnit::TestFixture 
{
  CPPUNIT_TEST_SUITE(PiecePairPieceEvalTest);
  CPPUNIT_TEST(testLoad);
  // testEval 以降は testLoad の後にテストすること
  CPPUNIT_TEST(testConsistentUpdate);
  CPPUNIT_TEST(testConsistentExpect);
  CPPUNIT_TEST(testPieces);
  CPPUNIT_TEST(testValues);
  CPPUNIT_TEST(testSum);
  CPPUNIT_TEST_SUITE_END();
public:
  void testLoad();
  void testEval();
  void testConsistentUpdate();
  void testConsistentExpect();
  void testPieces();
  void testValues();
  void testSum();
};

CPPUNIT_TEST_SUITE_REGISTRATION(PiecePairPieceEvalTest);

using namespace osl;
using namespace osl::eval;
using namespace osl::eval::ppair;
extern bool isShortTest;

void PiecePairPieceEvalTest::testLoad()
{
  CPPUNIT_ASSERT(PiecePairPieceEval::setUp());
}


void PiecePairPieceEvalTest::testConsistentUpdate()
{
  consistencyTestUpdate<PiecePairPieceEval>();
}

void PiecePairPieceEvalTest::testConsistentExpect()
{
  consistencyTestExpect<PiecePairPieceEval>();
}

void PiecePairPieceEvalTest::testPieces()
{
  for (int y=1; y<=9; ++y)
  {
    for (int x=1; x<=9; ++x)
    {
      const Position position(x,y);
      for (int p=PTYPEO_MIN; p<PTYPEO_MAX; ++p)
      {
	const PtypeO ptypeo = static_cast<PtypeO>(p);
	const unsigned int index = PiecePairIndex::indexOf(position, ptypeo);
	const int value = PiecePairPieceTable::Table.valueOf(index, index);
	const int ptype_value = Ptype_Eval_Table.value(ptypeo);
	CPPUNIT_ASSERT((abs(value - ptype_value) <= 150+PtypeEvalTraits<GOLD>::val*4/5));
      }
    }
  }

}

void PiecePairPieceEvalTest::testValues()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;
  for (int i=0;i<(isShortTest ? 100 : 10) && (ifs >> file_name) ; i++)
  {
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();

    NumEffectState state(rec.getInitialState());
    PieceValues values;

    PiecePairPieceEval sum_eval(state);
    PiecePairPieceEval::setValues(state, values);
    
    // 1/2 の切り捨てがあるので合計が一致するとは限らない
    CPPUNIT_ASSERT((sum_eval.rawValue() - values.sum()) < 40);
    
    for (unsigned int i=0; i<moves.size(); i++)
    {
      const Move m = moves[i];
      ApplyMoveOfTurn::doMove(state, m);
      sum_eval.update(state, m);
      PiecePairPieceEval::setValues(state, values);

      CPPUNIT_ASSERT((sum_eval.rawValue() - values.sum()) < 40);
    }
  }
}

void PiecePairPieceEvalTest::testSum()
{
  extern bool isShortTest;
  std::ifstream ifs(OslConfig::testCsaFile("FILES"));
  CPPUNIT_ASSERT(ifs);
  std::string file_name;

  const int tolerance = (40*39/2)*128/100+PtypeEvalTraits<GOLD>::val*4/5*4; // cumulative errors
  int large_errors = 0;
  for (int i=0;i<(isShortTest ? 20 : 400) && (ifs >> file_name) ; i++)
  {
    if (file_name == "") 
      break;
    file_name = OslConfig::testCsaFile(file_name);

    const Record rec=CsaFile(file_name).getRecord();
    const vector<osl::Move> moves=rec.getMoves();

    NumEffectState state(rec.getInitialState());
    PieceEval piece_eval(state);
    PiecePairRawEval raw_eval(state);
    PiecePairPieceEval sum_eval(state);

    CPPUNIT_ASSERT(abs(piece_eval.value() + raw_eval.value() 
		       - sum_eval.value()) < 5);
    for (unsigned int i=0; i<moves.size(); i++)
    {
      const Move m = moves[i];
      ApplyMoveOfTurn::doMove(state, m);
      piece_eval.update(state, m);
      raw_eval.update(state, m);
      sum_eval.update(state, m);

      const int expected = piece_eval.value() + (raw_eval.rawValue()*128/100)
	+ PiecePairPieceEval::standBonus(state);
      const int actual = sum_eval.rawValue();
      if (abs(expected - actual) >= tolerance)
      {
	++large_errors;
	std::cerr << expected - actual << "\n";
	if (! isShortTest)
	{
	  std::cerr << state << "\n";
	  std::cerr << piece_eval.value() 
		    << " + " << raw_eval.rawValue()*128/100
		    << " (" << raw_eval.rawValue() << ")"
		    << " != " << sum_eval.rawValue();
	  PieceValues values;
	  PiecePairPieceEval::setValues(state, values);
	  values.showValues(std::cerr, state);
	  PiecePairRawEval::setValues(state, values);
	  values.showValues(std::cerr, state);
	}
      }
      CPPUNIT_ASSERT(abs(expected - actual) < tolerance);
    }
  }
  CPPUNIT_ASSERT(large_errors < 10);
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
