/**
 * Validate moves in ki2 files.
 */

#include "osl/apply_move/applyMove.h"
#include "osl/record/ki2.h"
#include "osl/record/checkDuplicate.h"
#include "osl/record/ki2IOError.h"

#include <boost/algorithm/string/trim.hpp>
#include <boost/foreach.hpp>
#include <boost/program_options.hpp>
#include <boost/lambda/lambda.hpp>
#include <boost/lambda/bind.hpp>
#include <iostream>
#include <fstream>
#include <string>
#include <vector>
#include <algorithm>

// ----------------------------------------------
// Global variables
// ----------------------------------------------
boost::program_options::variables_map vm;
osl::record::CheckDuplicate check_duplicate;

using namespace boost::lambda;

void process( const std::string& file_name)
{
  try {
    bool verbose = false;
    if (vm.count("verbose"))
      verbose = true;
    const osl::Ki2File ki2(file_name, verbose);
    const osl::Record& record = ki2.getRecord();
    const osl::stl::vector<osl::Move>& moves = record.getMoves();

    if (check_duplicate.regist(moves)) {
      std::cerr << "Found a duplicated play: " << file_name << "\n";
      return;
    }

    osl::SimpleState state((osl::HIRATE));
    BOOST_FOREACH(const osl::Move& move, moves)
    {
      if (!state.isValidMove(move, false))
      {
        std::cout << file_name << "\n";
        continue;
      }
      osl::ApplyMoveOfTurn::doMove(state, move);
    }
  } catch (osl::Ki2IOError& err) {
    std::cerr << err.what() << "\n";
    std::cerr << "Found an error: " << file_name << "\n";
    return;
  }
}

int main(int argc, char **argv)
{
  boost::program_options::options_description command_line_options;
  command_line_options.add_options()
    ("input-file", boost::program_options::value< std::vector<std::string> >(),
     "input files in ki2 format (.ki2)")
    ("verbose,v", "Verbose mode")
    ("help,h", "Show this help message");
  boost::program_options::positional_options_description p;
  p.add("input-file", -1);

  try
  {
    boost::program_options::store(
      boost::program_options::command_line_parser(
	argc, argv).options(command_line_options).positional(p).run(), vm);
    boost::program_options::notify(vm);
    if (vm.count("help"))
    {
      std::cout << "Usage: " << argv[0] << " [options] ki2-file [ki2-file...]\n";
      std::cout << "       " << argv[0] << " [options]\n";
      std::cout << command_line_options << std::endl;
      return 0;
    }
  }
  catch (std::exception &e)
  {
    std::cerr << "error in parsing options" << std::endl
	      << e.what() << std::endl;
    std::cerr << "Usage: " << argv[0] << " [options] ki2-file [ki2-file...]\n";
    std::cerr << "       " << argv[0] << " [options]\n";
    std::cerr << command_line_options << std::endl;
    return 1;
  }

  std::vector<std::string> files;
  if (vm.count("input-file"))
  {
    const std::vector<std::string> temp = vm["input-file"].as<std::vector<std::string> >();
    files.insert(files.end(), temp.begin(), temp.end());
  }
  else
  {
    std::string line;
    while(std::getline(std::cin , line))
    {
      boost::algorithm::trim(line);
      files.push_back(line);
    } 
  }

  // main
  std::for_each(files.begin(), files.end(), bind(&process, _1));

  check_duplicate.print(std::cout);

  return 0;
}
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
