/* miniBoardChar50.cc
 */
#include "osl/record/miniBoardChar50.h"
#include "osl/record/compactBoard.h"
#include "osl/ptypeTable.h"
#include "osl/pieceTable.h"
#include <boost/tuple/tuple.hpp>
#include <boost/tuple/tuple_comparison.hpp>
#include <algorithm>
#include <stdexcept>

osl::record::
MiniBoardChar50::MiniBoardChar50()
{
  data.fill(0);
}

osl::record::
MiniBoardChar50::MiniBoardChar50(const SimpleState& org)
{
  data.fill(0);
  SimpleState board = (org.getTurn() == BLACK) ? org : org.rotate180();
  CArray<boost::tuple<int/* =Ptype*/,bool,int /* =Player*/,Position>, Piece::SIZE> pieces;
  for (int i=0; i<Piece::SIZE; ++i) 
  {
    const Piece p = board.getPieceOf(i);
    const int ptype_index = Ptype_Table.getIndexMin(unpromote(p.ptype()));
    pieces[i] = boost::make_tuple(ptype_index, p.isPromoted(), p.owner(), p.position());
  }
  std::sort(pieces.begin(), pieces.end());
  for (int i=0; i<Piece::SIZE; ++i) 
  {
    data[i] = OPiece::position2Bits(pieces[i].get<3>());
    data[Piece::SIZE + i/8] |= playerToIndex(static_cast<Player>(pieces[i].get<2>())) << (i%8);
    data[Piece::SIZE + i/8 + 5] |= pieces[i].get<1>() << (i%8);
  }
}

osl::record::
MiniBoardChar50::MiniBoardChar50(const std::string& src)
{
  if (src.size() != data.size())
    throw std::runtime_error("bad argument in MiniBoardChar50::MiniBoardChar50(const std::string&)");
  std::copy(src.begin(), src.end(), data.begin());
}

const osl::SimpleState osl::record::
MiniBoardChar50::toSimpleState(Player turn) const
{
  SimpleState state;
  state.init();

  for (int i = 0; i<Piece::SIZE; i++)
  {
    const Position position = OPiece::bits2Position(data[i]);
    const Player owner = indexToPlayer((data[40+i/8] >> (i%8)) & 1);
    const bool promoted = (data[40+i/8+5] >> (i%8)) & 1;
    Ptype ptype = Piece_Table.getPtypeOf(i);
    if (promoted)
      ptype = promote(ptype);
    state.setPiece(owner, position, ptype);
  }
  state.setTurn(BLACK);
  state.initPawnMask();
  if (turn != BLACK)
    state = state.rotate180();
  assert(state.getTurn() == turn);
  return state;
}

const std::string osl::record::
MiniBoardChar50::toString() const
{
  return std::string(data.begin(), data.end());
}

bool osl::record::operator<(const MiniBoardChar50& l, const MiniBoardChar50& r)
{
  return std::lexicographical_compare(l.data.begin(), l.data.end(), 
				      r.data.begin(), r.data.end());
}
bool osl::record::operator==(const MiniBoardChar50& l, const MiniBoardChar50& r)
{
  return std::equal(l.data.begin(), l.data.end(), r.data.begin());
}

// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
