/* numEffectState.h
 */
#ifndef OSL_NUM_EFFECT_STATE_H
#define OSL_NUM_EFFECT_STATE_H

#include "osl/effect/numSimpleEffect.h"
#include "osl/mobility/kingMobility.h"
#include "osl/misc/fastCopier.h"
#include "osl/misc/align16New.h"
#include <boost/cstdint.hpp>

namespace osl
{
  namespace checkmate
  {
    class King8Info;
  }
  namespace state
  {
    class NumEffectState;
    /**
     * 駒番に依存した局面（インスタンス）比較をする.
     * なお、駒番に非依存な局面比較をしたい場合は、osl::record::CompactBoardや
     * osl::hash::HashKeyを用いる.
     */
    bool operator==(const NumEffectState& st1, const NumEffectState& st2);

    /**
     * 利きを持つ局面
     * - effects (NumSimpleEffectTable) 利き
     * - onBoardMask (PieceMask) 盤上にある駒
     */
    class NumEffectState : public SimpleState
#if OSL_WORDSIZE == 32
			 , public misc::Align16New
#endif
    {
      effect::NumSimpleEffectTable effects;
      CArray<PieceMask,2> onBoardMask;
      /** 成駒一覧 */
      PieceMask promoted;
      CArray<PieceMask,2> pin_or_open;
      KingMobility king_mobility;
      CArray<uint64_t,2> king8infos;

      friend bool operator==(const NumEffectState& st1,const NumEffectState& st2);
      typedef NumEffectState state_t;
      friend class apply_move::ApplyDoUndoSimpleMove<BLACK,NumEffectState>;
      friend class apply_move::ApplyDoUndoCaptureMove<BLACK,NumEffectState>;
      friend class apply_move::ApplyDoUndoDropMove<BLACK,NumEffectState>;
      friend class apply_move::ApplyDoUndoSimpleMove<WHITE,NumEffectState>;
      friend class apply_move::ApplyDoUndoCaptureMove<WHITE,NumEffectState>;
      friend class apply_move::ApplyDoUndoDropMove<WHITE,NumEffectState>;
      friend class osl::misc::FastCopier;
      PieceMask& mutableOnBoardMask(Player p) {
	return onBoardMask[p];
      }
    public:
      const PieceMask& getOnBoardMask(Player p) const {
	return onBoardMask[p];
      }
      bool isOnBoardNum(int num) const
      {
	return getOnBoardMask(BLACK).test(num) ||
	  getOnBoardMask(WHITE).test(num);
      }

      /** T は isBasic でなくても動くが unpromote(T) の結果と同じ.
       */ 
      template<Player P,Ptype T,typename F>
      void forEachOnBoard(F& func) const {
	mask_t onMask=getOnBoardMask(P).template selectBit<T>() ;    
	while (onMask.any())
	{
	  int num=onMask.takeOneBit()+((PtypeFuns<T>::indexNum)<<5);
	  Piece p = getPieceOf(num);
	  func(p);
	}
      }
      /** T の成不成を区別
       */ 
      template<Player P,Ptype T,typename F>
      void forEachOnBoardPtypeStrict(F& func) const 
      {
	mask_t mask=getOnBoardMask(P).template selectBit<T>() ;    
	if (isPromoted(T)) 
	  mask &= promoted.getMask<T>();
	else
	  mask &= ~(promoted.getMask<T>());
	while (mask.any())
	{
	  int num=mask.takeOneBit()+((PtypeFuns<T>::indexNum)<<5);
	  func(getPieceOf(num));
	}
      }

      explicit NumEffectState(const state::SimpleState& st=SimpleState(HIRATE));
      ~NumEffectState();
      
      NumBitmapEffect getEffect(Position pos) const
      {
	return effects.getEffect(pos);
      }
      Position getMobility(Direction d,int num) const
      {
	return effects.mobilityTable.get(d,num);
      }
      void setMobility(Direction d,int num,Position pos)
      {
	effects.mobilityTable.set(d,num,pos);
      }
      Position kingMobilityAbs(Player p, Direction d) const
      {
	return Position::makeDirect(king_mobility[p][d]);
      }
      /** 
       * 玉がd方向にどこまで動けるかを返す
       * @param p 注目する玉のプレイヤ
       * @param d piece からみた向き
       */
      Position kingMobilityOfPlayer(Player p, Direction d) const
      {
	if (p == BLACK)
	  d = inverse(d);
	return kingMobilityAbs(p, d);
      }
      const EffectedNumTable& longEffectNumTable() const
      {
	return effects.effectedNumTable;
      }
      /** 
       * pieceのd方向から長い利きがある場合にその駒を返す。
       * @param d piece からみた向き
       */
      const Piece longEffectOfDirection(Player owner, int piece, Direction d) const
      {
	assert(getPieceOf(piece).isOnBoardByOwner(owner));
	if (owner == BLACK)
	  d = inverse(d);
	const int num = effects.effectedNumTable[piece][d];
	if (num == EMPTY_NUM)
	  return Piece::EMPTY();
	return getPieceOf(num);
      }
      const Piece longEffectOfDirection(Player owner, Piece piece, Direction d) const
      {
	assert(piece.isPiece());
	assert(piece.owner() == owner);
	return longEffectOfDirection(owner, piece.number(), d);
      }
      const Piece longEffectOfDirection(Piece piece, Direction d) const
      {
	assert(piece.isPiece());
	return longEffectOfDirection(piece.owner(), piece, d);
      }
      const Piece longEffectOfDirection(Position square, Direction d) const
      {
	return longEffectOfDirection(getPieceOnBoard(square), d);
      }
      template <Ptype PTYPE>
      const mask_t effectBit(Player P, Position target) const 
      {
	return getEffect(target).template selectBit<PTYPE>() & getOnBoardMask(P).template getMask<PTYPE>();
      }
      const mask_t effectBit(Player P, Ptype ptype, Position target) const;
      template <Ptype PTYPE> const mask_t selectLong(Position target) const 
      {
	return getEffect(target).selectLong<PTYPE>() >> 8;
      }
      template <Ptype PTYPE> const mask_t selectLong(Position target, Player owner) const 
      {
	return selectLong<PTYPE>(target) & getOnBoardMask(owner).getMask(1);
      }
      const mask_t selectLong(Position target, Player owner) const 
      {
	return (getEffect(target).selectLong() >> 8)
	  & getOnBoardMask(owner).getMask(1);
      }

      const BoardMask changedEffects(Player pl) const{
	assert(! effects.changedEffects(pl).isInvalid());
	return effects.changedEffects(pl);
      }
      const BoardMask changedEffects() const{
	BoardMask ret = changedEffects(BLACK);
	return ret |= changedEffects(WHITE);
      }
      const NumBitmapEffect changedPieces() const{
	return effects.changedPieces();
      }
      template <Ptype PTYPE> bool longEffectChanged() const 
      {
	return changedPieces().template hasLong<PTYPE>();
      }
      template <Ptype PTYPE> bool anyEffectChanged() const 
      {
	return changedPieces().template hasAny<PTYPE>();
      }
      PieceMask pin(Player king) const
      {
	return pin_or_open[king]&getOnBoardMask(king);
      }
      PieceMask pinOrOpen(Player king) const
      {
	return pin_or_open[king];
      }
      uint64_t Iking8Info(Player king) const
      {
	return king8infos[king];
      }
      const checkmate::King8Info king8Info(Player king) const;
      const PieceMask promotedPieces() const { return promoted; }
    private:
      template<Direction DIR>
      void makePinOpenDir(Position target,
			  PieceMask& pins, PieceMask const& onBoard,Player defense)
      {
	const Offset offset = DirectionTraits<DIR>::blackOffset();
	Position pos=target-offset;
	int num;
	while(Piece::isEmptyNum(num=getPieceAt(pos).number()))
	  pos-=offset;
	king_mobility[defense][DIR]=static_cast<unsigned char>(pos.uintValue());
	if(Piece::isEdgeNum(num)) return;
	int num1=longEffectNumTable()[num][DIR];
	if(Piece::isPieceNum(num1) && onBoard.test(num1)){
	  pins.set(num);
	}
      }
      void recalcPinOpen(Position changed, Direction &lastDir, Player defense)
      {
	Position target=getKingPosition(defense);
#ifdef ALLOW_KING_ABSENCE
	if (target.isPieceStand())
	  return;
#endif
	const Direction longD=Board_Table.getLongDirection<BLACK>(changed,target);
	if(!isLong(longD) || (lastDir!=UL && longD==lastDir)) return;
	lastDir=longD;
	Direction shortD=longToShort(longD);
	{
	  // reset old pins
	  Position oldPos=Position::makeDirect(king_mobility[defense][shortD]);
	  int oldNum=getPieceAt(oldPos).number();
	  if(Piece::isPieceNum(oldNum))
	    pin_or_open[defense].reset(oldNum);
	}
	const Offset offset = Board_Table.getOffsetForBlack(longD);
	Position pos=target-offset;
	int num;
	while(Piece::isEmptyNum(num=getPieceAt(pos).number()))
	  pos-=offset;
	king_mobility[defense][shortD]=static_cast<unsigned char>(pos.uintValue());
	if(Piece::isEdgeNum(num)) return;
	int num1=longEffectNumTable()[num][shortD];
	if(Piece::isPieceNum(num1) && getOnBoardMask(alt(defense)).test(num1)){
	  pin_or_open[defense].set(num);
	}
      }
      PieceMask makePinOpen(Position target,Player defense);
    public:
      void makePinOpen(Player defense);
      template<Player P>
      void makeKing8Info();
    public:
      bool isConsistent(bool showError=true) const;
      /** 局面更新に関する一貫性をテスト */
      bool isConsistent(const NumEffectState& prev, Move moved, bool show_error=true) const;
      // move methods
      void doSimpleMove(Position from, Position to, int promoteMask);
      void doDropMove(Position to,Ptype ptype);
      void doCaptureMove(Position from, Position to, Piece target,int promoteMask);
      /**
       * あるマスにあるDirectionでの長い利きがあるかどうか.
       * 64bit版対応済み
       */
      template<Direction Dir,Player P>
      bool hasEffectDir(Position to) const {
	BOOST_STATIC_ASSERT( (DirectionTraits<Dir>::isLong) );
	const PieceMask& onBoardMask=getOnBoardMask(P);
	mask_t mask1=onBoardMask.getMask(1);
	mask1 &= ((PtypeDirectionTraits<LANCE,Dir>::canMove 
		   ? mask_t::makeDirect(PtypeFuns<LANCE>::indexMask) 
		   : mask_t::makeDirect(0)) 
		  | (PtypeDirectionTraits<BISHOP,Dir>::canMove 
		     ?  mask_t::makeDirect(PtypeFuns<BISHOP>::indexMask) 
		     :  mask_t::makeDirect(0))
		  | (PtypeDirectionTraits<ROOK,Dir>::canMove 
		     ? mask_t::makeDirect(PtypeFuns<ROOK>::indexMask)
		     : mask_t::makeDirect(0)));
	mask1 <<= 8;
	// 短い利きを排除
	mask1&=getEffect(to).getMask(1)& NumBitmapEffect::longEffectMask();
	while (mask1.any())
	{
	  int num=mask1.takeOneBit()+NumBitmapEffect::longToNumOffset;
	  Position from = getPieceOf(num).position();
	  Direction dir=Board_Table.getLongDirection<BLACK>(Offset32(to,from));
	  if (dir==DirectionPlayerTraits<Dir,P>::directionByBlack)
	    return true;
	}
	return false;
      }
      /**
       * あるマスにPTYPEの長い利きがあるかどうか.
       */
      template <Ptype PTYPE>
      bool hasEffectLong(Player P, Position to) const {
	BOOST_STATIC_ASSERT( (PTYPE == LANCE || PTYPE == BISHOP || PTYPE == ROOK) );
	const PieceMask& onBoardMask=getOnBoardMask(P);
	mask_t mask1=onBoardMask.getMask(1);
	mask1 &= mask_t::makeDirect(PtypeFuns<PTYPE>::indexMask);
	mask1 <<= 8;
	// 短い利きを排除
	mask1&=getEffect(to).getMask(1)& NumBitmapEffect::longEffectMask();
	return mask1.any();
      }
      /** 
       * 対象とするマスにあるプレイヤーの利きがあるかどうか.
       * @param player 攻撃側
       * @param target 対象のマス
       */
      bool hasEffectBy(Player player,Position target) const {
	assert(target.isOnBoard());
	mask_t mask=getEffect(target).getMask(1);
	mask&=NumBitmapEffect::playerEffectMask(player);
	return !mask.none();
      }
      /**
       * pinされている駒以外からの利きがない
       */
      template<Player P>
      bool hasEffectByNotPinned(Position target) const{
	assert(target.isOnBoard());
	PieceMask m=getOnBoardMask(P)& ~pinOrOpen(alt(P)) & getEffect(target);
	return m.any();
      }
      /** 
       * 対象とするマスにあるプレイヤーの(ただしある駒以外)利きがあるかどうか.
       * 指定した駒の利きがあることが前提.
       * @param player 攻撃側
       * @param piece 攻撃側の駒
       * @param target 対象のマス
       */
      bool hasEffectNotBy(Player player,Piece piece,Position target) const {
	assert(piece.owner()==player);
	PieceMask onBoardMask=getOnBoardMask(player);
	int num=piece.number();
	onBoardMask.reset(num);
	return (onBoardMask&getEffect(target)).any();
      }
      /** 
       * 対象とするマスにあるプレイヤーの(ただしある駒以外)利きがあるかどうか.
       * 指定した駒の利きがあることが前提.
       * @param player 攻撃側
       * @param piece 攻撃側の駒
       * @param target 対象のマス
       */
      bool hasEffectNotMask(Player player,Position target,PieceMask const& notMask) const {
	PieceMask onBoardMask=getOnBoardMask(player);
	return (onBoardMask&getEffect(target)&notMask).any();
      }
      /** 
       * 二つ以上の駒から利きがある.
       * そもそも利きがない場合も呼ばれる
       * @param player 攻撃側
       * @param target 対象のマス
       */
      bool hasMultipleEffectBy(Player player,Position target) const 
      {
	mask_t mask=getEffect(target).getMask(1);
	mask&=NumBitmapEffect::playerEffectMask(player);
	return NumBitmapEffect::playerEffect(player).getMask(1) < mask;
      }

      /** 
       * 駒attack が target に利きを持つか (旧hasEffectToと統合)
       * @param target 対象のマス
       */
      bool hasEffectByPiece(Piece attack, Position target) const 
      {
	assert(attack.isPiece());
	assert(target.isOnBoard());
	return getEffect(target).test(attack.number());
      }

      /**
       * target に ptype の利きがあるか? 成不成を区別しない
       */
      template <Ptype PTYPE>
      bool hasEffectByPtype(Player attack, Position target) const
      {
	return effectBit<PTYPE>(attack, target).any();
      }
      /**
       * target に ptype の利きがあるか? 成不成を区別
       */
      template <Ptype PTYPE>
      bool hasEffectByPtypeStrict(Player attack, Position target) const
      {
	mask_t mask=effectBit<PTYPE>(attack, target);
	if (isPromoted(PTYPE)) 
	  mask &= promoted.getMask<PTYPE>();
	else
	  mask &= ~(promoted.getMask<PTYPE>());
	return mask.any();
      }
      /** return a piece s.t. owner == attack, ptype == PTYPE, has effect on target.  return Piece::EMPTY() otherwise */
      template <Ptype PTYPE>
      const Piece effectPtype(Player attack, Position target) const
      {
	mask_t mask=effectBit<PTYPE>(attack, target);
	if (mask.none())
	  return Piece::EMPTY();
	return getPieceOf(mask.takeOneBit()+PtypeFuns<PTYPE>::indexNum*32);
      }
      template <Ptype PTYPE>
      const Piece effectPtypeStrict(Player attack, Position target) const
      {
	mask_t mask=effectBit<PTYPE>(attack, target);
	if (isPromoted(PTYPE)) 
	  mask &= promoted.getMask<PTYPE>();
	else
	  mask &= ~(promoted.getMask<PTYPE>());
	if (mask.none())
	  return Piece::EMPTY();
	return getPieceOf(mask.takeOneBit()+PtypeFuns<PTYPE>::indexNum*32);
      }

      /**
       * 利きの数を数える. 
       * targetが盤をはみ出してはいけない
       */
      int countEffect(Player player,Position target) const 
      {
	assert(target.isOnBoard());
	return getEffect(target).countEffect(player);
      }
      /**
       * 利きの数を数える. 
       * targetが盤をはみ出してはいけない
       * @param pins この駒の利きは数えない
       */
      int countEffect(Player player,Position target, PieceMask pins) const 
      {
	assert(target.isOnBoard());
	const NumBitmapEffect effect = getEffect(target);
	const int all = effect.countEffect(player);
	pins &= effect;
	return all - pins.countBit();
      }
      /**
       * @param P 攻撃側
       * @param Type 攻撃駒
       */
      template<Player P,Ptype Type,class Action>
      void forEachEffectPtype(Position pos,Action & action) const {
	mask_t mask=effectBit<Type>(P, pos);
	while (mask.any())
	{
	  int num=mask.takeOneBit()+(PtypeFuns<Type>::indexNum<<5);
	  action.template doActionPtype<P,Type>(getPieceOf(num),pos);
	}
      }

    private:
      template<Player P,class Action>
      void forEachEffect(const PieceMask& pieces, Position pos,Action & action) const
      {
#if OSL_WORDSIZE == 64
	mask_t mask=pieces.getMask(0);
	while (mask.any())
	{
	  const int num=mask.takeOneBit();
	  action.template doAction<P>(getPieceOf(num),pos);
	}
#elif OSL_WORDSIZE == 32
	mask_t mask0=pieces.getMask(0);
	while (mask0.any())
	{
	  const int num=mask0.takeOneBit();
	  action.template doAction<P>(getPieceOf(num),pos);
	}
	mask_t mask1=pieces.getMask(1);
	while (mask1.any())
	{
	  const int num=mask1.takeOneBit()+32;
	  action.template doAction<P>(getPieceOf(num),pos);
	}
#endif
      }      
    public:
      /** 
       * pos への利きを持つ各駒に関して処理を行う.
       * @param action たとえば AlwaysMoveAction
       */
      template<Player P,class Action>
      void forEachEffect(Position pos,Action & action) const
      {
	const PieceMask pieceMask=getOnBoardMask(P)&getEffect(pos);
	forEachEffect<P,Action>(pieceMask, pos, action);
      }
      /** 
       * pos にある駒を取る move を生成して action の member を呼び出す.
       * @param pin 無視する駒
       * @param action たとえば AlwaysMoveAction
       */
      template<Player P,class Action>
      void forEachEffect(Position pos,Action & action,const PieceMask& pin) const
      {
	PieceMask pieceMask=getOnBoardMask(P)&getEffect(pos);
	pieceMask &= ~pin;
	forEachEffect<P,Action>(pieceMask, pos, action);
      }

      /** 
       * pos に移動する move を生成して action の member を呼び出す
       * @param action たとえば AlwayMoveAction
       * @param piece  これ以外の駒を使う
       */
      template<Player P,class Action>
      void forEachEffectNotBy(Position pos,Piece piece,Action & action) const {
	PieceMask pieces=getOnBoardMask(P)&getEffect(pos);
	pieces.reset(piece.number());
	forEachEffect<P,Action>(pieces, pos, action);
      }

      /** 
       * 対象とするマスにあるプレイヤーの利きがあるかどうか.
       * @param player 攻撃側
       * @param target 対象のマス
       */
      template<Player P>
      bool hasEffectBy(Position target) const {
	assert(target.isOnBoard());
	mask_t mask=getEffect(target).getMask(1);
	mask&=NumBitmapEffect::playerEffectMask<P>();
	return !mask.none();
      }
    
      bool hasEffectBy(Player P, Position target,Piece& attackerPiece) const 
      {
	if (P == BLACK)
	  return hasEffectBy<BLACK>(target, attackerPiece);
	else
	  return hasEffectBy<WHITE>(target, attackerPiece);
      }
      /**
       * @param P(template) - 利きをつけている側のプレイヤ
       * @param target - 利きをつけられた場所
       * @param attackerPiece - multiple attackの場合はPiece::EMPTY()
       *        そうでないなら利きをつけている駒を返す
       */
      template<Player P>
      bool hasEffectBy(Position target,Piece& attackerPiece) const {
	attackerPiece=Piece::EMPTY();
	const PieceMask& pieceMask=getOnBoardMask(P)&getEffect(target);
#if OSL_WORDSIZE == 64
	mask_t mask=pieceMask.getMask(0);
	if (mask.none()) return false;
	/**
	 * mask|=0x8000000000000000ll;
	 * if((mask&(mask-1))!=0x8000000000000000ll) なら1つのif文で済む
	 */
	if (mask.hasMultipleBit())
	  return true;
	int num=mask.bsf();
	attackerPiece=getPieceOf(num);
	return true;
#elif OSL_WORDSIZE == 32
	mask_t mask0=pieceMask.getMask(0);
	mask_t mask1=pieceMask.getMask(1);
	if (mask0.any())
	{
	  if (mask1.any())
	    return true;
	  int num=mask0.bsf();
	  if (mask0 == PieceMask::numToMask(num))
	    attackerPiece=getPieceOf(num);
	  return true;
	}
	else if (mask1.any())
	{
	  int num=mask1.bsf();
	  if (mask1==PieceMask::numToMask(num))
	    attackerPiece=getPieceOf(num+32);
	  return true;
	}
	else 
	  return false;
#endif
      }
      /**
       * 利きの中から安そうな駒を選ぶ
       */
      const Piece selectCheapPiece(PieceMask effect) const;
      /**
       * @param P - 利きをつけている側のプレイヤ
       * @param square - 調査する場所
       * @return 利きを付けている中で安そうな駒 (複数の場合でもEMPTYにはしない)
       */
      const Piece findCheapThreat(Player P, Position square) const 
      {
	return selectCheapPiece(getOnBoardMask(P) & getEffect(square));
      }
      /**
       * @param P - 利きをつけている側のプレイヤ
       * @param square - 調査する場所
       * @return 利きを付けている中で安そうな駒 (複数の場合でもEMPTYにはしない)
       */
      const Piece findCheapThreatNotBy(Player P, Position square, const PieceMask& ignore) const 
      {
	PieceMask pieces = getOnBoardMask(P);
	pieces &= ~ignore;
	return selectCheapPiece(pieces & getEffect(square));
      }
      const Piece findThreatNotBy(Player P, Position square, const PieceMask& ignore) const 
      {
	PieceMask pieces = getOnBoardMask(P);
	pieces &= ~ignore;
	pieces &= getEffect(square);
	if (pieces.none())
	  return Piece::EMPTY();
	return getPieceOf(pieces.takeOneBit());
      }

      /** 取られそうなPの駒で価値が最大のもの */
      const Piece findThreatenedPiece(Player P) const;

      /** Pの玉が王手状態 */
      bool inCheck(Player P) const 
      {
	const Position king = getKingPosition(P);
#ifdef ALLOW_KING_ABSENCE
	if (king.isPieceStand())
	  return false;
#endif
	return hasEffectBy(alt(P), king);
      }
      /** 手番の玉が王手状態 */
      bool inCheck() const { return inCheck(getTurn()); }
      /**
       * 王手駒を探す
       * @return 王手かどうか
       * @param attack_piece
       * 一つの駒による王手の場合はattck_pieceにその駒を入れる
       * 複数の駒による王手の場合はPiece::EMPTY()を入れる
       * @param P(template) 玉
       */
      template<Player P>
      bool findCheckPiece(Piece& attack_piece) const
      {
	return hasEffectBy<PlayerTraits<P>::opponent>(getKingPosition(P),attack_piece);
      }
      /**
       * 合法手かどうかを簡単に検査する．局面に依存するチェックのみ．
       * ルール上指せない手である可能性がある場合は，isValidMove を用いる．
       *
       * Pをtemplate引数にできると少しは速くなりそう
       * 局面に依存する検査でも，玉の素抜きや王手を防いでいるか，
       * 千日手，打歩詰かどうかは検査しない．
       * @param showError(template) - falseを返す時には何か表示する.
       * @param move - 手
       */
      template <bool show_error>
      bool isAlmostValidMove(Move move) const;
      bool isAlmostValidMove(Move move,bool show_error=true) const;

    private:
      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceDir(Position, Action&, Int2Type<false>) const {}
      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceDir(Position piecePosition,Action & action,Int2Type<true>) const {
	const Offset offset=DirectionPlayerTraits<Dir,P>::offset();
	action.template doAction<P>(this->getPieceAt(piecePosition),piecePosition+offset);
      }

      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceDir(Position piecePosition,Action & action) const {
	forEachEffectOfPieceDir<P,Type,Action,Dir>(piecePosition,action,Int2Type<(PtypeTraits<Type>::moveMask & DirectionTraits<Dir>::mask)!=0>());
      }
      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceLongDir(Position, Action&, Int2Type<false>) const {}
      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceLongDir(Position piecePosition,Action & action,Int2Type<true>) const {
	Piece piece=this->getPieceAt(piecePosition);
	const Offset offset=DirectionPlayerTraits<Dir,P>::offset();
	assert(offset.intValue() != 35);
	Position pos=piecePosition+offset;
	for (;this->getPieceAt(pos).isEmpty();pos+=offset)
	  action.template doAction<P>(piece,pos);
	action.template doAction<P>(piece,pos);
      }

      template<Player P,Ptype Type,class Action,Direction Dir>
      void forEachEffectOfPieceLongDir(Position piecePosition,Action & action) const {
	forEachEffectOfPieceLongDir<P,Type,Action,Dir>(piecePosition,action,Int2Type<(PtypeTraits<Type>::moveMask & DirectionTraits<Dir>::mask)!=0>());
      }
    public:
      /**
       * piecePositionにある駒によって利きを受けるすべてのposition
       * (空白含む)について
       * actionを実行する
       *
       * SimpleState より移植: ImmediateAddSupportがなければ消せる
       */
      template<Player P,Ptype Type,class Action>
      void forEachEffectOfPiece(Position piecePosition,Action & action) const;
      template<class Action>
      void forEachEffectOfPiece(Piece piece,Action & action) const;

      /**
       * attackerにptypeoの駒がいると仮定した場合にtargetに利きがあるかどうか
       * を stateをupdateしないで確かめる.
       * targetPositionは空白でも良い
       * 盤上の駒を動かす事を検討しているときはに，
       * 自分自身の陰に入って利かないと見なされることがある
       */
      bool hasEffectFromTo(PtypeO ptypeo,Position attacker,
			   Position target) const
      {
	Offset32 offset32=Offset32(target,attacker);
	EffectContent effect=Ptype_Table.getEffect(ptypeo,offset32);
	if (! effect.hasEffect()) 
	  return false;
	if (effect.hasUnblockableEffect())
	  return true;
	assert(Board_Table.getShortOffset(offset32) == effect.offset());
	return this->isEmptyBetween(attacker,target,effect.offset());
      }
      /**
       * 
       */
      template<Player P>
      bool hasEffectByWithRemove(Position target,Position removed) const;
    
      bool hasEffectByWithRemove(Player player, Position target,Position removed) const{
	if (player==BLACK)
	  return hasEffectByWithRemove<BLACK>(target,removed);
	else
	  return hasEffectByWithRemove<WHITE>(target,removed);
      }

      void showEffect(std::ostream& os) const;

      /** pl からの利きが(1つ以上)ある駒一覧 */
      const PieceMask effectedMask(Player pl) const
      {
	return effects.effected_mask[pl];
      }
      /** 前の指手でeffectedMask(pl)が変化したか.
       * 取られた駒は現在の実装ではリストされないようだ.
       */
      const PieceMask effectedChanged(Player pl) const
      {
	return effects.effected_changed_mask[pl];
      }
      void makeMovePass()
      {
	changeTurn();
	effects.clearChangedEffects();
	effects.clearEffectedChanged();
      }
      /**
       * pinされた駒がPのKingから見てどの方向か?
       * Pから見たdirectionを返す
       */
      template<Player P>
      Direction pinnedDir(Piece p) const
      {
	assert(pinOrOpen(P).test(p.number()));
	Position king=getKingPosition<P>();
	return Board_Table.getShort8<P>(p.position(),king);
      }
      /**
       * pinされた駒pがtoに動けるか?
       * pinに関係がなければtoへ動けるという前提
       */
      template<Player P>
      bool pinnedCanMoveTo(Piece p,Position to) const
      {
	Direction d=pinnedDir<P>(p);
	Position from=p.position();
	assert(hasEffectFromTo(p.ptypeO(),from,to));
	return primDir(d)==primDirUnsafe(Board_Table.getShort8Unsafe<P>(from,to));
      }
    };

    inline bool operator!=(const NumEffectState& s1,
			   const NumEffectState& s2)
    {
      return !(s1==s2);
    }  
  } // namespace effect
  using state::NumEffectState;

  namespace apply_move
  {
    template<Player P>
    struct ApplyDoUndoSimpleMove<P,NumEffectState>
    {
      typedef NumEffectState state_t;
      template <typename F>
      static void doUndoSimpleMove(state_t& s, 
				   Position from, Position to, int promoteMask,F& func);

      static 
      void prologue(state_t& s, Position from, Position to, int promoteMask,
		    Piece& oldPiece, int& num, 
		    PtypeO& oldPtypeO, PtypeO& new_ptypeo,
		    CArray<PieceMask,2>& pin_or_open_backup,
		    KingMobility& king_mobility_backup,
		    PieceMask& promoted_backup,
		    CArray<PieceMask,2>& effected_mask_backup,
		    CArray<PieceMask,2>& effected_changed_mask_backup,
		    CArray<uint64_t,2>& king8infos_backup,
		    MobilityTable &mobility_backup
		    );
      static 
      void epilogue(state_t& s, Position from, Position to, Piece oldPiece, 
		    int num, PtypeO oldPtypeO, PtypeO newPtypeO,
		    const CArray<PieceMask,2>& pin_or_open_backup,
		    const KingMobility& king_mobility_backup,
		    const PieceMask& promoted_backup,
		    const CArray<PieceMask,2>& effected_mask_backup,
		    const CArray<PieceMask,2>& effected_changed_mask_backup,
		    const CArray<uint64_t,2>& king8infos_backup,
		    const MobilityTable &mobility_backup
);
    };
  
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoSimpleMove<P,NumEffectState>::
    doUndoSimpleMove(state_t& s, 
		     Position from, Position to, int promoteMask,F& func)
    {
      Piece oldPiece;
      int num;
      PtypeO oldPtypeO, newPtypeO;
      CArray<PieceMask,2> pin_or_open_backup;
      KingMobility king_mobility_backup;
      PieceMask promoted_backup;
      CArray<PieceMask,2> effected_mask_backup;
      CArray<PieceMask,2> effected_changed_mask_backup;
      CArray<uint64_t,2> king8infos_backup;
      MobilityTable mobility_backup;
      prologue(s, from, to, promoteMask, oldPiece, num, oldPtypeO, newPtypeO, 
	       pin_or_open_backup, 
	       king_mobility_backup,
	       promoted_backup,
		     effected_mask_backup, effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);

      if (promoteMask!=0 && num < PtypeTraits<PAWN>::indexLimit)
	{
	  s.clearPawn(P,from);
	  s.changeTurn();
	  func(to);
	  s.changeTurn();
	  s.setPawn(P,from);
	}
      else
	{
	  s.changeTurn();
	  func(to);
	  s.changeTurn();
	}

      epilogue(s, from, to, oldPiece, num, oldPtypeO, newPtypeO, 
	       pin_or_open_backup, 
	       king_mobility_backup,
	       promoted_backup, effected_mask_backup, effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);
    }

    template<Player P>
    struct ApplyDoUndoDropMove<P,NumEffectState>
    {
      typedef NumEffectState state_t;
      template <typename F>
      static void doUndoDropMove(state_t& s, 
				 Position to, Ptype ptype,F& func);
      static 
      void prologue(state_t& s, Position to, Ptype ptype,
		    Piece& oldPiece, int& num, PtypeO& ptypeO, 
		    int& numIndex, mask_t& numMask,
		    CArray<PieceMask,2>& pin_or_open_backup,
		    KingMobility& king_mobility_backup,
		    CArray<PieceMask,2>& effected_mask_backup,
		    CArray<PieceMask,2>& effected_changed_mask_backup,
		    CArray<uint64_t,2>& king8infos_backup,
 		    MobilityTable &mobility_backup
		    );
      static 
      void epilogue(state_t& s, Position to, Ptype ptype, Piece oldPiece, 
		    int num, PtypeO ptypeO, int numIndex, mask_t numMask,
		    const CArray<PieceMask,2>& pin_or_open_backup,
		    const KingMobility& king_mobility_backup,
		    const CArray<PieceMask,2>& effected_mask_backup,
		    const CArray<PieceMask,2>& effected_changed_mask_backup,
		    const CArray<uint64_t,2>& king8infos_backup,
		    const MobilityTable &mobility_backup
		    );
    };
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoDropMove<P,NumEffectState>::
    doUndoDropMove(state_t& s, 
		   Position to, Ptype ptype,F& func)
    {
      Piece oldPiece;
      PtypeO ptypeO;
      int num, numIndex;
      mask_t numMask;
      CArray<PieceMask,2> pin_or_open_backup;
      KingMobility king_mobility_backup;
      CArray<PieceMask,2> effected_mask_backup;
      CArray<PieceMask,2> effected_changed_mask_backup;
      CArray<uint64_t,2> king8infos_backup;
      MobilityTable mobility_backup;
      prologue(s, to, ptype, oldPiece, num, ptypeO, numIndex, numMask, 
	       pin_or_open_backup, king_mobility_backup,
	       effected_mask_backup,effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);

      if (ptype==PAWN)
	{
	  s.setPawn(P,to);
	  s.changeTurn();
	  func(to);
	  s.changeTurn();
	  s.clearPawn(P,to);
	}
      else
	{
	  s.changeTurn();
	  func(to);
	  s.changeTurn();
	}
      epilogue(s, to, ptype, oldPiece, num, ptypeO, numIndex, numMask, 
	       pin_or_open_backup, king_mobility_backup,
	       effected_mask_backup,effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);
    }

    template<Player P>
    struct ApplyDoUndoCaptureMove<P,NumEffectState>
    {
      typedef NumEffectState state_t;
      template <typename F>
      static void doUndoCaptureMove(state_t& s, Position from,Position to, 
				    Piece target, int promoteMask,F& func);
      static
      void prologue(state_t& s, Position from, Position to, Piece target, 
		    int promoteMask,
		    Piece& oldPiece, PtypeO& oldPtypeO, PtypeO& capturePtypeO, 
		    PtypeO& new_ptypeo, int& num0, int& num1, 
		    int& num1Index, mask_t& num1Mask,
		    CArray<PieceMask,2>& pin_or_open_backup,
		    KingMobility& king_mobility_backup,
		    PieceMask& promoted_backup,
		    CArray<PieceMask,2>& effected_mask_backup,
		    CArray<PieceMask,2>& effected_changed_mask_backup,
		    CArray<uint64_t,2>& king8infos_backup,
		    MobilityTable &mobility_backup
);
      static
      void epilogue(state_t& s, Position from, Position to, Piece target, 
		    Piece oldPiece, PtypeO oldPtypeO, PtypeO capturePtypeO, 
		    PtypeO newPtypeO, int num0, int num1, 
		    int num1Index, mask_t num1Mask,
		    const CArray<PieceMask,2>& pin_or_open_backup,
		    const KingMobility& king_mobility_backup,
		    const PieceMask& promoted_backup,
		    const CArray<PieceMask,2>& effected_mask_backup,
		    const CArray<PieceMask,2>& effected_changed_mask_backup,
		    const CArray<uint64_t,2>& king8infos_backup,
		    const MobilityTable &mobility_backup
);
    };
  
    template<Player P>
    template <typename F>
    void ApplyDoUndoCaptureMove<P,NumEffectState>::
    doUndoCaptureMove(state_t& s, Position from,Position to, Piece target, 
		      int promoteMask,F& func)
    {
      Piece oldPiece;
      PtypeO oldPtypeO, capturePtypeO, newPtypeO;
      int num0, num1, num1Index;
      mask_t num1Mask;
      CArray<PieceMask,2> pin_or_open_backup;
      KingMobility king_mobility_backup;
      PieceMask promoted_backup;
      CArray<PieceMask,2> effected_mask_backup;
      CArray<PieceMask,2> effected_changed_mask_backup;
      CArray<uint64_t,2> king8infos_backup;
      MobilityTable mobility_backup;
      prologue(s, from, to, target, promoteMask, oldPiece, oldPtypeO, 
	       capturePtypeO, newPtypeO, num0, num1, num1Index,num1Mask, 
	       pin_or_open_backup, king_mobility_backup,
	       promoted_backup,
	       effected_mask_backup, effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);

      s.changeTurn();
      const Ptype capturePtype=target.ptype();
      if (capturePtype==PAWN)
	{
	  s.clearPawn(PlayerTraits<P>::opponent,to);
	  if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit)
	    {
	      s.clearPawn(P,from);
	      func(to);
	      s.setPawn(P,from);
	    }
	  else
	    {
	      func(to);
	    }
	  s.setPawn(PlayerTraits<P>::opponent,to);
	}
      else if (promoteMask!=0 && num0<PtypeTraits<PAWN>::indexLimit)
	{
	  s.clearPawn(P,from);
	  func(to);
	  s.setPawn(P,from);
	}
      else
	{
	  func(to);
	}
      s.changeTurn();

      epilogue(s, from, to, target, oldPiece, oldPtypeO, capturePtypeO, newPtypeO, 
	       num0, num1, num1Index,num1Mask, 
	       pin_or_open_backup, king_mobility_backup,
	       promoted_backup,effected_mask_backup, effected_changed_mask_backup,
	       king8infos_backup,
	       mobility_backup);
    }
} // namespace state
  using state::NumEffectState;
} // namespace osl

template <class Action>
void osl::NumEffectState::
forEachEffectOfPiece(Piece piece,Action & action) const 
{
  Position piecePosition = piece.position();
  switch (piece.ptypeO()) {
  case NEW_PTYPEO(WHITE,PAWN): forEachEffectOfPiece<WHITE,PAWN,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,LANCE): forEachEffectOfPiece<WHITE,LANCE,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,KNIGHT): forEachEffectOfPiece<WHITE,KNIGHT,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,SILVER): forEachEffectOfPiece<WHITE,SILVER,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PPAWN): forEachEffectOfPiece<WHITE,PPAWN,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PLANCE): forEachEffectOfPiece<WHITE,PLANCE,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PKNIGHT): forEachEffectOfPiece<WHITE,PKNIGHT,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PSILVER): forEachEffectOfPiece<WHITE,PSILVER,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,GOLD): forEachEffectOfPiece<WHITE,GOLD,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,BISHOP): forEachEffectOfPiece<WHITE,BISHOP,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PBISHOP): forEachEffectOfPiece<WHITE,PBISHOP,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,ROOK): forEachEffectOfPiece<WHITE,ROOK,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,PROOK): forEachEffectOfPiece<WHITE,PROOK,Action>(piecePosition,action); break;
  case NEW_PTYPEO(WHITE,KING): forEachEffectOfPiece<WHITE,KING,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PAWN): forEachEffectOfPiece<BLACK,PAWN,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,LANCE): forEachEffectOfPiece<BLACK,LANCE,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,KNIGHT): forEachEffectOfPiece<BLACK,KNIGHT,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,SILVER): forEachEffectOfPiece<BLACK,SILVER,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PPAWN): forEachEffectOfPiece<BLACK,PPAWN,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PLANCE): forEachEffectOfPiece<BLACK,PLANCE,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PKNIGHT): forEachEffectOfPiece<BLACK,PKNIGHT,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PSILVER): forEachEffectOfPiece<BLACK,PSILVER,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,GOLD): forEachEffectOfPiece<BLACK,GOLD,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,BISHOP): forEachEffectOfPiece<BLACK,BISHOP,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PBISHOP): forEachEffectOfPiece<BLACK,PBISHOP,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,ROOK): forEachEffectOfPiece<BLACK,ROOK,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,PROOK): forEachEffectOfPiece<BLACK,PROOK,Action>(piecePosition,action); break;
  case NEW_PTYPEO(BLACK,KING): forEachEffectOfPiece<BLACK,KING,Action>(piecePosition,action); break;
  default: assert(0);
  }
}

template <osl::Player P, osl::Ptype Type, class Action>
void osl::NumEffectState::
forEachEffectOfPiece(Position piecePosition,Action & action) const 
{
  forEachEffectOfPieceDir<P,Type,Action,UL>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,U>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,UR>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,L>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,R>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,DL>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,D>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,DR>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,UUL>(piecePosition,action);
  forEachEffectOfPieceDir<P,Type,Action,UUR>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_UL>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_U>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_UR>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_L>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_R>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_DL>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_D>(piecePosition,action);
  forEachEffectOfPieceLongDir<P,Type,Action,LONG_DR>(piecePosition,action);
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoSimpleMove<P,osl::state::NumEffectState>::
prologue(state_t& s, Position from, Position to, int promoteMask,
	      Piece& oldPiece, int& num, 
	      PtypeO& oldPtypeO, PtypeO& new_ptypeo,
	 CArray<PieceMask,2>& pin_or_open_backup,
	 KingMobility& king_mobility_backup,
	      PieceMask& promoted_backup,
	 CArray<PieceMask,2>& effected_mask_backup,
	 CArray<PieceMask,2>& effected_changed_mask_backup,
	 CArray<uint64_t,2>& king8infos_backup,
	 MobilityTable &mobility_backup)
{
  mobility_backup = s.effects.mobilityTable;
  pin_or_open_backup = s.pin_or_open;
  king_mobility_backup = s.king_mobility;
  effected_mask_backup = s.effects.effected_mask;
  effected_changed_mask_backup = s.effects.effected_changed_mask;
  king8infos_backup=s.king8infos;

  oldPiece=s.getPieceAt(from);
  Piece newPiece=oldPiece.promoteWithMask(promoteMask);
  newPiece+=(to-from);
  num=oldPiece.number();

  oldPtypeO=oldPiece.ptypeO();
  new_ptypeo=newPiece.ptypeO();
  // 自分自身の効きを外す
  s.setPieceOf(num,newPiece);
  s.effects.clearChangedEffects();
  s.effects.clearEffectedChanged();
  s.effects.template doEffect<NumBitmapEffect::Sub,true>(s,oldPtypeO,from,num);
  // 自分自身がブロックしていたpromote?の延長
  // あるいは自分自身のブロック
  s.effects.effectedNumTable[num].clear();
  s.setBoard(to,newPiece);
  s.effects.template doBlockAt<NumBitmapEffect::Sub,true>(s,to,num);
  s.setBoard(from,Piece::EMPTY());
  s.effects.template doBlockAt<NumBitmapEffect::Add,true>(s,from,num);
  s.effects.template doEffect<NumBitmapEffect::Add,true>(s,new_ptypeo,to,num);

  if (oldPtypeO == newPtypeO(P,KING))
    s.makePinOpen(P);
  else {
    Direction lastD=UL;
    s.pin_or_open[P].reset(num);
    s.recalcPinOpen(from,lastD,P);
    s.recalcPinOpen(to,lastD,P);
  }
  {
    Direction lastD=UL;
    s.pin_or_open[alt(P)].reset(num);
    s.recalcPinOpen(from,lastD,alt(P));
    s.recalcPinOpen(to,lastD,alt(P));
  }
  promoted_backup = s.promoted;
  if (promoteMask)
    s.promoted.set(num);
  if(s.hasEffectBy(BLACK,to))
    s.effects.effected_mask[BLACK].set(num);
  else
    s.effects.effected_mask[BLACK].reset(num);
  if(s.hasEffectBy(WHITE,to))
    s.effects.effected_mask[WHITE].set(num);
  else
    s.effects.effected_mask[WHITE].reset(num);
  s.effects.effected_changed_mask[BLACK].set(num);
  s.effects.effected_changed_mask[WHITE].set(num);
  {
    BoardMask changed=s.changedEffects(BLACK)|s.changedEffects(WHITE);
    changed.set(from);
    changed.set(to);
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<BLACK>()))
       || s.pin_or_open[BLACK]!=pin_or_open_backup[BLACK])
      s.makeKing8Info<BLACK>();
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<WHITE>()))
       || s.pin_or_open[WHITE]!=pin_or_open_backup[WHITE])
      s.makeKing8Info<WHITE>();
  }
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoSimpleMove<P,osl::state::NumEffectState>::
epilogue(state_t& s, Position from, Position to, Piece oldPiece, 
	      int num, PtypeO oldPtypeO, PtypeO newPtypeO,
	      const CArray<PieceMask,2>& pin_or_open_backup,
	 const KingMobility& king_mobility_backup,
	      const PieceMask& promoted_backup,
	 const CArray<PieceMask,2>& effected_mask_backup,
	 const CArray<PieceMask,2>& effected_changed_mask_backup,
	 const CArray<uint64_t,2>& king8infos_backup,
	 const MobilityTable & mobility_backup
	 )
{
  s.setPieceOf(num,oldPiece);
  s.effects.template doEffect<NumBitmapEffect::Sub,false>(s,newPtypeO,to,num);
  s.setBoard(from,oldPiece);
  s.effects.effectedNumTable[num].clear();
  s.effects.template doBlockAt<NumBitmapEffect::Sub,false>(s,from,num);
  s.setBoard(to,Piece::EMPTY());
  s.effects.template doBlockAt<NumBitmapEffect::Add,false>(s,to,num);
  s.effects.template doEffect<NumBitmapEffect::Add,false>(s,oldPtypeO,from,num);
  s.effects.invalidateChangedEffects();
  s.pin_or_open = pin_or_open_backup;
  s.king_mobility = king_mobility_backup;
  s.promoted = promoted_backup;
  s.effects.effected_mask = effected_mask_backup;
  s.effects.effected_changed_mask = effected_changed_mask_backup;
  s.effects.mobilityTable = mobility_backup;
  s.king8infos = king8infos_backup;
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoDropMove<P,osl::state::NumEffectState>::
prologue(state_t& s, Position to, Ptype ptype,
	      Piece& oldPiece, int& num, PtypeO& ptypeO, 
	      int& numIndex, mask_t& numMask,
	      CArray<PieceMask,2>& pin_or_open_backup,
	 KingMobility& king_mobility_backup,
	 CArray<PieceMask,2>& effected_mask_backup,
	 CArray<PieceMask,2>& effected_changed_mask_backup,
	 CArray<uint64_t,2>& king8infos_backup,
	 MobilityTable &mobility_backup)
{
  king8infos_backup = s.king8infos;
  mobility_backup = s.effects.mobilityTable;
  pin_or_open_backup = s.pin_or_open;
  king_mobility_backup = s.king_mobility;
  effected_mask_backup = s.effects.effected_mask;
  effected_changed_mask_backup = s.effects.effected_changed_mask;
#if OSL_WORDSIZE == 64
  numIndex=0;
#elif OSL_WORDSIZE == 32
  numIndex=Ptype_Table.getIndex(ptype);
#endif
  const mask_t ownMochigoma=
    s.standMask(P).getMask(numIndex) & Ptype_Table.getMaskLow(ptype);
  assert(ownMochigoma.any());
  numMask=ownMochigoma.lowestBit();
  int numLow = ownMochigoma.bsf();
  num = numLow|(numIndex<<5);
  oldPiece=s.getPieceOf(num);
  Piece newPiece=oldPiece;
  newPiece+=to-Position::STAND();
  ptypeO=newPiece.ptypeO();
  s.setPieceOf(num,newPiece);
  s.effects.clearChangedEffects();
  s.effects.clearEffectedChanged();
  s.effects.template doBlockAt<NumBitmapEffect::Sub,true>(s,to,num);
  s.effects.template doEffect<NumBitmapEffect::Add,true>(s,ptypeO,to,num);
  s.setBoard(to,newPiece);
  s.standMask(P).xorMask(numIndex,numMask);
  s.stand_count[P][ptype-PTYPE_BASIC_MIN]--;
  s.mutableOnBoardMask(P).xorMask(numIndex,numMask);
  {
    Direction lastD=UL;
    s.recalcPinOpen(to,lastD,P);
  }
  {
    Direction lastD=UL;
    s.recalcPinOpen(to,lastD,alt(P));
  }
  if(s.hasEffectBy(BLACK,to))
    s.effects.effected_mask[BLACK].set(num);
  else
    s.effects.effected_mask[BLACK].reset(num);
  if(s.hasEffectBy(WHITE,to))
    s.effects.effected_mask[WHITE].set(num);
  else
    s.effects.effected_mask[WHITE].reset(num);
  s.effects.effected_changed_mask[BLACK].set(num);
  s.effects.effected_changed_mask[WHITE].set(num);
  {
    BoardMask changed=s.changedEffects(BLACK)|s.changedEffects(WHITE);
    changed.set(to);
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<BLACK>()))
       || s.pin_or_open[BLACK]!=pin_or_open_backup[BLACK])
      s.makeKing8Info<BLACK>();
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<WHITE>()))
       || s.pin_or_open[WHITE]!=pin_or_open_backup[WHITE])
      s.makeKing8Info<WHITE>();
  }
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoDropMove<P,osl::state::NumEffectState>::
epilogue(state_t& s, Position to, Ptype ptype, Piece oldPiece, 
	      int num, PtypeO ptypeO, int numIndex, mask_t numMask,
	      const CArray<PieceMask,2>& pin_or_open_backup,
	 const KingMobility& king_mobility_backup,
	 const CArray<PieceMask,2>& effected_mask_backup,
	 const CArray<PieceMask,2>& effected_changed_mask_backup,
	 const CArray<uint64_t,2>& king8infos_backup,
	 const MobilityTable& mobility_backup
	 )
{
  s.standMask(P).xorMask(numIndex,numMask);
  s.stand_count[P][ptype-PTYPE_BASIC_MIN]++;
  s.mutableOnBoardMask(P).xorMask(numIndex,numMask);
  s.setBoard(to,Piece::EMPTY());
  s.effects.template doEffect<NumBitmapEffect::Sub,false>(s,ptypeO,to,num);
  s.effects.template doBlockAt<NumBitmapEffect::Add,false>(s,to,num);
  s.setPieceOf(num,oldPiece);
  s.effects.effectedNumTable[num].clear();
  s.effects.invalidateChangedEffects();
  s.pin_or_open = pin_or_open_backup;
  s.king_mobility = king_mobility_backup;
  s.effects.effected_mask = effected_mask_backup;
  s.effects.effected_changed_mask = effected_changed_mask_backup;
  s.effects.mobilityTable = mobility_backup;
  s.king8infos = king8infos_backup;
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoCaptureMove<P,osl::state::NumEffectState>::
prologue(state_t& s, Position from, Position to, Piece target, 
	 int promoteMask,
	 Piece& oldPiece, PtypeO& oldPtypeO, PtypeO& capturePtypeO, 
	 PtypeO& new_ptypeo, int& num0, int& num1, 
	 int& num1Index, mask_t& num1Mask,
	 CArray<PieceMask,2>& pin_or_open_backup,
	 KingMobility& king_mobility_backup,
	 PieceMask& promoted_backup,
	 CArray<PieceMask,2>& effected_mask_backup,
	 CArray<PieceMask,2>& effected_changed_mask_backup,
	 CArray<uint64_t,2>& king8infos_backup,
	 MobilityTable &mobility_backup)
{
  mobility_backup = s.effects.mobilityTable;
  pin_or_open_backup = s.pin_or_open;
  king_mobility_backup = s.king_mobility;
  effected_mask_backup = s.effects.effected_mask;
  effected_changed_mask_backup = s.effects.effected_changed_mask;
  king8infos_backup = s.king8infos;

  num1=target.number();
  num1Index=PieceMask::numToIndex(num1);
  num1Mask=PieceMask::numToMask(num1);
  s.mutableOnBoardMask(PlayerTraits<P>::opponent).xorMask(num1Index,num1Mask);
  s.standMask(P).xorMask(num1Index,num1Mask);
  oldPiece=s.getPieceAt(from);
  Piece newPiece=oldPiece.promoteWithMask(promoteMask);
  newPiece+=(to-from);
  num0=oldPiece.number();
  s.setPieceOf(num0,newPiece);
  s.setPieceOf(num1,target.captured());
      
  oldPtypeO=oldPiece.ptypeO();
  new_ptypeo=newPiece.ptypeO();
  capturePtypeO=target.ptypeO();
  s.stand_count[P][unpromote(getPtype(capturePtypeO))-PTYPE_BASIC_MIN]++;
  s.effects.clearChangedEffects();
  s.effects.clearEffectedChanged();
  s.effects.setChangedPieces(s.getEffect(to));
  s.effects.template doEffect<NumBitmapEffect::Sub,true>(s,capturePtypeO,to,num1);
  s.effects.template doEffect<NumBitmapEffect::Sub,true>(s,oldPtypeO,from,num0);
  s.setBoard(from,Piece::EMPTY());
  s.effects.template doBlockAt<NumBitmapEffect::Add,true>(s,from,num0);
  s.effects.effectedNumTable[num0]=s.effects.effectedNumTable[num1];
  s.effects.effectedNumTable[num1].clear();
  s.setBoard(to,newPiece);
  s.effects.template doEffect<NumBitmapEffect::Add,true>(s,new_ptypeo,to,num0);

  if (oldPtypeO == newPtypeO(P,KING))
    s.makePinOpen(P);
  else {
    Direction lastD=UL;
    s.pin_or_open[P].reset(num0);
    s.pin_or_open[P].reset(num1); // captured is not pin
    s.recalcPinOpen(from,lastD,P);
    s.recalcPinOpen(to,lastD,P);
  }
  {
    Direction lastD=UL;
    s.pin_or_open[alt(P)].reset(num0);
    s.pin_or_open[alt(P)].reset(num1); // captured is not pin
    s.recalcPinOpen(from,lastD,alt(P));
    s.recalcPinOpen(to,lastD,alt(P));
  }
  promoted_backup = s.promoted;
  s.promoted.reset(num1);
  s.effects.effected_mask[BLACK].reset(num1);
  s.effects.effected_mask[WHITE].reset(num1);
  if (promoteMask)
    s.promoted.set(num0);
  if(s.hasEffectBy(BLACK,to))
    s.effects.effected_mask[BLACK].set(num0);
  else
    s.effects.effected_mask[BLACK].reset(num0);
  if(s.hasEffectBy(WHITE,to))
    s.effects.effected_mask[WHITE].set(num0);
  else
    s.effects.effected_mask[WHITE].reset(num0);
  s.effects.effected_changed_mask[BLACK].set(num0);
  s.effects.effected_changed_mask[WHITE].set(num0);
  {
    BoardMask changed=s.changedEffects(BLACK)|s.changedEffects(WHITE);
    changed.set(from);
    changed.set(to);
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<BLACK>()))
       || s.pin_or_open[BLACK]!=pin_or_open_backup[BLACK])
      s.makeKing8Info<BLACK>();
    if(changed.anyInRange(Board_Mask_Table3x3.mask(s.getKingPosition<WHITE>()))
       || s.pin_or_open[WHITE]!=pin_or_open_backup[WHITE])
      s.makeKing8Info<WHITE>();
  }
}

template<osl::Player P>
void osl::apply_move::ApplyDoUndoCaptureMove<P,osl::state::NumEffectState>::
epilogue(state_t& s, Position from, Position to, Piece target, 
	      Piece oldPiece, PtypeO oldPtypeO, PtypeO capturePtypeO, 
	      PtypeO newPtypeO, int num0, int num1, 
	      int num1Index, mask_t num1Mask,
	      const CArray<PieceMask,2>& pin_or_open_backup,
	 const KingMobility& king_mobility_backup,
	      const PieceMask& promoted_backup,
	 const CArray<PieceMask,2>& effected_mask_backup,
	 const CArray<PieceMask,2>& effected_changed_mask_backup,
	 const CArray<uint64_t,2>& king8infos_backup,
	 const MobilityTable &mobility_backup
)
{
  s.standMask(P).xorMask(num1Index,num1Mask);
  s.stand_count[P][unpromote(getPtype(capturePtypeO))-PTYPE_BASIC_MIN]--;
  s.mutableOnBoardMask(PlayerTraits<P>::opponent).xorMask(num1Index,num1Mask);
  s.effects.effectedNumTable[num1]=s.effects.effectedNumTable[num0];
  s.effects.effectedNumTable[num0].clear();
  s.setPieceOf(num0,oldPiece);
  s.setPieceOf(num1,target);
  s.effects.template doEffect<NumBitmapEffect::Sub,false>(s,newPtypeO,to,num0);
  s.setBoard(from,oldPiece);
  s.setBoard(to,target);
  s.effects.template doBlockAt<NumBitmapEffect::Sub,false>(s,from,num0);
  s.effects.template doEffect<NumBitmapEffect::Add,false>(s,capturePtypeO,to,num1);
  s.effects.template doEffect<NumBitmapEffect::Add,false>(s,oldPtypeO,from,num0);
  s.effects.invalidateChangedEffects();
  s.pin_or_open = pin_or_open_backup;
  s.king_mobility = king_mobility_backup;
  s.promoted = promoted_backup;
  s.effects.effected_mask = effected_mask_backup;
  s.effects.effected_changed_mask = effected_changed_mask_backup;
  s.effects.mobilityTable = mobility_backup;
  s.king8infos = king8infos_backup;
}

#endif /* OSL_NUM_EFFECT_STATE_H */
// ;;; Local Variables:
// ;;; mode:c++
// ;;; c-basic-offset:2
// ;;; End:
