/*
 * JBoss, Home of Professional Open Source
 *
 * Distributable under LGPL license.
 * See terms of license at gnu.org.
 */
package org.jboss.cache;

import org.apache.commons.logging.LogFactory;
import org.jboss.cache.lock.IdentityLock;
import org.jboss.cache.optimistic.DataVersion;
import org.jboss.cache.optimistic.DefaultDataVersion;

import java.util.Map;

/**
 * VersionedNode extends the {@link org.jboss.cache.UnversionedNode} by adding a {@link org.jboss.cache.optimistic.DataVersion} property.
 * <p/>
 * Unlike {@link org.jboss.cache.UnversionedNode}, this node supports {@link #getVersion} and {@link #setVersion(org.jboss.cache.optimistic.DataVersion)}
 * defined in {@link org.jboss.cache.NodeSPI}
 * <p/>
 * Typically used when the cache mode configured is {@link org.jboss.cache.config.Configuration.NodeLockingScheme#OPTIMISTIC}
 *
 * @author <a href="mailto:manik@jboss.org">Manik Surtani (manik@jboss.org)</a>
 * @since 2.0.0
 */
public class VersionedNode<K, V> extends UnversionedNode<K, V>
{
   private static final String DATA_VERSION_INTERNAL_KEY = "_JBOSS_INTERNAL_OPTIMISTIC_DATA_VERSION";
   private DataVersion version; // make sure this is NOT initialized to anything, even a null!  Since the UnversionedNode constructor may set this value based on a data version passed along in the data map. 

   /**
    * Although this object has a reference to the CacheImpl, the optimistic
    * node is actually disconnected from the CacheImpl itself.
    * The parent could be looked up from the TransactionWorkspace.
    */
   private NodeSPI<K, V> parent;

   protected VersionedNode(Fqn fqn, NodeSPI<K, V> parent, Map<K, V> data, CacheSPI<K, V> cache)
   {
      super(fqn.getLastElement(), fqn, data, false, cache);
      if (version == null) version = DefaultDataVersion.ZERO;
      if (parent == null && !fqn.isRoot()) throw new NullPointerException("parent");
      this.parent = parent;

      log = LogFactory.getLog(VersionedNode.class);
   }

   /**
    * Returns the version id of this node.
    *
    * @return the version
    */
   @Override
   public DataVersion getVersion()
   {
      return version;
   }

   /**
    * Returns the parent.
    */
   @Override
   public NodeSPI<K, V> getParent()
   {
      return parent;
   }

   /**
    * Sets the version id of this node.
    *
    * @param version
    */
   @Override
   public void setVersion(DataVersion version)
   {
      this.version = version;
   }

   /**
    * Optimistically locked nodes (VersionedNodes) will always use repeatable read.
    */
   @Override
   protected synchronized void initLock()
   {
      if (lock == null) lock = new IdentityLock(lockStrategyFactory, delegate);
   }

   @Override
   public Map getInternalState(boolean onlyInternalState)
   {
      Map state = super.getInternalState(onlyInternalState);
      state.put(DATA_VERSION_INTERNAL_KEY, version);
      return state;
   }

   @Override
   public void setInternalState(Map state)
   {
      if (state != null)
      {
         DataVersion dv = (DataVersion) state.remove(DATA_VERSION_INTERNAL_KEY);
         if (dv != null) version = dv;
      }
      super.setInternalState(state);
   }
}
