/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: UndoableActionCommand.java,v 1.5 2006/02/25 21:29:04 pietschy Exp $
 */
package org.pietschy.command.undo;

import org.pietschy.command.ActionCommand;
import org.pietschy.command.CommandManager;

import javax.swing.event.UndoableEditListener;
import javax.swing.undo.UndoableEdit;
import javax.swing.undo.UndoableEditSupport;


/**
 * UndoableActionCommands provide a default implementation for commands that can be undone.
 * UndoableActionCommands implement {@link UndoableEventSource} and can be monitored by an
 * {@link UndoController}
 *
 * @version $Revision: 1.5 $
 * @author andrewp
 */
public abstract class
UndoableActionCommand
extends ActionCommand
implements UndoableEventSource
{
   private UndoableEditSupport undoSupport = new UndoableEditSupport(this);

    /**
    * Creates a new anonymous UndoableActionCommand.  Anonymous commands must be fully programatically
    * generated and can only be added to groups manually by calling
    * <code>groupCommand.installFace(myAnonymousCommand)</code>.
    */
   public UndoableActionCommand()
   {
      super();
   }

   /**
    * Creates a new command with the speicifed Id that is bound to the
    * {@link CommandManager#defaultInstance()}.
    */
   public UndoableActionCommand(String commandId)
   {
      super(commandId);
   }

   /**
    * Creates a new command with the specified id that is bound to the
    * specified {@link CommandManager}.
    */
   public UndoableActionCommand(CommandManager manager, String commandId)
   {
      super(manager, commandId);
   }


   /**
    * Called by {@link #handleExecute()} to perform the undoable edit.  This method must return an undoable edit
    * that can be used by an undo controller to reverse the effects of the command.  This method may return
    * <tt>null</tt> if nothing was done.
    * @return an UndoableEdit that can be used to undo the effects of the command or <tt>null</tt> if no action
    * was taken.
    */
   protected abstract UndoableEdit
   performEdit();

   /**
    * Delegates to {@link #performEdit()} and notifies any {@link UndoableEditListener}s of any edit
    * that performEdit returns.
    */
   protected final void
   handleExecute()
   {
      UndoableEdit edit = performEdit();
      if (edit != null)
         postEdit(edit);
   }

  /**
    * Notifies all the {@link javax.swing.event.UndoableEditListener}s of the undoable event.
    * @param e the undoable event.
    */
   protected void
   postEdit(UndoableEdit e)
   {
      undoSupport.postEdit(e);
   }


   /**
    * Adds a new {@link javax.swing.event.UndoableEditListener} to this command.
    *
    * @param l the listener to installFace.
    * @see UndoController
    */
   public void
   addUndoableEditListener(UndoableEditListener l)
   {
      undoSupport.addUndoableEditListener(l);
   }

   /**
    * Removes an {@link javax.swing.event.UndoableEditListener} from this command.
    *
    * @param l the listener to remove.
    * @see UndoController
    */
   public void
   removeUndoableEditListener(UndoableEditListener l)
   {
      undoSupport.removeUndoableEditListener(l);
   }

   /**
    * Gets the {@link javax.swing.event.UndoableEditListener}s regstered with this command.
    *
    * @see UndoController
    */
   public UndoableEditListener[]
   getUndoableEditListeners()
   {
      return undoSupport.getUndoableEditListeners();
   }
}
