/**
 * GUI Commands
 * Copyright 2004 Andrew Pietsch
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 * $Id: ExtensionFileFilter.java,v 1.3 2004/12/12 05:29:32 pietschy Exp $
 */
package org.pietschy.command.file;

import javax.swing.filechooser.FileFilter;
import java.io.File;

/**
 * An implementation of {@link FileFilter} that is based on the extension of a file.
 * @version $Revision: 1.3 $
 * @author andrewp
 */
public class
ExtensionFileFilter
extends FileFilter
{
   private static final String _ID_ = "$Id: ExtensionFileFilter.java,v 1.3 2004/12/12 05:29:32 pietschy Exp $";

   private String extension;
   private String description;
   private boolean ignoreCase = true;

   /**
    * Creates a new filter for the specified file extension with that has the specified description.
    *
    * @param extension the file extension to excluding the preceeding '.', ie "java", not ".java".
    * @param description the user friendly description of the file filter.
    */
   public ExtensionFileFilter(String extension, String description)
   {
      this.extension = "." + extension;
      this.description = description;
   }

   /**
    * Accepts files that end with the specified extension.
    * @param f the file to check.
    * @return <tt>true</tt> if the file name ends with this filters extension.
    */
   public boolean
   accept(File f)
   {
      if (f.isDirectory())
         return true;

      String name = f.getName();
      if (ignoreCase)
         return name.toUpperCase().endsWith(extension.toUpperCase());
      else
         return name.endsWith(extension);
   }

   public String
   getDescription()
   {
      return description;
   }

   public boolean
   isIgnoreCase()
   {
      return ignoreCase;
   }

   public void
   setIgnoreCase(boolean ignoreCase)
   {
      this.ignoreCase = ignoreCase;
   }

   /**
    * Checks that the specified file has an extension defined.  If it does, it is
    * returned as is, even if the extension doesn't match that of the filter.  If no extension is
    * specified, a new file is returned based on the original but with the
    * extension defined by this filter.
    * @param file the file to check.
    * @return a file guarenteed to have the extension specified by this filter.
    */
   public File
   checkAndAddExtension(File file)
   {
      if (file.getName().lastIndexOf('.') < 0)
      {
         return new File(file.getAbsoluteFile() + extension);
      }

      return file;
   }
}
