package org.pietschy.command.delegate;

import java.util.*;
import java.awt.*;
import java.beans.PropertyChangeListener;
import java.beans.PropertyChangeEvent;

/**
 * This mediator derives the {@link DelegateContainer} heirarchy by traversing up
 * the component hierarchy from the currently focused component.  The final container
 * list consists of any static constainers (see {@link #setStaticContainers(DelegateContainer[])}) and
 * those compulted from the focus hierarchy.  Those found from the focus hierarchy are
 * given precedence over the static containers.
 * <p>
 * This mediator is used when you install the {@link FocusTrackingDelegateMediatorFactory} into the
 * {@link DelegateManager}.  There are also convenience methods for getting the current mediator
 * from the {@link DelegateManager} without explicitly casting.
 *
 * @see #setDelegateContainers(DelegateContainer[])
 * @see #setDelegateContainer(DelegateContainer)
 * @see #getMediatorFor(java.awt.Window)
 * @see #getMediatorFor(java.awt.Component)
 * @see DelegateManager#setDelegateMediatorFactory(DelegateMediatorFactory)
 */
public class
FocusTrackingDelegateMediator
extends DelegateMediator
{
   private Window window;

   private RelatedWindowDiscriminator relatedWindowDiscriminator;

   private DelegateContainer[] staticContainers = EMPTY_CONTAINER_ARRAY;
   private DelegateContainer[] focusedContainers = EMPTY_CONTAINER_ARRAY;

   /**
    * Conveinence method for {@link DelegateManager#getMediatorFor(java.awt.Window)} for when
    * the {@link FocusTrackingDelegateMediatorFactory} is being used.
    *
    * @param window the window of interest.
    * @return
    * @throws IllegalStateException if the mediator isn't an instance of {@link FocusTrackingDelegateMediator}.
    */
   public static FocusTrackingDelegateMediator getMediatorFor(Window window)
   {
      return verifyType(DelegateManager.getMediatorFor(window));
   }

   /**
    * Conveinence method for {@link DelegateManager#getMediatorFor(java.awt.Component)} for when
    * the {@link FocusTrackingDelegateMediatorFactory} is being used.
    *
    * @param component the component of interest.
    * @return
    * @throws IllegalStateException if the mediator isn't an instance of {@link FocusTrackingDelegateMediator}.
    */
   public static FocusTrackingDelegateMediator getMediatorFor(Component component)
   {
      return verifyType(DelegateManager.getMediatorFor(component));
   }

   private static FocusTrackingDelegateMediator verifyType(DelegateMediator mediator)
   {
      if (!(mediator instanceof FocusTrackingDelegateMediator))
      {
         throw new IllegalStateException("incorrect mediator type.  Please install FocusTrackingDelegateMediatorFactory.");
      }

      return (FocusTrackingDelegateMediator) mediator;
   }

   /**
    * Creates a new DelegateManager for the specified window.
    *
    * @param window the window of interest.
    */
   public FocusTrackingDelegateMediator(Window window, RelatedWindowDiscriminator discriminator)
   {
      if (window == null)
      {
         throw new NullPointerException("window is null");
      }

      if (discriminator == null)
      {
         throw new NullPointerException("discriminator is null");
      }

      this.window = window;
      this.relatedWindowDiscriminator = discriminator;

      getFocusManager().addPropertyChangeListener("permanentFocusOwner", new PropertyChangeListener()
      {
         public void propertyChange(PropertyChangeEvent evt)
         {
            recomputeContainersFromComponent((Component) evt.getNewValue());
         }
      });
   }

   private DelegateContainer[] combineArrays(DelegateContainer[] a, DelegateContainer[] b)
   {
      if (a == null)
      {
         throw new NullPointerException("a is null");
      }

      if (b == null)
      {
         throw new NullPointerException("b is null");
      }

      DelegateContainer[] dest = new DelegateContainer[a.length + b.length];
      System.arraycopy(a, 0, dest, 0, a.length);
      System.arraycopy(b, 0, dest, a.length, b.length);

      return dest;
   }

   /**
    * Sets the current static container list to be the specified container.  This method
    * will removed all focus specific containers until the next focus change.  If the container
    * is <code>null</code> then the static container list will be emptied, thus clearing all
    * delegates until the next focus event.
    *
    * @param staticContainer the container to use or <code>null</code> to clear the static
    * container list.
    */
   public void setStaticContainer(DelegateContainer staticContainer)
   {
      setStaticContainers(nullSafeArray(staticContainer));
   }

   /**
    * Sets the current static container list to the specified list.  This method
    * will removed all focus specific containers until the next focus change.  If containers
    * is <code>null</code> then the container list will be emptied, thus clearing all
    * delegates until the next focus event.
    *
    * @param containers the containers to use or <code>null</code> to clear the container list.
    */
   public void setStaticContainers(DelegateContainer[] containers)
   {
      focusedContainers = EMPTY_CONTAINER_ARRAY;
      staticContainers = nullSafeArray(containers);
      updateContainers();
   }

   private void updateContainers()
   {
      setDelegateContainers(combineArrays(focusedContainers, staticContainers));
   }


   /**
    * Sets the discriminator to use when tracking focus events.  The discriminator is
    * used to determine if the currently focused window should be tracked for delegates
    * pertaining to DelegatingCommands bound to the parent window.
    * <p/>
    * The default discriminator only tracks delegates in the same window as the DelegatingCommand.
    *
    * @param relatedWindowDiscriminator the discriminator to use.
    * @throws NullPointerException if the discriminator is <code>null</code>.
    */
   public void setRelatedWindowDiscriminator(RelatedWindowDiscriminator relatedWindowDiscriminator)
   {
      if (relatedWindowDiscriminator == null)
      {
         throw new NullPointerException("relatedWindowDiscriminator is null");
      }

      this.relatedWindowDiscriminator = relatedWindowDiscriminator;
   }


   private KeyboardFocusManager
   getFocusManager()
   {
      return KeyboardFocusManager.getCurrentKeyboardFocusManager();
   }

   private void extractContainers(Component component, ArrayList containers)
   {
      do
      {
         if (component instanceof DelegateContainer)
         {
            containers.add(component);
         }

         component = component.getParent();
      }
      while (component != null);
   }

   private void recomputeContainersFromComponent(Component component)
   {
      ArrayList containers = new ArrayList();
      if (component != null)
      {
         Window focusedWindow = getFocusManager().getFocusedWindow();
         if (focusedWindow != null && relatedWindowDiscriminator.isRelated(window, focusedWindow))
         {
            extractContainers(component, containers);

            // Include the window in the heirarchy.
            if (focusedWindow instanceof DelegateContainer)
            {
               containers.add(focusedWindow);
            }
         }
      }

      // now update all our listeners.. we only update if we've found some containers.  This means
      // the user can click on a toolbar without de-registering all the current delegates.  If this
      // isn't done, clicking a DelegatingCommand on the toolbar would always fail (as the delegate
      // would be uninstalled as part of the focus change, thus disableing the button.. doh).
      if (containers.size() > 0)
      {
         // only fire the event if the heirarchy is different
         focusedContainers = (DelegateContainer[]) containers.toArray(new DelegateContainer[containers.size()]);
      }

      updateContainers();
   }
}
