/*
 *  @(#)ErrorsImpl.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.mbtf.v1.engine;


import net.sourceforge.groboutils.mbtf.v1.IError;
import net.sourceforge.groboutils.mbtf.v1.IErrors;
import net.sourceforge.groboutils.mbtf.v1.IPathHistory;
import net.sourceforge.groboutils.mbtf.v1.TestHaltRuntimeException;
import net.sourceforge.groboutils.mbtf.v1.TestFailRuntimeException;

import java.util.Vector;


/**
 * Default implementation of IErrors.  Designed so that the same object can
 * be reused.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:52:26 $
 * @since      June 12, 2002
 */
public class ErrorsImpl implements IErrors
{
    private Vector errors = new Vector();
    private Vector warnings = new Vector();
    private boolean haltTests = false;
    private boolean haltPath = false;
    private IPathHistory ph;
    
    
    /*
     * Reset the implementation for reuse.
    public void reset()
    {
        this.errors.removeAllElements();
        this.warnings.removeAllElements();
        this.haltTests = false;
        this.haltPath = false;
        this.ph = null;
    }
     */
    
    
    public void setCurrentPathHistory( IPathHistory ph )
    {
        this.ph = ph.copy();
    }
    
    
    /**
     * Immediately stop the path processing, and do not continue other paths
     * for processing.  This will throw a <tt>RuntimeException</tt>.
     * <P>
     * Halts should be a last-recourse to indicate that the system cannot be
     * used for further testing.
     *
     * @param msg a human-readable error message.
     * @exception TestHaltRuntimeException will always be generated.
     */
    public void halt( String msg )
    {
        addErrorType( msg, null );
        this.haltPath = true;
        this.haltTests = true;
        
        throw new TestHaltRuntimeException( this, msg );
    }
    
    
    /**
     * Add a failure to the list of current errors.  Validation methods that
     * register failures will halt the current path's testing.  This method
     * will not throw an exception, so validation processing must leave the
     * method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     */
    public void addFailure( String msg )
    {
        addFailure( msg, null );
    }
    
    
    /**
     * Add a failure associated with a Throwable to the list of current errors.
     * Validation methods that register failures will halt the current path's
     * testing.  This method will not throw an exception, so validation
     * processing must leave the method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     */
    public void addFailure( String msg, Throwable t )
    {
        addErrorType( msg, t );
        this.haltPath = true;
    }
    
    
    /**
     * Add a failure to the list of current errors.  Validation methods that
     * register failures will halt the current path's testing.  This method
     * will not throw a <tt>TestFailRuntimeException</tt>, so validation
     * processing must leave the method on its own.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @exception TestFailRuntimeException allows for easy exiting of a
     *      burried method call stack.
     */
    public void fail( String msg )
            throws TestFailRuntimeException
    {
        addFailure( msg );
        throw new TestFailRuntimeException( this, msg );
    }
    
    
    /**
     * Add a failure associated with a Throwable to the list of current errors.
     * Validation methods that register failures will halt the current path's
     * testing.  This method will throw a <tt>TestFailRuntimeException</tt> to
     * allow for an easy exit from a burried method call stack.
     * <P>
     * Failures should be registered when a non-recoverable error occurs in the
     * system.  The framework may still process other paths, though.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     * @exception TestFailRuntimeException allows for easy exiting of a
     *      burried method call stack.
     */
    public void fail( String msg, Throwable t )
            throws TestFailRuntimeException
    {
        addFailure( msg, t );
        throw new TestFailRuntimeException( this, msg );
    }
    
    
    /**
     * Add an error to the list of current errors.  Validation methods that
     * register errors will not halt the current path's testing, but the error
     * will be listed in the report with the associated path where the error
     * condition occured.
     * <P>
     * Errors should be registered when an error occurs in the system, but
     * the system can continue processing the path.
     *
     * @param msg a human-readable error message.
     */
    public void addError( String msg )
    {
        addError( msg, null );
    }
    
    
    /**
     * Add an error associated with a Throwable to the list of current errors.
     * Validation methods that register errors will halt the current path's
     * testing.
     * <P>
     * Errors should be registered when an error occurs in the system, but
     * the system can continue processing the path.
     *
     * @param msg a human-readable error message.
     * @param t the exception associated with the error.
     */
    public void addError( String msg, Throwable t )
    {
        addErrorType( msg, t );
    }
    
    
    /**
     * Add a warning to the list of current warnings.  Warnings will not
     * halt the testing process, and will not register an error.
     * <P>
     * Warnings should be used when a questionable system state occurs, or if
     * the tester wants to perform debugging.
     *
     * @param msg a human-readable message.
     */
    public void addWarning( String msg )
    {
        addMessage( this.warnings, msg, null );
    }
    
    
    /**
     * Retrieve all registered errors.
     */
    public IError[] getErrors()
    {
        IError e[] = new IError[ this.errors.size() ];
        this.errors.copyInto( e );
        return e;
    }
    
    
    
    
    
    //-------------------------------------------------------------------------

    
    public IError[] getWarnings()
    {
        IError e[] = new IError[ this.warnings.size() ];
        this.warnings.copyInto( e );
        return e;
    }
    
    
    public boolean isHaltPath()
    {
        return this.haltPath;
    }
    
    
    public boolean isHaltTests()
    {
        return this.haltTests;
    }
    
    
    public String toString()
    {
        IError[] e = getErrors();
        StringBuffer sb = new StringBuffer("[ Registered Errors:\n");
        for (int i = 0; i < e.length; ++i)
        {
            sb.append( e[i] );
        }
        
        sb.append("Registered Warnings:\n");
        e = getWarnings();
        for (int i = 0; i < e.length; ++i)
        {
            sb.append( e[i] );
        }
        
        sb.append( "\n]" );
        return sb.toString();
    }
    
    
    //-------------------------------------------------------------------------
    
    
    public void addErrors( IErrors e )
    {
        addErrors( e.getErrors() );
        if (e instanceof ErrorsImpl)
        {
            ErrorsImpl ei = (ErrorsImpl)e;
            addWarnings( ei.getWarnings() );
            if (ei.isHaltPath())
            {
                this.haltPath = true;
            }
            if (ei.isHaltTests())
            {
                this.haltTests = true;
            }
        }
    }
    
    
    public void addErrors( IError[] r )
    {
        if (r != null)
        {
            for (int i = 0; i < r.length; ++i)
            {
                this.errors.addElement( r[i] );
            }
        }
    }
    
    
    public void addWarnings( IError[] r )
    {
        if (r != null)
        {
            for (int i = 0; i < r.length; ++i)
            {
                this.warnings.addElement( r[i] );
            }
        }
    }
    
    
    
    //-------------------------------------------------------------------------
    
    
    protected void addErrorType( String msg, Throwable t )
    {
        addMessage( this.errors, msg, t );
    }
    
    
    protected void addMessage( Vector v, String msg, Throwable t )
    {
        IError err = new ErrorImpl( msg, t, this.ph );
        v.addElement( err );
    }
}

