/*
 *  @(#)DefaultListAttribute.java
 *
 * Copyright (C) 2002-2003 Matt Albrecht
 * groboclown@users.sourceforge.net
 * http://groboutils.sourceforge.net
 *
 *  Part of the GroboUtils package at:
 *  http://groboutils.sourceforge.net
 *
 *  Permission is hereby granted, free of charge, to any person obtaining a
 *  copy of this software and associated documentation files (the "Software"),
 *  to deal in the Software without restriction, including without limitation
 *  the rights to use, copy, modify, merge, publish, distribute, sublicense,
 *  and/or sell copies of the Software, and to permit persons to whom the 
 *  Software is furnished to do so, subject to the following conditions:
 *
 *  The above copyright notice and this permission notice shall be included in 
 *  all copies or substantial portions of the Software. 
 *
 *  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR 
 *  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, 
 *  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.  IN NO EVENT SHALL 
 *  THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER 
 *  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING 
 *  FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER 
 *  DEALINGS IN THE SOFTWARE.
 */
package net.sourceforge.groboutils.pmti.v1.defimpl;

import net.sourceforge.groboutils.pmti.v1.IListAttribute;
import net.sourceforge.groboutils.pmti.v1.IAttributeInfo;

import java.util.Enumeration;
import java.util.NoSuchElementException;


/**
 * A specialization of an attribute which contains a list of values.
 *
 * @author     Matt Albrecht <a href="mailto:groboclown@users.sourceforge.net">groboclown@users.sourceforge.net</a>
 * @version    $Date: 2003/02/10 22:51:59 $
 * @since      July 12, 2002
 */
public class DefaultListAttribute implements IListAttribute
{
    // don't use a vector for memory and synchronization efficiency
    private Object[] obj;
    private IAttributeInfo info;
    
    
    public DefaultListAttribute( Object[] o, IAttributeInfo i )
    {
        if (i == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        this.obj = copyObj( o );
        this.info = i;
    }
    
    
    
    /**
     * Returns the enumeration for all values in this attribute.
     */
    public Object getValue()
    {
        return getValues();
    }
    
    
    /**
     * Returns the meta-information for this attribute.
     */
    public IAttributeInfo getInfo()
    {
        return this.info;
    }
    
    
    /**
     * A synonym for <tt>getValue()</tt>, but redefined here to explicitly
     * declare the returned type.
     */
    public Enumeration getValues()
    {
        // don't need to copy the object array - it's immutable!
        return new ArrayEnum( this.obj );
    }
    
    
    /**
     * 
     */
    public int getValueCount()
    {
        return this.obj.length;
    }
    
    
    /**
     * 
     */
    public boolean containsValue( Object value )
    {
        for (int i = 0; i < this.obj.length; ++i)
        {
            Object o = this.obj[i];
            if (value == null)
            {
                if (o == null)
                {
                    return true;
                }
            }
            else
            if (value.equals( o ))
            {
                return true;
            }
        }
        return false;
    }
    
    
    
    private Object[] copyObj( Object[] o )
    {
        if (o == null)
        {
            throw new IllegalArgumentException("no null arguments");
        }
        Object ret[] = new Object[ o.length ];
        System.arraycopy( o, 0, ret, 0, o.length );
        return ret;
    }
    
    
    private static class ArrayEnum implements Enumeration
    {
        private Object vals[];
        private int index;
        public ArrayEnum( Object o[] )
        {
            this.vals = o;
            this.index = 0;
        }
        
        
        public boolean hasMoreElements()
        {
            return (this.index < this.vals.length);
        }
        
        
        public Object nextElement()
        {
            if (this.index >= this.vals.length)
            {
                throw new NoSuchElementException();
            }
            Object ret = this.vals[ this.index ];
            ++this.index;
            return ret;
        }
    }
}

