/**
 * @file libgalago-gtk/galago-gtk-contact-path-bar.c
 *       Contact chooser path bar widget
 *
 * @Copyright (C) 2005-2006 Christian Hammond.
 *
 * This library is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 *
 * This library is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 *
 * You should have received a copy of the GNU Library General Public
 * License along with this library; if not, write to the
 * Free Software Foundation, Inc., 59 Temple Place - Suite 330,
 * Boston, MA  02111-1307, USA.
 */
#include <libgalago-gtk/galago-gtk-contact-path-bar.h>
#include <libgalago-gtk/galago-gdk-pixbuf.h>
#include <libgalago-gtk/galago-gtk-private.h>
#include <libgalago-gtk/galago-gtk-stock.h>
#include <gtk/gtk.h>

enum
{
	ROOT,
	PERSON
};

enum
{
	ROOT_ACTIVATED,
	LAST_SIGNAL
};

typedef struct
{
	GtkWidget *image;
	GtkWidget *label;
	GtkWidget *button;

	gboolean ignore_changes;

} ButtonData;

struct _GalagoGtkContactPathBarPriv
{
	union
	{
		GalagoPerson *person;
		GalagoService *service;
	} u;

	GalagoGtkContactPathBarRootType root_type;
	GalagoAccount *contact;

	GtkWidget *left_button;
	GtkWidget *right_button;

	ButtonData buttons[2];
};

static void galago_gtk_contact_path_bar_class_init(GalagoGtkContactPathBarClass *klass);
static void galago_gtk_contact_path_bar_init(GalagoGtkContactPathBar *path_bar);
static void galago_gtk_contact_path_bar_finalize(GObject *obj);
static void galago_gtk_contact_path_bar_destroy(GtkObject *obj);

static GtkHBoxClass *parent_class = NULL;
static guint signals[LAST_SIGNAL] = {0};

GType
galago_gtk_contact_path_bar_get_type(void)
{
	static GType type = 0;

	if (!type)
	{
		static const GTypeInfo info =
		{
			sizeof(GalagoGtkContactPathBarClass),
			NULL,
			NULL,
			(GClassInitFunc)galago_gtk_contact_path_bar_class_init,
			NULL,
			NULL,
			sizeof(GalagoGtkContactPathBar),
			0,
			(GInstanceInitFunc)galago_gtk_contact_path_bar_init
		};

		type = g_type_register_static(GTK_TYPE_HBOX,
									  "GalagoGtkContactPathBar",
									  &info, 0);
	}

	return type;
}

static void
galago_gtk_contact_path_bar_class_init(GalagoGtkContactPathBarClass *klass)
{
	GObjectClass *gobject_class;
	GtkObjectClass *object_class;

	galago_gtk_stock_init();

	parent_class = g_type_class_peek_parent(klass);

	gobject_class = G_OBJECT_CLASS(klass);
	object_class  = GTK_OBJECT_CLASS(klass);

	gobject_class->finalize = galago_gtk_contact_path_bar_finalize;

	object_class->destroy = galago_gtk_contact_path_bar_destroy;

	signals[ROOT_ACTIVATED] =
		g_signal_new("root-activated",
					 G_TYPE_FROM_CLASS(gobject_class),
					 G_SIGNAL_RUN_LAST | G_SIGNAL_ACTION,
					 G_STRUCT_OFFSET(GalagoGtkContactPathBarClass,
									 root_activated),
					 NULL, NULL,
					 g_cclosure_marshal_VOID__VOID,
					 G_TYPE_NONE, 0);
}

static void
root_button_clicked_cb(GtkWidget *button, GalagoGtkContactPathBar *path_bar)
{
	if (path_bar->priv->buttons[ROOT].ignore_changes)
		return;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);

	if (path_bar->priv->root_type ==
		GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE)
	{
		galago_gtk_contact_path_bar_set_person(path_bar, NULL);

		g_signal_emit(path_bar, signals[ROOT_ACTIVATED], 0);
	}
}

static void
person_button_clicked_cb(GtkWidget *button, GalagoGtkContactPathBar *path_bar)
{
	if (path_bar->priv->buttons[PERSON].ignore_changes)
		return;

	gtk_toggle_button_set_active(GTK_TOGGLE_BUTTON(button), TRUE);
}

static GtkWidget *
construct_button(GalagoGtkContactPathBar *path_bar, ButtonData *button_data)
{
	GtkWidget *hbox;

	button_data->button = gtk_toggle_button_new();

	hbox = gtk_hbox_new(FALSE, 6);
	gtk_widget_show(hbox);
	gtk_container_add(GTK_CONTAINER(button_data->button), hbox);

	button_data->image = gtk_image_new();
	gtk_widget_show(button_data->image);
	gtk_box_pack_start(GTK_BOX(hbox), button_data->image, FALSE, FALSE, 0);

	button_data->label = gtk_label_new(NULL);
	gtk_widget_show(button_data->label);
	gtk_box_pack_start(GTK_BOX(hbox), button_data->label, TRUE, TRUE, 0);

	return button_data->button;
}

static void
galago_gtk_contact_path_bar_init(GalagoGtkContactPathBar *path_bar)
{
	GtkWidget *left_arrow;
	GtkWidget *right_arrow;
	GtkWidget *button;

	path_bar->priv = g_new0(GalagoGtkContactPathBarPriv, 1);
	gtk_box_set_spacing(GTK_BOX(path_bar), 3);

	/* Left arrow button */
	path_bar->priv->left_button = gtk_button_new();
	gtk_box_pack_start(GTK_BOX(path_bar), path_bar->priv->left_button,
					   FALSE, FALSE, 0);

	left_arrow = gtk_arrow_new(GTK_ARROW_LEFT, GTK_SHADOW_NONE);
	gtk_widget_show(left_arrow);
	gtk_container_add(GTK_CONTAINER(path_bar->priv->left_button), left_arrow);


	/* Right arrow button */
	path_bar->priv->right_button = gtk_button_new();
	gtk_box_pack_end(GTK_BOX(path_bar), path_bar->priv->right_button,
					   FALSE, FALSE, 0);

	right_arrow = gtk_arrow_new(GTK_ARROW_RIGHT, GTK_SHADOW_NONE);
	gtk_widget_show(right_arrow);
	gtk_container_add(GTK_CONTAINER(path_bar->priv->right_button), right_arrow);


	/* Root button */
	button = construct_button(path_bar, &path_bar->priv->buttons[ROOT]);
	gtk_box_pack_start(GTK_BOX(path_bar), button, FALSE, FALSE, 0);
	gtk_widget_show(path_bar->priv->buttons[ROOT].button);

	g_signal_connect(G_OBJECT(button), "clicked",
					 G_CALLBACK(root_button_clicked_cb), path_bar);

	/* Contact button */
	button = construct_button(path_bar, &path_bar->priv->buttons[PERSON]);
	gtk_box_pack_start(GTK_BOX(path_bar), button, FALSE, FALSE, 0);

	g_signal_connect(G_OBJECT(button), "clicked",
					 G_CALLBACK(person_button_clicked_cb), path_bar);

	galago_gtk_contact_path_bar_set_root_type(path_bar,
		GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE);
}

static void
galago_gtk_contact_path_bar_finalize(GObject *obj)
{
	GalagoGtkContactPathBar *path_bar;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(obj));

	path_bar = GALAGO_GTK_CONTACT_PATH_BAR(obj);

	g_free(path_bar->priv);

	if (G_OBJECT_CLASS(parent_class)->finalize)
		G_OBJECT_CLASS(parent_class)->finalize(obj);
}

static void
galago_gtk_contact_path_bar_destroy(GtkObject *obj)
{
	GalagoGtkContactPathBar *path_bar;

	g_return_if_fail(obj != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(obj));

	path_bar = GALAGO_GTK_CONTACT_PATH_BAR(obj);

	if (GTK_OBJECT_CLASS(parent_class)->destroy)
		GTK_OBJECT_CLASS(parent_class)->destroy(obj);
}

GtkWidget *
galago_gtk_contact_path_bar_new(void)
{
	return GTK_WIDGET(g_object_new(GALAGO_GTK_TYPE_CONTACT_PATH_BAR, NULL));
}

void
_galago_gtk_contact_path_bar_set_root_type(
	GalagoGtkContactPathBar *path_bar,
	GalagoGtkContactPathBarRootType root_type)
{
	GdkPixbuf *pixbuf;
	const char *text;

	g_return_if_fail(path_bar != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar));

	if (path_bar->priv->root_type == root_type)
		return;

	path_bar->priv->root_type = root_type;

	switch (root_type)
	{
		case GALAGO_GTK_CONTACT_PATH_BAR_ROOT_SERVICE:
			text = galago_service_get_name(path_bar->priv->u.service);
			pixbuf = galago_gdk_pixbuf_new_from_service_with_size(
				path_bar->priv->u.service, GTK_ICON_SIZE_MENU);
			break;

		case GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE:
			text = _("People");
			pixbuf = gtk_widget_render_icon(GTK_WIDGET(path_bar),
											GALAGO_GTK_STOCK_PERSON,
											GTK_ICON_SIZE_MENU, NULL);
			break;

		case GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS:
			text = _("Contacts");
			pixbuf = galago_gdk_pixbuf_new_generic(16, 16);
			break;

		default:
			return;
	}

	gtk_label_set_text(GTK_LABEL(path_bar->priv->buttons[ROOT].label), text);
	gtk_image_set_from_pixbuf(GTK_IMAGE(path_bar->priv->buttons[ROOT].image),
							  pixbuf);
	g_object_unref(G_OBJECT(pixbuf));
}

void
galago_gtk_contact_path_bar_set_root_type(
	GalagoGtkContactPathBar *path_bar,
	GalagoGtkContactPathBarRootType root_type)
{
	g_return_if_fail(path_bar != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar));
	g_return_if_fail(
		root_type == GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS ||
		root_type == GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE);

	_galago_gtk_contact_path_bar_set_root_type(path_bar, root_type);

	gtk_widget_hide(path_bar->priv->buttons[PERSON].button);

	path_bar->priv->buttons[ROOT].ignore_changes = TRUE;
	gtk_toggle_button_set_active(
		GTK_TOGGLE_BUTTON(path_bar->priv->buttons[ROOT].button), TRUE);
	path_bar->priv->buttons[ROOT].ignore_changes = FALSE;
}

void
galago_gtk_contact_path_bar_set_service(GalagoGtkContactPathBar *path_bar,
										GalagoService *service)
{
	g_return_if_fail(path_bar != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar));

	if (path_bar->priv->u.service == service)
		return;

	path_bar->priv->u.service = service;

	_galago_gtk_contact_path_bar_set_root_type(path_bar,
		(service != NULL
		 ? GALAGO_GTK_CONTACT_PATH_BAR_ROOT_SERVICE
		 : GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS));
}

void
galago_gtk_contact_path_bar_set_person(GalagoGtkContactPathBar *path_bar,
									   GalagoPerson *person)
{
	g_return_if_fail(path_bar != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar));
	g_return_if_fail(person == NULL || GALAGO_IS_PERSON(person));

	if (path_bar->priv->u.person == person)
		return;

	path_bar->priv->u.person = person;

	_galago_gtk_contact_path_bar_set_root_type(path_bar,
		GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE);

	if (person == NULL)
	{
		gtk_widget_hide(path_bar->priv->buttons[PERSON].button);
		path_bar->priv->buttons[ROOT].ignore_changes = TRUE;
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(path_bar->priv->buttons[ROOT].button), TRUE);
		path_bar->priv->buttons[ROOT].ignore_changes = FALSE;
	}
	else
	{
		GdkPixbuf *pixbuf;

		gtk_widget_show(path_bar->priv->buttons[PERSON].button);
		gtk_label_set_text(GTK_LABEL(path_bar->priv->buttons[PERSON].label),
						   galago_person_get_display_name(person));

		pixbuf = gtk_widget_render_icon(GTK_WIDGET(path_bar),
										GALAGO_GTK_STOCK_PERSON,
										GTK_ICON_SIZE_MENU, NULL);
		gtk_image_set_from_pixbuf(
			GTK_IMAGE(path_bar->priv->buttons[PERSON].image), pixbuf);
		g_object_unref(G_OBJECT(pixbuf));

		path_bar->priv->buttons[ROOT].ignore_changes = TRUE;
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(path_bar->priv->buttons[ROOT].button), FALSE);
		path_bar->priv->buttons[ROOT].ignore_changes = FALSE;

		path_bar->priv->buttons[PERSON].ignore_changes = TRUE;
		gtk_toggle_button_set_active(
			GTK_TOGGLE_BUTTON(path_bar->priv->buttons[PERSON].button), TRUE);
		path_bar->priv->buttons[PERSON].ignore_changes = FALSE;

	}
}

void
galago_gtk_contact_path_bar_set_contact(GalagoGtkContactPathBar *path_bar,
										GalagoAccount *contact)
{
	g_return_if_fail(path_bar != NULL);
	g_return_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar));
	g_return_if_fail(contact != NULL);
	g_return_if_fail(GALAGO_IS_ACCOUNT(contact));

	if (path_bar->priv->contact == contact)
		return;

	path_bar->priv->contact = contact;
}

GalagoGtkContactPathBarRootType
galago_gtk_contact_path_bar_get_root_type(
	const GalagoGtkContactPathBar *path_bar)
{
	g_return_val_if_fail(path_bar != NULL,
						 GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS);
	g_return_val_if_fail(GALAGO_GTK_IS_CONTACT_PATH_BAR(path_bar),
						 GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_CONTACTS);

	return path_bar->priv->root_type;
}

GalagoService *
galago_gtk_contact_path_bar_get_service(const GalagoGtkContactPathBar *path_bar)
{
	g_return_val_if_fail(path_bar != NULL,                      NULL);
	g_return_val_if_fail(GALAGO_GTK_CONTACT_PATH_BAR(path_bar), NULL);

	if (path_bar->priv->root_type != GALAGO_GTK_CONTACT_PATH_BAR_ROOT_SERVICE)
		return NULL;

	return path_bar->priv->u.service;
}

GalagoPerson *
galago_gtk_contact_path_bar_get_person(const GalagoGtkContactPathBar *path_bar)
{
	g_return_val_if_fail(path_bar != NULL,                      NULL);
	g_return_val_if_fail(GALAGO_GTK_CONTACT_PATH_BAR(path_bar), NULL);

	if (path_bar->priv->root_type !=
		GALAGO_GTK_CONTACT_PATH_BAR_ROOT_ALL_PEOPLE)
	{
		return NULL;
	}

	return path_bar->priv->u.person;
}

GalagoAccount *
galago_gtk_contact_path_bar_get_contact(const GalagoGtkContactPathBar *path_bar)
{
	g_return_val_if_fail(path_bar != NULL,                      NULL);
	g_return_val_if_fail(GALAGO_GTK_CONTACT_PATH_BAR(path_bar), NULL);

	return path_bar->priv->contact;
}
